/* eri_enb_log.c
 *
 * Ericsson eNode-B raw log file format decoder for the Wiretap library.
 *
 * Wiretap Library
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 */

#include "eri_enb_log.h"

#include <string.h>

#include "file_wrappers.h"
#include "wtap-int.h"

static const char eri_enb_log_magic[] = "com_ericsson";

static int eri_enb_log_file_type_subtype = -1;

void register_eri_enb_log(void);

#define MAX_LINE_LENGTH            131072

typedef struct {
    char line[MAX_LINE_LENGTH];
} eri_enb_log_t;

static bool eri_enb_log_get_packet(wtap *wth, FILE_T fh, wtap_rec* rec,
	int* err _U_, char** err_info _U_)
{
	eri_enb_log_t *eri_enb = (eri_enb_log_t *)wth->priv;
	/* Read in a line */
	int64_t pos_before = file_tell(fh);

	while (file_gets(eri_enb->line, sizeof(eri_enb->line), fh) != NULL)
	{
		nstime_t packet_time;
		int length;
		/* Set length (avoiding strlen()) and offset.. */
		length = (int)(file_tell(fh) - pos_before);

		/* ...but don't want to include newline in line length */
		if (length > 0 && eri_enb->line[length - 1] == '\n') {
			eri_enb->line[length - 1] = '\0';
			length = length - 1;
		}
		/* Nor do we want '\r' (as will be written when log is created on windows) */
		if (length > 0 && eri_enb->line[length - 1] == '\r') {
			eri_enb->line[length - 1] = '\0';
			length = length - 1;
		}

		if (NULL != iso8601_to_nstime(&packet_time, eri_enb->line+1, ISO8601_DATETIME)) {
			rec->ts.secs = packet_time.secs;
			rec->ts.nsecs = packet_time.nsecs;
			rec->presence_flags |= WTAP_HAS_TS;
		} else {
			rec->ts.secs = 0;
			rec->ts.nsecs = 0;
			rec->presence_flags = 0; /* no time stamp, no separate "on the wire" length */
		}
		/* We've got a full packet! */
		wtap_setup_packet_rec(rec, wth->file_encap);
		rec->block = wtap_block_create(WTAP_BLOCK_PACKET);
		rec->rec_header.packet_header.caplen = length;
		rec->rec_header.packet_header.len = length;

		*err = 0;

		/* Append data to the packet buffer */
		ws_buffer_append(&rec->data, eri_enb->line, rec->rec_header.packet_header.caplen);

		return true;

	}
	return false;
}

/* Find the next packet and parse it; called from wtap_read(). */
static bool eri_enb_log_read(wtap* wth, wtap_rec* rec,
	int* err, char** err_info, int64_t* data_offset)
{
	*data_offset = file_tell(wth->fh);

	return eri_enb_log_get_packet(wth, wth->fh, rec, err, err_info);
}

/* Used to read packets in random-access fashion */
static bool eri_enb_log_seek_read(wtap* wth, int64_t seek_off,
	wtap_rec* rec, int* err, char** err_info)
{
	if (file_seek(wth->random_fh, seek_off, SEEK_SET, err) == -1)
	{
		return false;
	}

	return eri_enb_log_get_packet(wth, wth->random_fh, rec, err, err_info);
}

static void eri_enb_log_close(wtap *wth)
{
	eri_enb_log_t *eri_enb = (eri_enb_log_t *)wth->priv;

	g_free(eri_enb);
	/* XXX: Prevent double free by wtap_close() */
	wth->priv = NULL;
}

wtap_open_return_val
eri_enb_log_open(wtap *wth, int *err, char **err_info)
{
	char line1[64];

	/* Look for Ericsson eNode-B log header */
	if (file_gets(line1, sizeof(line1), wth->fh) == NULL)
	{
		*err = file_error(wth->fh, err_info);
		if (*err != 0 && *err != WTAP_ERR_SHORT_READ)
			return WTAP_OPEN_ERROR;
		return WTAP_OPEN_NOT_MINE;
	}

	if (g_strstr_len(line1, sizeof(line1), eri_enb_log_magic) == NULL)
	{
		return WTAP_OPEN_NOT_MINE;
	}

	if (file_seek(wth->fh, 0, SEEK_SET, err) == -1)
		return WTAP_OPEN_ERROR;

	wth->file_type_subtype = eri_enb_log_file_type_subtype;
	wth->file_encap = WTAP_ENCAP_ERI_ENB_LOG;
	wth->file_tsprec = WTAP_TSPREC_NSEC;
	wth->subtype_read = eri_enb_log_read;
	wth->subtype_seek_read = eri_enb_log_seek_read;
	wth->subtype_close = eri_enb_log_close;
	wth->snapshot_length = 0;
	wth->priv = g_new(eri_enb_log_t, 1);

	return WTAP_OPEN_MINE;
}

static const struct supported_block_type eri_enb_log_blocks_supported[] = {
	/*
	 * This is a file format that we dissect, so we provide
	 * only one "packet" with the file's contents, and don't
	 * support any options.
	 */
	{ WTAP_BLOCK_PACKET, MULTIPLE_BLOCKS_SUPPORTED, NO_OPTIONS_SUPPORTED }
};

static const struct file_type_subtype_info eri_enb_log_info = {
	"Ericsson eNode-B raw log", "eri_enb_log", "eri_enb_log", NULL,
	false, BLOCKS_SUPPORTED(eri_enb_log_blocks_supported),
	NULL, NULL, NULL
};

void register_eri_enb_log(void)
{
	eri_enb_log_file_type_subtype = wtap_register_file_type_subtype(&eri_enb_log_info);

	/*
	 * Register name for backwards compatibility with the
	 * wtap_filetypes table in Lua.
	 */
	//wtap_register_backwards_compatibility_lua_name("MP4",
	//    eri_enb_log_file_type_subtype);
}

/*
 * Editor modelines  -  https://www.wireshark.org/tools/modelines.html
 *
 * Local variables:
 * c-basic-offset: 8
 * tab-width: 8
 * indent-tabs-mode: t
 * End:
 *
 * vi: set shiftwidth=8 tabstop=8 noexpandtab:
 * :indentSize=8:tabSize=8:noTabs=false:
 */
