/* packet-netlink-nl80211.c
 * Dissector for nl80211 (over Netlink).
 *
 * Copyright (c) 2017, Peter Wu <peter@lekensteyn.nl>
 * Copyright (c) 2018, Mikael Kanstrup <mikael.kanstrup@sony.com>
 *
 * Wireshark - Network traffic analyzer
 * By Gerald Combs <gerald@wireshark.org>
 * Copyright 1998 Gerald Combs
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 */

#include "config.h"

#include <epan/packet.h>
#include "packet-ieee80211.h"
#include "packet-netlink.h"

void proto_register_netlink_nl80211(void);
void proto_reg_handoff_netlink_nl80211(void);

typedef struct  {
    packet_info *pinfo;
} netlink_nl80211_info_t;

static dissector_handle_t ieee80211_handle;
static dissector_table_t ieee80211_tag_dissector_table;

/* Extracted using tools/generate-nl80211-fields.py */
/* Definitions from linux/nl80211.h {{{ */
enum ws_nl80211_commands {
    WS_NL80211_CMD_UNSPEC,
    WS_NL80211_CMD_GET_WIPHY,
    WS_NL80211_CMD_SET_WIPHY,
    WS_NL80211_CMD_NEW_WIPHY,
    WS_NL80211_CMD_DEL_WIPHY,
    WS_NL80211_CMD_GET_INTERFACE,
    WS_NL80211_CMD_SET_INTERFACE,
    WS_NL80211_CMD_NEW_INTERFACE,
    WS_NL80211_CMD_DEL_INTERFACE,
    WS_NL80211_CMD_GET_KEY,
    WS_NL80211_CMD_SET_KEY,
    WS_NL80211_CMD_NEW_KEY,
    WS_NL80211_CMD_DEL_KEY,
    WS_NL80211_CMD_GET_BEACON,
    WS_NL80211_CMD_SET_BEACON,
    WS_NL80211_CMD_START_AP,
    WS_NL80211_CMD_STOP_AP,
    WS_NL80211_CMD_GET_STATION,
    WS_NL80211_CMD_SET_STATION,
    WS_NL80211_CMD_NEW_STATION,
    WS_NL80211_CMD_DEL_STATION,
    WS_NL80211_CMD_GET_MPATH,
    WS_NL80211_CMD_SET_MPATH,
    WS_NL80211_CMD_NEW_MPATH,
    WS_NL80211_CMD_DEL_MPATH,
    WS_NL80211_CMD_SET_BSS,
    WS_NL80211_CMD_SET_REG,
    WS_NL80211_CMD_REQ_SET_REG,
    WS_NL80211_CMD_GET_MESH_CONFIG,
    WS_NL80211_CMD_SET_MESH_CONFIG,
    WS_NL80211_CMD_SET_MGMT_EXTRA_IE,
    WS_NL80211_CMD_GET_REG,
    WS_NL80211_CMD_GET_SCAN,
    WS_NL80211_CMD_TRIGGER_SCAN,
    WS_NL80211_CMD_NEW_SCAN_RESULTS,
    WS_NL80211_CMD_SCAN_ABORTED,
    WS_NL80211_CMD_REG_CHANGE,
    WS_NL80211_CMD_AUTHENTICATE,
    WS_NL80211_CMD_ASSOCIATE,
    WS_NL80211_CMD_DEAUTHENTICATE,
    WS_NL80211_CMD_DISASSOCIATE,
    WS_NL80211_CMD_MICHAEL_MIC_FAILURE,
    WS_NL80211_CMD_REG_BEACON_HINT,
    WS_NL80211_CMD_JOIN_IBSS,
    WS_NL80211_CMD_LEAVE_IBSS,
    WS_NL80211_CMD_TESTMODE,
    WS_NL80211_CMD_CONNECT,
    WS_NL80211_CMD_ROAM,
    WS_NL80211_CMD_DISCONNECT,
    WS_NL80211_CMD_SET_WIPHY_NETNS,
    WS_NL80211_CMD_GET_SURVEY,
    WS_NL80211_CMD_NEW_SURVEY_RESULTS,
    WS_NL80211_CMD_SET_PMKSA,
    WS_NL80211_CMD_DEL_PMKSA,
    WS_NL80211_CMD_FLUSH_PMKSA,
    WS_NL80211_CMD_REMAIN_ON_CHANNEL,
    WS_NL80211_CMD_CANCEL_REMAIN_ON_CHANNEL,
    WS_NL80211_CMD_SET_TX_BITRATE_MASK,
    WS_NL80211_CMD_REGISTER_FRAME,
    WS_NL80211_CMD_FRAME,
    WS_NL80211_CMD_FRAME_TX_STATUS,
    WS_NL80211_CMD_SET_POWER_SAVE,
    WS_NL80211_CMD_GET_POWER_SAVE,
    WS_NL80211_CMD_SET_CQM,
    WS_NL80211_CMD_NOTIFY_CQM,
    WS_NL80211_CMD_SET_CHANNEL,
    WS_NL80211_CMD_SET_WDS_PEER,
    WS_NL80211_CMD_FRAME_WAIT_CANCEL,
    WS_NL80211_CMD_JOIN_MESH,
    WS_NL80211_CMD_LEAVE_MESH,
    WS_NL80211_CMD_UNPROT_DEAUTHENTICATE,
    WS_NL80211_CMD_UNPROT_DISASSOCIATE,
    WS_NL80211_CMD_NEW_PEER_CANDIDATE,
    WS_NL80211_CMD_GET_WOWLAN,
    WS_NL80211_CMD_SET_WOWLAN,
    WS_NL80211_CMD_START_SCHED_SCAN,
    WS_NL80211_CMD_STOP_SCHED_SCAN,
    WS_NL80211_CMD_SCHED_SCAN_RESULTS,
    WS_NL80211_CMD_SCHED_SCAN_STOPPED,
    WS_NL80211_CMD_SET_REKEY_OFFLOAD,
    WS_NL80211_CMD_PMKSA_CANDIDATE,
    WS_NL80211_CMD_TDLS_OPER,
    WS_NL80211_CMD_TDLS_MGMT,
    WS_NL80211_CMD_UNEXPECTED_FRAME,
    WS_NL80211_CMD_PROBE_CLIENT,
    WS_NL80211_CMD_REGISTER_BEACONS,
    WS_NL80211_CMD_UNEXPECTED_4ADDR_FRAME,
    WS_NL80211_CMD_SET_NOACK_MAP,
    WS_NL80211_CMD_CH_SWITCH_NOTIFY,
    WS_NL80211_CMD_START_P2P_DEVICE,
    WS_NL80211_CMD_STOP_P2P_DEVICE,
    WS_NL80211_CMD_CONN_FAILED,
    WS_NL80211_CMD_SET_MCAST_RATE,
    WS_NL80211_CMD_SET_MAC_ACL,
    WS_NL80211_CMD_RADAR_DETECT,
    WS_NL80211_CMD_GET_PROTOCOL_FEATURES,
    WS_NL80211_CMD_UPDATE_FT_IES,
    WS_NL80211_CMD_FT_EVENT,
    WS_NL80211_CMD_CRIT_PROTOCOL_START,
    WS_NL80211_CMD_CRIT_PROTOCOL_STOP,
    WS_NL80211_CMD_GET_COALESCE,
    WS_NL80211_CMD_SET_COALESCE,
    WS_NL80211_CMD_CHANNEL_SWITCH,
    WS_NL80211_CMD_VENDOR,
    WS_NL80211_CMD_SET_QOS_MAP,
    WS_NL80211_CMD_ADD_TX_TS,
    WS_NL80211_CMD_DEL_TX_TS,
    WS_NL80211_CMD_GET_MPP,
    WS_NL80211_CMD_JOIN_OCB,
    WS_NL80211_CMD_LEAVE_OCB,
    WS_NL80211_CMD_CH_SWITCH_STARTED_NOTIFY,
    WS_NL80211_CMD_TDLS_CHANNEL_SWITCH,
    WS_NL80211_CMD_TDLS_CANCEL_CHANNEL_SWITCH,
    WS_NL80211_CMD_WIPHY_REG_CHANGE,
    WS_NL80211_CMD_ABORT_SCAN,
    WS_NL80211_CMD_START_NAN,
    WS_NL80211_CMD_STOP_NAN,
    WS_NL80211_CMD_ADD_NAN_FUNCTION,
    WS_NL80211_CMD_DEL_NAN_FUNCTION,
    WS_NL80211_CMD_CHANGE_NAN_CONFIG,
    WS_NL80211_CMD_NAN_MATCH,
    WS_NL80211_CMD_SET_MULTICAST_TO_UNICAST,
    WS_NL80211_CMD_UPDATE_CONNECT_PARAMS,
    WS_NL80211_CMD_SET_PMK,
    WS_NL80211_CMD_DEL_PMK,
    WS_NL80211_CMD_PORT_AUTHORIZED,
    WS_NL80211_CMD_RELOAD_REGDB,
    WS_NL80211_CMD_EXTERNAL_AUTH,
    WS_NL80211_CMD_STA_OPMODE_CHANGED,
    WS_NL80211_CMD_CONTROL_PORT_FRAME,
    WS_NL80211_CMD_GET_FTM_RESPONDER_STATS,
    WS_NL80211_CMD_PEER_MEASUREMENT_START,
    WS_NL80211_CMD_PEER_MEASUREMENT_RESULT,
    WS_NL80211_CMD_PEER_MEASUREMENT_COMPLETE,
    WS_NL80211_CMD_NOTIFY_RADAR,
    WS_NL80211_CMD_UPDATE_OWE_INFO,
    WS_NL80211_CMD_PROBE_MESH_LINK,
    WS_NL80211_CMD_SET_TID_CONFIG,
    WS_NL80211_CMD_UNPROT_BEACON,
    WS_NL80211_CMD_CONTROL_PORT_FRAME_TX_STATUS,
    WS_NL80211_CMD_SET_SAR_SPECS,
    WS_NL80211_CMD_OBSS_COLOR_COLLISION,
    WS_NL80211_CMD_COLOR_CHANGE_REQUEST,
    WS_NL80211_CMD_COLOR_CHANGE_STARTED,
    WS_NL80211_CMD_COLOR_CHANGE_ABORTED,
    WS_NL80211_CMD_COLOR_CHANGE_COMPLETED,
    WS_NL80211_CMD_SET_FILS_AAD,
    WS_NL80211_CMD_ASSOC_COMEBACK,
    WS_NL80211_CMD_ADD_LINK,
    WS_NL80211_CMD_REMOVE_LINK,
    WS_NL80211_CMD_ADD_LINK_STA,
    WS_NL80211_CMD_MODIFY_LINK_STA,
    WS_NL80211_CMD_REMOVE_LINK_STA,
    WS_NL80211_CMD_SET_HW_TIMESTAMP,
    WS_NL80211_CMD_LINKS_REMOVED,
    WS_NL80211_CMD_SET_TID_TO_LINK_MAPPING,
    WS_NL80211_CMD_ASSOC_MLO_RECONF,
    WS_NL80211_CMD_EPCS_CFG,
    WS_NL80211_CMD_NAN_NEXT_DW_NOTIFICATION,
    WS_NL80211_CMD_NAN_CLUSTER_JOINED,
};

enum ws_nl80211_attrs {
    WS_NL80211_ATTR_UNSPEC,
    WS_NL80211_ATTR_WIPHY,
    WS_NL80211_ATTR_WIPHY_NAME,
    WS_NL80211_ATTR_IFINDEX,
    WS_NL80211_ATTR_IFNAME,
    WS_NL80211_ATTR_IFTYPE,
    WS_NL80211_ATTR_MAC,
    WS_NL80211_ATTR_KEY_DATA,
    WS_NL80211_ATTR_KEY_IDX,
    WS_NL80211_ATTR_KEY_CIPHER,
    WS_NL80211_ATTR_KEY_SEQ,
    WS_NL80211_ATTR_KEY_DEFAULT,
    WS_NL80211_ATTR_BEACON_INTERVAL,
    WS_NL80211_ATTR_DTIM_PERIOD,
    WS_NL80211_ATTR_BEACON_HEAD,
    WS_NL80211_ATTR_BEACON_TAIL,
    WS_NL80211_ATTR_STA_AID,
    WS_NL80211_ATTR_STA_FLAGS,
    WS_NL80211_ATTR_STA_LISTEN_INTERVAL,
    WS_NL80211_ATTR_STA_SUPPORTED_RATES,
    WS_NL80211_ATTR_STA_VLAN,
    WS_NL80211_ATTR_STA_INFO,
    WS_NL80211_ATTR_WIPHY_BANDS,
    WS_NL80211_ATTR_MNTR_FLAGS,
    WS_NL80211_ATTR_MESH_ID,
    WS_NL80211_ATTR_STA_PLINK_ACTION,
    WS_NL80211_ATTR_MPATH_NEXT_HOP,
    WS_NL80211_ATTR_MPATH_INFO,
    WS_NL80211_ATTR_BSS_CTS_PROT,
    WS_NL80211_ATTR_BSS_SHORT_PREAMBLE,
    WS_NL80211_ATTR_BSS_SHORT_SLOT_TIME,
    WS_NL80211_ATTR_HT_CAPABILITY,
    WS_NL80211_ATTR_SUPPORTED_IFTYPES,
    WS_NL80211_ATTR_REG_ALPHA2,
    WS_NL80211_ATTR_REG_RULES,
    WS_NL80211_ATTR_MESH_CONFIG,
    WS_NL80211_ATTR_BSS_BASIC_RATES,
    WS_NL80211_ATTR_WIPHY_TXQ_PARAMS,
    WS_NL80211_ATTR_WIPHY_FREQ,
    WS_NL80211_ATTR_WIPHY_CHANNEL_TYPE,
    WS_NL80211_ATTR_KEY_DEFAULT_MGMT,
    WS_NL80211_ATTR_MGMT_SUBTYPE,
    WS_NL80211_ATTR_IE,
    WS_NL80211_ATTR_MAX_NUM_SCAN_SSIDS,
    WS_NL80211_ATTR_SCAN_FREQUENCIES,
    WS_NL80211_ATTR_SCAN_SSIDS,
    WS_NL80211_ATTR_GENERATION,
    WS_NL80211_ATTR_BSS,
    WS_NL80211_ATTR_REG_INITIATOR,
    WS_NL80211_ATTR_REG_TYPE,
    WS_NL80211_ATTR_SUPPORTED_COMMANDS,
    WS_NL80211_ATTR_FRAME,
    WS_NL80211_ATTR_SSID,
    WS_NL80211_ATTR_AUTH_TYPE,
    WS_NL80211_ATTR_REASON_CODE,
    WS_NL80211_ATTR_KEY_TYPE,
    WS_NL80211_ATTR_MAX_SCAN_IE_LEN,
    WS_NL80211_ATTR_CIPHER_SUITES,
    WS_NL80211_ATTR_FREQ_BEFORE,
    WS_NL80211_ATTR_FREQ_AFTER,
    WS_NL80211_ATTR_FREQ_FIXED,
    WS_NL80211_ATTR_WIPHY_RETRY_SHORT,
    WS_NL80211_ATTR_WIPHY_RETRY_LONG,
    WS_NL80211_ATTR_WIPHY_FRAG_THRESHOLD,
    WS_NL80211_ATTR_WIPHY_RTS_THRESHOLD,
    WS_NL80211_ATTR_TIMED_OUT,
    WS_NL80211_ATTR_USE_MFP,
    WS_NL80211_ATTR_STA_FLAGS2,
    WS_NL80211_ATTR_CONTROL_PORT,
    WS_NL80211_ATTR_TESTDATA,
    WS_NL80211_ATTR_PRIVACY,
    WS_NL80211_ATTR_DISCONNECTED_BY_AP,
    WS_NL80211_ATTR_STATUS_CODE,
    WS_NL80211_ATTR_CIPHER_SUITES_PAIRWISE,
    WS_NL80211_ATTR_CIPHER_SUITE_GROUP,
    WS_NL80211_ATTR_WPA_VERSIONS,
    WS_NL80211_ATTR_AKM_SUITES,
    WS_NL80211_ATTR_REQ_IE,
    WS_NL80211_ATTR_RESP_IE,
    WS_NL80211_ATTR_PREV_BSSID,
    WS_NL80211_ATTR_KEY,
    WS_NL80211_ATTR_KEYS,
    WS_NL80211_ATTR_PID,
    WS_NL80211_ATTR_4ADDR,
    WS_NL80211_ATTR_SURVEY_INFO,
    WS_NL80211_ATTR_PMKID,
    WS_NL80211_ATTR_MAX_NUM_PMKIDS,
    WS_NL80211_ATTR_DURATION,
    WS_NL80211_ATTR_COOKIE,
    WS_NL80211_ATTR_WIPHY_COVERAGE_CLASS,
    WS_NL80211_ATTR_TX_RATES,
    WS_NL80211_ATTR_FRAME_MATCH,
    WS_NL80211_ATTR_ACK,
    WS_NL80211_ATTR_PS_STATE,
    WS_NL80211_ATTR_CQM,
    WS_NL80211_ATTR_LOCAL_STATE_CHANGE,
    WS_NL80211_ATTR_AP_ISOLATE,
    WS_NL80211_ATTR_WIPHY_TX_POWER_SETTING,
    WS_NL80211_ATTR_WIPHY_TX_POWER_LEVEL,
    WS_NL80211_ATTR_TX_FRAME_TYPES,
    WS_NL80211_ATTR_RX_FRAME_TYPES,
    WS_NL80211_ATTR_FRAME_TYPE,
    WS_NL80211_ATTR_CONTROL_PORT_ETHERTYPE,
    WS_NL80211_ATTR_CONTROL_PORT_NO_ENCRYPT,
    WS_NL80211_ATTR_SUPPORT_IBSS_RSN,
    WS_NL80211_ATTR_WIPHY_ANTENNA_TX,
    WS_NL80211_ATTR_WIPHY_ANTENNA_RX,
    WS_NL80211_ATTR_MCAST_RATE,
    WS_NL80211_ATTR_OFFCHANNEL_TX_OK,
    WS_NL80211_ATTR_BSS_HT_OPMODE,
    WS_NL80211_ATTR_KEY_DEFAULT_TYPES,
    WS_NL80211_ATTR_MAX_REMAIN_ON_CHANNEL_DURATION,
    WS_NL80211_ATTR_MESH_SETUP,
    WS_NL80211_ATTR_WIPHY_ANTENNA_AVAIL_TX,
    WS_NL80211_ATTR_WIPHY_ANTENNA_AVAIL_RX,
    WS_NL80211_ATTR_SUPPORT_MESH_AUTH,
    WS_NL80211_ATTR_STA_PLINK_STATE,
    WS_NL80211_ATTR_WOWLAN_TRIGGERS,
    WS_NL80211_ATTR_WOWLAN_TRIGGERS_SUPPORTED,
    WS_NL80211_ATTR_SCHED_SCAN_INTERVAL,
    WS_NL80211_ATTR_INTERFACE_COMBINATIONS,
    WS_NL80211_ATTR_SOFTWARE_IFTYPES,
    WS_NL80211_ATTR_REKEY_DATA,
    WS_NL80211_ATTR_MAX_NUM_SCHED_SCAN_SSIDS,
    WS_NL80211_ATTR_MAX_SCHED_SCAN_IE_LEN,
    WS_NL80211_ATTR_SCAN_SUPP_RATES,
    WS_NL80211_ATTR_HIDDEN_SSID,
    WS_NL80211_ATTR_IE_PROBE_RESP,
    WS_NL80211_ATTR_IE_ASSOC_RESP,
    WS_NL80211_ATTR_STA_WME,
    WS_NL80211_ATTR_SUPPORT_AP_UAPSD,
    WS_NL80211_ATTR_ROAM_SUPPORT,
    WS_NL80211_ATTR_SCHED_SCAN_MATCH,
    WS_NL80211_ATTR_MAX_MATCH_SETS,
    WS_NL80211_ATTR_PMKSA_CANDIDATE,
    WS_NL80211_ATTR_TX_NO_CCK_RATE,
    WS_NL80211_ATTR_TDLS_ACTION,
    WS_NL80211_ATTR_TDLS_DIALOG_TOKEN,
    WS_NL80211_ATTR_TDLS_OPERATION,
    WS_NL80211_ATTR_TDLS_SUPPORT,
    WS_NL80211_ATTR_TDLS_EXTERNAL_SETUP,
    WS_NL80211_ATTR_DEVICE_AP_SME,
    WS_NL80211_ATTR_DONT_WAIT_FOR_ACK,
    WS_NL80211_ATTR_FEATURE_FLAGS,
    WS_NL80211_ATTR_PROBE_RESP_OFFLOAD,
    WS_NL80211_ATTR_PROBE_RESP,
    WS_NL80211_ATTR_DFS_REGION,
    WS_NL80211_ATTR_DISABLE_HT,
    WS_NL80211_ATTR_HT_CAPABILITY_MASK,
    WS_NL80211_ATTR_NOACK_MAP,
    WS_NL80211_ATTR_INACTIVITY_TIMEOUT,
    WS_NL80211_ATTR_RX_SIGNAL_DBM,
    WS_NL80211_ATTR_BG_SCAN_PERIOD,
    WS_NL80211_ATTR_WDEV,
    WS_NL80211_ATTR_USER_REG_HINT_TYPE,
    WS_NL80211_ATTR_CONN_FAILED_REASON,
    WS_NL80211_ATTR_AUTH_DATA,
    WS_NL80211_ATTR_VHT_CAPABILITY,
    WS_NL80211_ATTR_SCAN_FLAGS,
    WS_NL80211_ATTR_CHANNEL_WIDTH,
    WS_NL80211_ATTR_CENTER_FREQ1,
    WS_NL80211_ATTR_CENTER_FREQ2,
    WS_NL80211_ATTR_P2P_CTWINDOW,
    WS_NL80211_ATTR_P2P_OPPPS,
    WS_NL80211_ATTR_LOCAL_MESH_POWER_MODE,
    WS_NL80211_ATTR_ACL_POLICY,
    WS_NL80211_ATTR_MAC_ADDRS,
    WS_NL80211_ATTR_MAC_ACL_MAX,
    WS_NL80211_ATTR_RADAR_EVENT,
    WS_NL80211_ATTR_EXT_CAPA,
    WS_NL80211_ATTR_EXT_CAPA_MASK,
    WS_NL80211_ATTR_STA_CAPABILITY,
    WS_NL80211_ATTR_STA_EXT_CAPABILITY,
    WS_NL80211_ATTR_PROTOCOL_FEATURES,
    WS_NL80211_ATTR_SPLIT_WIPHY_DUMP,
    WS_NL80211_ATTR_DISABLE_VHT,
    WS_NL80211_ATTR_VHT_CAPABILITY_MASK,
    WS_NL80211_ATTR_MDID,
    WS_NL80211_ATTR_IE_RIC,
    WS_NL80211_ATTR_CRIT_PROT_ID,
    WS_NL80211_ATTR_MAX_CRIT_PROT_DURATION,
    WS_NL80211_ATTR_PEER_AID,
    WS_NL80211_ATTR_COALESCE_RULE,
    WS_NL80211_ATTR_CH_SWITCH_COUNT,
    WS_NL80211_ATTR_CH_SWITCH_BLOCK_TX,
    WS_NL80211_ATTR_CSA_IES,
    WS_NL80211_ATTR_CNTDWN_OFFS_BEACON,
    WS_NL80211_ATTR_CNTDWN_OFFS_PRESP,
    WS_NL80211_ATTR_RXMGMT_FLAGS,
    WS_NL80211_ATTR_STA_SUPPORTED_CHANNELS,
    WS_NL80211_ATTR_STA_SUPPORTED_OPER_CLASSES,
    WS_NL80211_ATTR_HANDLE_DFS,
    WS_NL80211_ATTR_SUPPORT_5_MHZ,
    WS_NL80211_ATTR_SUPPORT_10_MHZ,
    WS_NL80211_ATTR_OPMODE_NOTIF,
    WS_NL80211_ATTR_VENDOR_ID,
    WS_NL80211_ATTR_VENDOR_SUBCMD,
    WS_NL80211_ATTR_VENDOR_DATA,
    WS_NL80211_ATTR_VENDOR_EVENTS,
    WS_NL80211_ATTR_QOS_MAP,
    WS_NL80211_ATTR_MAC_HINT,
    WS_NL80211_ATTR_WIPHY_FREQ_HINT,
    WS_NL80211_ATTR_MAX_AP_ASSOC_STA,
    WS_NL80211_ATTR_TDLS_PEER_CAPABILITY,
    WS_NL80211_ATTR_SOCKET_OWNER,
    WS_NL80211_ATTR_CSA_C_OFFSETS_TX,
    WS_NL80211_ATTR_MAX_CSA_COUNTERS,
    WS_NL80211_ATTR_TDLS_INITIATOR,
    WS_NL80211_ATTR_USE_RRM,
    WS_NL80211_ATTR_WIPHY_DYN_ACK,
    WS_NL80211_ATTR_TSID,
    WS_NL80211_ATTR_USER_PRIO,
    WS_NL80211_ATTR_ADMITTED_TIME,
    WS_NL80211_ATTR_SMPS_MODE,
    WS_NL80211_ATTR_OPER_CLASS,
    WS_NL80211_ATTR_MAC_MASK,
    WS_NL80211_ATTR_WIPHY_SELF_MANAGED_REG,
    WS_NL80211_ATTR_EXT_FEATURES,
    WS_NL80211_ATTR_SURVEY_RADIO_STATS,
    WS_NL80211_ATTR_NETNS_FD,
    WS_NL80211_ATTR_SCHED_SCAN_DELAY,
    WS_NL80211_ATTR_REG_INDOOR,
    WS_NL80211_ATTR_MAX_NUM_SCHED_SCAN_PLANS,
    WS_NL80211_ATTR_MAX_SCAN_PLAN_INTERVAL,
    WS_NL80211_ATTR_MAX_SCAN_PLAN_ITERATIONS,
    WS_NL80211_ATTR_SCHED_SCAN_PLANS,
    WS_NL80211_ATTR_PBSS,
    WS_NL80211_ATTR_BSS_SELECT,
    WS_NL80211_ATTR_STA_SUPPORT_P2P_PS,
    WS_NL80211_ATTR_PAD,
    WS_NL80211_ATTR_IFTYPE_EXT_CAPA,
    WS_NL80211_ATTR_MU_MIMO_GROUP_DATA,
    WS_NL80211_ATTR_MU_MIMO_FOLLOW_MAC_ADDR,
    WS_NL80211_ATTR_SCAN_START_TIME_TSF,
    WS_NL80211_ATTR_SCAN_START_TIME_TSF_BSSID,
    WS_NL80211_ATTR_MEASUREMENT_DURATION,
    WS_NL80211_ATTR_MEASUREMENT_DURATION_MANDATORY,
    WS_NL80211_ATTR_MESH_PEER_AID,
    WS_NL80211_ATTR_NAN_MASTER_PREF,
    WS_NL80211_ATTR_BANDS,
    WS_NL80211_ATTR_NAN_FUNC,
    WS_NL80211_ATTR_NAN_MATCH,
    WS_NL80211_ATTR_FILS_KEK,
    WS_NL80211_ATTR_FILS_NONCES,
    WS_NL80211_ATTR_MULTICAST_TO_UNICAST_ENABLED,
    WS_NL80211_ATTR_BSSID,
    WS_NL80211_ATTR_SCHED_SCAN_RELATIVE_RSSI,
    WS_NL80211_ATTR_SCHED_SCAN_RSSI_ADJUST,
    WS_NL80211_ATTR_TIMEOUT_REASON,
    WS_NL80211_ATTR_FILS_ERP_USERNAME,
    WS_NL80211_ATTR_FILS_ERP_REALM,
    WS_NL80211_ATTR_FILS_ERP_NEXT_SEQ_NUM,
    WS_NL80211_ATTR_FILS_ERP_RRK,
    WS_NL80211_ATTR_FILS_CACHE_ID,
    WS_NL80211_ATTR_PMK,
    WS_NL80211_ATTR_SCHED_SCAN_MULTI,
    WS_NL80211_ATTR_SCHED_SCAN_MAX_REQS,
    WS_NL80211_ATTR_WANT_1X_4WAY_HS,
    WS_NL80211_ATTR_PMKR0_NAME,
    WS_NL80211_ATTR_PORT_AUTHORIZED,
    WS_NL80211_ATTR_EXTERNAL_AUTH_ACTION,
    WS_NL80211_ATTR_EXTERNAL_AUTH_SUPPORT,
    WS_NL80211_ATTR_NSS,
    WS_NL80211_ATTR_ACK_SIGNAL,
    WS_NL80211_ATTR_CONTROL_PORT_OVER_NL80211,
    WS_NL80211_ATTR_TXQ_STATS,
    WS_NL80211_ATTR_TXQ_LIMIT,
    WS_NL80211_ATTR_TXQ_MEMORY_LIMIT,
    WS_NL80211_ATTR_TXQ_QUANTUM,
    WS_NL80211_ATTR_HE_CAPABILITY,
    WS_NL80211_ATTR_FTM_RESPONDER,
    WS_NL80211_ATTR_FTM_RESPONDER_STATS,
    WS_NL80211_ATTR_TIMEOUT,
    WS_NL80211_ATTR_PEER_MEASUREMENTS,
    WS_NL80211_ATTR_AIRTIME_WEIGHT,
    WS_NL80211_ATTR_STA_TX_POWER_SETTING,
    WS_NL80211_ATTR_STA_TX_POWER,
    WS_NL80211_ATTR_SAE_PASSWORD,
    WS_NL80211_ATTR_TWT_RESPONDER,
    WS_NL80211_ATTR_HE_OBSS_PD,
    WS_NL80211_ATTR_WIPHY_EDMG_CHANNELS,
    WS_NL80211_ATTR_WIPHY_EDMG_BW_CONFIG,
    WS_NL80211_ATTR_VLAN_ID,
    WS_NL80211_ATTR_HE_BSS_COLOR,
    WS_NL80211_ATTR_IFTYPE_AKM_SUITES,
    WS_NL80211_ATTR_TID_CONFIG,
    WS_NL80211_ATTR_CONTROL_PORT_NO_PREAUTH,
    WS_NL80211_ATTR_PMK_LIFETIME,
    WS_NL80211_ATTR_PMK_REAUTH_THRESHOLD,
    WS_NL80211_ATTR_RECEIVE_MULTICAST,
    WS_NL80211_ATTR_WIPHY_FREQ_OFFSET,
    WS_NL80211_ATTR_CENTER_FREQ1_OFFSET,
    WS_NL80211_ATTR_SCAN_FREQ_KHZ,
    WS_NL80211_ATTR_HE_6GHZ_CAPABILITY,
    WS_NL80211_ATTR_FILS_DISCOVERY,
    WS_NL80211_ATTR_UNSOL_BCAST_PROBE_RESP,
    WS_NL80211_ATTR_S1G_CAPABILITY,
    WS_NL80211_ATTR_S1G_CAPABILITY_MASK,
    WS_NL80211_ATTR_SAE_PWE,
    WS_NL80211_ATTR_RECONNECT_REQUESTED,
    WS_NL80211_ATTR_SAR_SPEC,
    WS_NL80211_ATTR_DISABLE_HE,
    WS_NL80211_ATTR_OBSS_COLOR_BITMAP,
    WS_NL80211_ATTR_COLOR_CHANGE_COUNT,
    WS_NL80211_ATTR_COLOR_CHANGE_COLOR,
    WS_NL80211_ATTR_COLOR_CHANGE_ELEMS,
    WS_NL80211_ATTR_MBSSID_CONFIG,
    WS_NL80211_ATTR_MBSSID_ELEMS,
    WS_NL80211_ATTR_RADAR_BACKGROUND,
    WS_NL80211_ATTR_AP_SETTINGS_FLAGS,
    WS_NL80211_ATTR_EHT_CAPABILITY,
    WS_NL80211_ATTR_DISABLE_EHT,
    WS_NL80211_ATTR_MLO_LINKS,
    WS_NL80211_ATTR_MLO_LINK_ID,
    WS_NL80211_ATTR_MLD_ADDR,
    WS_NL80211_ATTR_MLO_SUPPORT,
    WS_NL80211_ATTR_MAX_NUM_AKM_SUITES,
    WS_NL80211_ATTR_EML_CAPABILITY,
    WS_NL80211_ATTR_MLD_CAPA_AND_OPS,
    WS_NL80211_ATTR_TX_HW_TIMESTAMP,
    WS_NL80211_ATTR_RX_HW_TIMESTAMP,
    WS_NL80211_ATTR_TD_BITMAP,
    WS_NL80211_ATTR_PUNCT_BITMAP,
    WS_NL80211_ATTR_MAX_HW_TIMESTAMP_PEERS,
    WS_NL80211_ATTR_HW_TIMESTAMP_ENABLED,
    WS_NL80211_ATTR_EMA_RNR_ELEMS,
    WS_NL80211_ATTR_MLO_LINK_DISABLED,
    WS_NL80211_ATTR_BSS_DUMP_INCLUDE_USE_DATA,
    WS_NL80211_ATTR_MLO_TTLM_DLINK,
    WS_NL80211_ATTR_MLO_TTLM_ULINK,
    WS_NL80211_ATTR_ASSOC_SPP_AMSDU,
    WS_NL80211_ATTR_WIPHY_RADIOS,
    WS_NL80211_ATTR_WIPHY_INTERFACE_COMBINATIONS,
    WS_NL80211_ATTR_VIF_RADIO_MASK,
    WS_NL80211_ATTR_SUPPORTED_SELECTORS,
    WS_NL80211_ATTR_MLO_RECONF_REM_LINKS,
    WS_NL80211_ATTR_EPCS,
    WS_NL80211_ATTR_ASSOC_MLD_EXT_CAPA_OPS,
    WS_NL80211_ATTR_WIPHY_RADIO_INDEX,
    WS_NL80211_ATTR_S1G_LONG_BEACON_PERIOD,
    WS_NL80211_ATTR_S1G_SHORT_BEACON,
    WS_NL80211_ATTR_BSS_PARAM,
    WS_NL80211_ATTR_NAN_CONFIG,
    WS_NL80211_ATTR_NAN_NEW_CLUSTER,
    WS_NL80211_ATTR_NAN_CAPABILITIES,
    WS_NL80211_ATTR_S1G_PRIMARY_2MHZ,
};

enum ws_nl80211_iftype {
    WS_NL80211_IFTYPE_UNSPECIFIED,
    WS_NL80211_IFTYPE_ADHOC,
    WS_NL80211_IFTYPE_STATION,
    WS_NL80211_IFTYPE_AP,
    WS_NL80211_IFTYPE_AP_VLAN,
    WS_NL80211_IFTYPE_WDS,
    WS_NL80211_IFTYPE_MONITOR,
    WS_NL80211_IFTYPE_MESH_POINT,
    WS_NL80211_IFTYPE_P2P_CLIENT,
    WS_NL80211_IFTYPE_P2P_GO,
    WS_NL80211_IFTYPE_P2P_DEVICE,
    WS_NL80211_IFTYPE_OCB,
    WS_NL80211_IFTYPE_NAN,
};

enum ws_nl80211_sta_flags {
    WS___NL80211_STA_FLAG_INVALID,
    WS_NL80211_STA_FLAG_AUTHORIZED,
    WS_NL80211_STA_FLAG_SHORT_PREAMBLE,
    WS_NL80211_STA_FLAG_WME,
    WS_NL80211_STA_FLAG_MFP,
    WS_NL80211_STA_FLAG_AUTHENTICATED,
    WS_NL80211_STA_FLAG_TDLS_PEER,
    WS_NL80211_STA_FLAG_ASSOCIATED,
    WS_NL80211_STA_FLAG_SPP_AMSDU,
};

enum ws_nl80211_sta_p2p_ps_status {
    WS_NL80211_P2P_PS_UNSUPPORTED = 0,
    WS_NL80211_P2P_PS_SUPPORTED,
};

enum ws_nl80211_he_gi {
    WS_NL80211_RATE_INFO_HE_GI_0_8,
    WS_NL80211_RATE_INFO_HE_GI_1_6,
    WS_NL80211_RATE_INFO_HE_GI_3_2,
};

enum ws_nl80211_he_ltf {
    WS_NL80211_RATE_INFO_HE_1XLTF,
    WS_NL80211_RATE_INFO_HE_2XLTF,
    WS_NL80211_RATE_INFO_HE_4XLTF,
};

enum ws_nl80211_he_ru_alloc {
    WS_NL80211_RATE_INFO_HE_RU_ALLOC_26,
    WS_NL80211_RATE_INFO_HE_RU_ALLOC_52,
    WS_NL80211_RATE_INFO_HE_RU_ALLOC_106,
    WS_NL80211_RATE_INFO_HE_RU_ALLOC_242,
    WS_NL80211_RATE_INFO_HE_RU_ALLOC_484,
    WS_NL80211_RATE_INFO_HE_RU_ALLOC_996,
    WS_NL80211_RATE_INFO_HE_RU_ALLOC_2x996,
};

enum ws_nl80211_eht_gi {
    WS_NL80211_RATE_INFO_EHT_GI_0_8,
    WS_NL80211_RATE_INFO_EHT_GI_1_6,
    WS_NL80211_RATE_INFO_EHT_GI_3_2,
};

enum ws_nl80211_eht_ru_alloc {
    WS_NL80211_RATE_INFO_EHT_RU_ALLOC_26,
    WS_NL80211_RATE_INFO_EHT_RU_ALLOC_52,
    WS_NL80211_RATE_INFO_EHT_RU_ALLOC_52P26,
    WS_NL80211_RATE_INFO_EHT_RU_ALLOC_106,
    WS_NL80211_RATE_INFO_EHT_RU_ALLOC_106P26,
    WS_NL80211_RATE_INFO_EHT_RU_ALLOC_242,
    WS_NL80211_RATE_INFO_EHT_RU_ALLOC_484,
    WS_NL80211_RATE_INFO_EHT_RU_ALLOC_484P242,
    WS_NL80211_RATE_INFO_EHT_RU_ALLOC_996,
    WS_NL80211_RATE_INFO_EHT_RU_ALLOC_996P484,
    WS_NL80211_RATE_INFO_EHT_RU_ALLOC_996P484P242,
    WS_NL80211_RATE_INFO_EHT_RU_ALLOC_2x996,
    WS_NL80211_RATE_INFO_EHT_RU_ALLOC_2x996P484,
    WS_NL80211_RATE_INFO_EHT_RU_ALLOC_3x996,
    WS_NL80211_RATE_INFO_EHT_RU_ALLOC_3x996P484,
    WS_NL80211_RATE_INFO_EHT_RU_ALLOC_4x996,
};

enum ws_nl80211_rate_info {
    WS___NL80211_RATE_INFO_INVALID,
    WS_NL80211_RATE_INFO_BITRATE,
    WS_NL80211_RATE_INFO_MCS,
    WS_NL80211_RATE_INFO_40_MHZ_WIDTH,
    WS_NL80211_RATE_INFO_SHORT_GI,
    WS_NL80211_RATE_INFO_BITRATE32,
    WS_NL80211_RATE_INFO_VHT_MCS,
    WS_NL80211_RATE_INFO_VHT_NSS,
    WS_NL80211_RATE_INFO_80_MHZ_WIDTH,
    WS_NL80211_RATE_INFO_80P80_MHZ_WIDTH,
    WS_NL80211_RATE_INFO_160_MHZ_WIDTH,
    WS_NL80211_RATE_INFO_10_MHZ_WIDTH,
    WS_NL80211_RATE_INFO_5_MHZ_WIDTH,
    WS_NL80211_RATE_INFO_HE_MCS,
    WS_NL80211_RATE_INFO_HE_NSS,
    WS_NL80211_RATE_INFO_HE_GI,
    WS_NL80211_RATE_INFO_HE_DCM,
    WS_NL80211_RATE_INFO_HE_RU_ALLOC,
    WS_NL80211_RATE_INFO_320_MHZ_WIDTH,
    WS_NL80211_RATE_INFO_EHT_MCS,
    WS_NL80211_RATE_INFO_EHT_NSS,
    WS_NL80211_RATE_INFO_EHT_GI,
    WS_NL80211_RATE_INFO_EHT_RU_ALLOC,
    WS_NL80211_RATE_INFO_S1G_MCS,
    WS_NL80211_RATE_INFO_S1G_NSS,
    WS_NL80211_RATE_INFO_1_MHZ_WIDTH,
    WS_NL80211_RATE_INFO_2_MHZ_WIDTH,
    WS_NL80211_RATE_INFO_4_MHZ_WIDTH,
    WS_NL80211_RATE_INFO_8_MHZ_WIDTH,
    WS_NL80211_RATE_INFO_16_MHZ_WIDTH,
};

enum ws_nl80211_sta_bss_param {
    WS___NL80211_STA_BSS_PARAM_INVALID,
    WS_NL80211_STA_BSS_PARAM_CTS_PROT,
    WS_NL80211_STA_BSS_PARAM_SHORT_PREAMBLE,
    WS_NL80211_STA_BSS_PARAM_SHORT_SLOT_TIME,
    WS_NL80211_STA_BSS_PARAM_DTIM_PERIOD,
    WS_NL80211_STA_BSS_PARAM_BEACON_INTERVAL,
};

enum ws_nl80211_sta_info {
    WS___NL80211_STA_INFO_INVALID,
    WS_NL80211_STA_INFO_INACTIVE_TIME,
    WS_NL80211_STA_INFO_RX_BYTES,
    WS_NL80211_STA_INFO_TX_BYTES,
    WS_NL80211_STA_INFO_LLID,
    WS_NL80211_STA_INFO_PLID,
    WS_NL80211_STA_INFO_PLINK_STATE,
    WS_NL80211_STA_INFO_SIGNAL,
    WS_NL80211_STA_INFO_TX_BITRATE,
    WS_NL80211_STA_INFO_RX_PACKETS,
    WS_NL80211_STA_INFO_TX_PACKETS,
    WS_NL80211_STA_INFO_TX_RETRIES,
    WS_NL80211_STA_INFO_TX_FAILED,
    WS_NL80211_STA_INFO_SIGNAL_AVG,
    WS_NL80211_STA_INFO_RX_BITRATE,
    WS_NL80211_STA_INFO_BSS_PARAM,
    WS_NL80211_STA_INFO_CONNECTED_TIME,
    WS_NL80211_STA_INFO_STA_FLAGS,
    WS_NL80211_STA_INFO_BEACON_LOSS,
    WS_NL80211_STA_INFO_T_OFFSET,
    WS_NL80211_STA_INFO_LOCAL_PM,
    WS_NL80211_STA_INFO_PEER_PM,
    WS_NL80211_STA_INFO_NONPEER_PM,
    WS_NL80211_STA_INFO_RX_BYTES64,
    WS_NL80211_STA_INFO_TX_BYTES64,
    WS_NL80211_STA_INFO_CHAIN_SIGNAL,
    WS_NL80211_STA_INFO_CHAIN_SIGNAL_AVG,
    WS_NL80211_STA_INFO_EXPECTED_THROUGHPUT,
    WS_NL80211_STA_INFO_RX_DROP_MISC,
    WS_NL80211_STA_INFO_BEACON_RX,
    WS_NL80211_STA_INFO_BEACON_SIGNAL_AVG,
    WS_NL80211_STA_INFO_TID_STATS,
    WS_NL80211_STA_INFO_RX_DURATION,
    WS_NL80211_STA_INFO_PAD,
    WS_NL80211_STA_INFO_ACK_SIGNAL,
    WS_NL80211_STA_INFO_ACK_SIGNAL_AVG,
    WS_NL80211_STA_INFO_RX_MPDUS,
    WS_NL80211_STA_INFO_FCS_ERROR_COUNT,
    WS_NL80211_STA_INFO_CONNECTED_TO_GATE,
    WS_NL80211_STA_INFO_TX_DURATION,
    WS_NL80211_STA_INFO_AIRTIME_WEIGHT,
    WS_NL80211_STA_INFO_AIRTIME_LINK_METRIC,
    WS_NL80211_STA_INFO_ASSOC_AT_BOOTTIME,
    WS_NL80211_STA_INFO_CONNECTED_TO_AS,
};

enum ws_nl80211_tid_stats {
    WS___NL80211_TID_STATS_INVALID,
    WS_NL80211_TID_STATS_RX_MSDU,
    WS_NL80211_TID_STATS_TX_MSDU,
    WS_NL80211_TID_STATS_TX_MSDU_RETRIES,
    WS_NL80211_TID_STATS_TX_MSDU_FAILED,
    WS_NL80211_TID_STATS_PAD,
    WS_NL80211_TID_STATS_TXQ_STATS,
};

enum ws_nl80211_txq_stats {
    WS___NL80211_TXQ_STATS_INVALID,
    WS_NL80211_TXQ_STATS_BACKLOG_BYTES,
    WS_NL80211_TXQ_STATS_BACKLOG_PACKETS,
    WS_NL80211_TXQ_STATS_FLOWS,
    WS_NL80211_TXQ_STATS_DROPS,
    WS_NL80211_TXQ_STATS_ECN_MARKS,
    WS_NL80211_TXQ_STATS_OVERLIMIT,
    WS_NL80211_TXQ_STATS_OVERMEMORY,
    WS_NL80211_TXQ_STATS_COLLISIONS,
    WS_NL80211_TXQ_STATS_TX_BYTES,
    WS_NL80211_TXQ_STATS_TX_PACKETS,
    WS_NL80211_TXQ_STATS_MAX_FLOWS,
};

enum ws_nl80211_mpath_flags {
    WS_NL80211_MPATH_FLAG_ACTIVE = 1<<0,
    WS_NL80211_MPATH_FLAG_RESOLVING = 1<<1,
    WS_NL80211_MPATH_FLAG_SN_VALID = 1<<2,
    WS_NL80211_MPATH_FLAG_FIXED = 1<<3,
    WS_NL80211_MPATH_FLAG_RESOLVED = 1<<4,
};

enum ws_nl80211_mpath_info {
    WS___NL80211_MPATH_INFO_INVALID,
    WS_NL80211_MPATH_INFO_FRAME_QLEN,
    WS_NL80211_MPATH_INFO_SN,
    WS_NL80211_MPATH_INFO_METRIC,
    WS_NL80211_MPATH_INFO_EXPTIME,
    WS_NL80211_MPATH_INFO_FLAGS,
    WS_NL80211_MPATH_INFO_DISCOVERY_TIMEOUT,
    WS_NL80211_MPATH_INFO_DISCOVERY_RETRIES,
    WS_NL80211_MPATH_INFO_HOP_COUNT,
    WS_NL80211_MPATH_INFO_PATH_CHANGE,
};

enum ws_nl80211_band_iftype_attr {
    WS___NL80211_BAND_IFTYPE_ATTR_INVALID,
    WS_NL80211_BAND_IFTYPE_ATTR_IFTYPES,
    WS_NL80211_BAND_IFTYPE_ATTR_HE_CAP_MAC,
    WS_NL80211_BAND_IFTYPE_ATTR_HE_CAP_PHY,
    WS_NL80211_BAND_IFTYPE_ATTR_HE_CAP_MCS_SET,
    WS_NL80211_BAND_IFTYPE_ATTR_HE_CAP_PPE,
    WS_NL80211_BAND_IFTYPE_ATTR_HE_6GHZ_CAPA,
    WS_NL80211_BAND_IFTYPE_ATTR_VENDOR_ELEMS,
    WS_NL80211_BAND_IFTYPE_ATTR_EHT_CAP_MAC,
    WS_NL80211_BAND_IFTYPE_ATTR_EHT_CAP_PHY,
    WS_NL80211_BAND_IFTYPE_ATTR_EHT_CAP_MCS_SET,
    WS_NL80211_BAND_IFTYPE_ATTR_EHT_CAP_PPE,
};

enum ws_nl80211_band_attr {
    WS___NL80211_BAND_ATTR_INVALID,
    WS_NL80211_BAND_ATTR_FREQS,
    WS_NL80211_BAND_ATTR_RATES,
    WS_NL80211_BAND_ATTR_HT_MCS_SET,
    WS_NL80211_BAND_ATTR_HT_CAPA,
    WS_NL80211_BAND_ATTR_HT_AMPDU_FACTOR,
    WS_NL80211_BAND_ATTR_HT_AMPDU_DENSITY,
    WS_NL80211_BAND_ATTR_VHT_MCS_SET,
    WS_NL80211_BAND_ATTR_VHT_CAPA,
    WS_NL80211_BAND_ATTR_IFTYPE_DATA,
    WS_NL80211_BAND_ATTR_EDMG_CHANNELS,
    WS_NL80211_BAND_ATTR_EDMG_BW_CONFIG,
    WS_NL80211_BAND_ATTR_S1G_MCS_NSS_SET,
    WS_NL80211_BAND_ATTR_S1G_CAPA,
};

enum ws_nl80211_wmm_rule {
    WS___NL80211_WMMR_INVALID,
    WS_NL80211_WMMR_CW_MIN,
    WS_NL80211_WMMR_CW_MAX,
    WS_NL80211_WMMR_AIFSN,
    WS_NL80211_WMMR_TXOP,
};

enum ws_nl80211_frequency_attr {
    WS___NL80211_FREQUENCY_ATTR_INVALID,
    WS_NL80211_FREQUENCY_ATTR_FREQ,
    WS_NL80211_FREQUENCY_ATTR_DISABLED,
    WS_NL80211_FREQUENCY_ATTR_NO_IR,
    WS___NL80211_FREQUENCY_ATTR_NO_IBSS,
    WS_NL80211_FREQUENCY_ATTR_RADAR,
    WS_NL80211_FREQUENCY_ATTR_MAX_TX_POWER,
    WS_NL80211_FREQUENCY_ATTR_DFS_STATE,
    WS_NL80211_FREQUENCY_ATTR_DFS_TIME,
    WS_NL80211_FREQUENCY_ATTR_NO_HT40_MINUS,
    WS_NL80211_FREQUENCY_ATTR_NO_HT40_PLUS,
    WS_NL80211_FREQUENCY_ATTR_NO_80MHZ,
    WS_NL80211_FREQUENCY_ATTR_NO_160MHZ,
    WS_NL80211_FREQUENCY_ATTR_DFS_CAC_TIME,
    WS_NL80211_FREQUENCY_ATTR_INDOOR_ONLY,
    WS_NL80211_FREQUENCY_ATTR_IR_CONCURRENT,
    WS_NL80211_FREQUENCY_ATTR_NO_20MHZ,
    WS_NL80211_FREQUENCY_ATTR_NO_10MHZ,
    WS_NL80211_FREQUENCY_ATTR_WMM,
    WS_NL80211_FREQUENCY_ATTR_NO_HE,
    WS_NL80211_FREQUENCY_ATTR_OFFSET,
    WS_NL80211_FREQUENCY_ATTR_1MHZ,
    WS_NL80211_FREQUENCY_ATTR_2MHZ,
    WS_NL80211_FREQUENCY_ATTR_4MHZ,
    WS_NL80211_FREQUENCY_ATTR_8MHZ,
    WS_NL80211_FREQUENCY_ATTR_16MHZ,
    WS_NL80211_FREQUENCY_ATTR_NO_320MHZ,
    WS_NL80211_FREQUENCY_ATTR_NO_EHT,
    WS_NL80211_FREQUENCY_ATTR_PSD,
    WS_NL80211_FREQUENCY_ATTR_DFS_CONCURRENT,
    WS_NL80211_FREQUENCY_ATTR_NO_6GHZ_VLP_CLIENT,
    WS_NL80211_FREQUENCY_ATTR_NO_6GHZ_AFC_CLIENT,
    WS_NL80211_FREQUENCY_ATTR_CAN_MONITOR,
    WS_NL80211_FREQUENCY_ATTR_ALLOW_6GHZ_VLP_AP,
    WS_NL80211_FREQUENCY_ATTR_ALLOW_20MHZ_ACTIVITY,
    WS_NL80211_FREQUENCY_ATTR_NO_4MHZ,
    WS_NL80211_FREQUENCY_ATTR_NO_8MHZ,
    WS_NL80211_FREQUENCY_ATTR_NO_16MHZ,
};

enum ws_nl80211_bitrate_attr {
    WS___NL80211_BITRATE_ATTR_INVALID,
    WS_NL80211_BITRATE_ATTR_RATE,
    WS_NL80211_BITRATE_ATTR_2GHZ_SHORTPREAMBLE,
};

enum ws_nl80211_reg_initiator {
    WS_NL80211_REGDOM_SET_BY_CORE,
    WS_NL80211_REGDOM_SET_BY_USER,
    WS_NL80211_REGDOM_SET_BY_DRIVER,
    WS_NL80211_REGDOM_SET_BY_COUNTRY_IE,
};

enum ws_nl80211_reg_type {
    WS_NL80211_REGDOM_TYPE_COUNTRY,
    WS_NL80211_REGDOM_TYPE_WORLD,
    WS_NL80211_REGDOM_TYPE_CUSTOM_WORLD,
    WS_NL80211_REGDOM_TYPE_INTERSECTION,
};

enum ws_nl80211_reg_rule_attr {
    WS___NL80211_REG_RULE_ATTR_INVALID,
    WS_NL80211_ATTR_REG_RULE_FLAGS,
    WS_NL80211_ATTR_FREQ_RANGE_START,
    WS_NL80211_ATTR_FREQ_RANGE_END,
    WS_NL80211_ATTR_FREQ_RANGE_MAX_BW,
    WS_NL80211_ATTR_POWER_RULE_MAX_ANT_GAIN,
    WS_NL80211_ATTR_POWER_RULE_MAX_EIRP,
    WS_NL80211_ATTR_DFS_CAC_TIME,
    WS_NL80211_ATTR_POWER_RULE_PSD,
};

enum ws_nl80211_sched_scan_match_attr {
    WS___NL80211_SCHED_SCAN_MATCH_ATTR_INVALID,
    WS_NL80211_SCHED_SCAN_MATCH_ATTR_SSID,
    WS_NL80211_SCHED_SCAN_MATCH_ATTR_RSSI,
    WS_NL80211_SCHED_SCAN_MATCH_ATTR_RELATIVE_RSSI,
    WS_NL80211_SCHED_SCAN_MATCH_ATTR_RSSI_ADJUST,
    WS_NL80211_SCHED_SCAN_MATCH_ATTR_BSSID,
    WS_NL80211_SCHED_SCAN_MATCH_PER_BAND_RSSI,
};

enum ws_nl80211_reg_rule_flags {
    WS_NL80211_RRF_NO_OFDM = 1 << 0,
    WS_NL80211_RRF_NO_CCK = 1 << 1,
    WS_NL80211_RRF_NO_INDOOR = 1 << 2,
    WS_NL80211_RRF_NO_OUTDOOR = 1 << 3,
    WS_NL80211_RRF_DFS = 1 << 4,
    WS_NL80211_RRF_PTP_ONLY = 1 << 5,
    WS_NL80211_RRF_PTMP_ONLY = 1 << 6,
    WS_NL80211_RRF_NO_IR = 1 << 7,
    WS___NL80211_RRF_NO_IBSS = 1 << 8,
    WS_NL80211_RRF_AUTO_BW = 1 << 11,
    WS_NL80211_RRF_IR_CONCURRENT = 1 << 12,
    WS_NL80211_RRF_NO_HT40MINUS = 1 << 13,
    WS_NL80211_RRF_NO_HT40PLUS = 1 << 14,
    WS_NL80211_RRF_NO_80MHZ = 1 << 15,
    WS_NL80211_RRF_NO_160MHZ = 1 << 16,
    WS_NL80211_RRF_NO_HE = 1 << 17,
    WS_NL80211_RRF_NO_320MHZ = 1 << 18,
    WS_NL80211_RRF_NO_EHT = 1 << 19,
    WS_NL80211_RRF_PSD = 1 << 20,
    WS_NL80211_RRF_DFS_CONCURRENT = 1 << 21,
    WS_NL80211_RRF_NO_6GHZ_VLP_CLIENT = 1 << 22,
    WS_NL80211_RRF_NO_6GHZ_AFC_CLIENT = 1 << 23,
    WS_NL80211_RRF_ALLOW_6GHZ_VLP_AP = 1 << 24,
    WS_NL80211_RRF_ALLOW_20MHZ_ACTIVITY = 1 << 25,
};

enum ws_nl80211_dfs_regions {
    WS_NL80211_DFS_UNSET = 0,
    WS_NL80211_DFS_FCC = 1,
    WS_NL80211_DFS_ETSI = 2,
    WS_NL80211_DFS_JP = 3,
};

enum ws_nl80211_user_reg_hint_type {
    WS_NL80211_USER_REG_HINT_USER = 0,
    WS_NL80211_USER_REG_HINT_CELL_BASE = 1,
    WS_NL80211_USER_REG_HINT_INDOOR = 2,
};

enum ws_nl80211_survey_info {
    WS___NL80211_SURVEY_INFO_INVALID,
    WS_NL80211_SURVEY_INFO_FREQUENCY,
    WS_NL80211_SURVEY_INFO_NOISE,
    WS_NL80211_SURVEY_INFO_IN_USE,
    WS_NL80211_SURVEY_INFO_TIME,
    WS_NL80211_SURVEY_INFO_TIME_BUSY,
    WS_NL80211_SURVEY_INFO_TIME_EXT_BUSY,
    WS_NL80211_SURVEY_INFO_TIME_RX,
    WS_NL80211_SURVEY_INFO_TIME_TX,
    WS_NL80211_SURVEY_INFO_TIME_SCAN,
    WS_NL80211_SURVEY_INFO_PAD,
    WS_NL80211_SURVEY_INFO_TIME_BSS_RX,
    WS_NL80211_SURVEY_INFO_FREQUENCY_OFFSET,
};

enum ws_nl80211_mntr_flags {
    WS___NL80211_MNTR_FLAG_INVALID,
    WS_NL80211_MNTR_FLAG_FCSFAIL,
    WS_NL80211_MNTR_FLAG_PLCPFAIL,
    WS_NL80211_MNTR_FLAG_CONTROL,
    WS_NL80211_MNTR_FLAG_OTHER_BSS,
    WS_NL80211_MNTR_FLAG_COOK_FRAMES,
    WS_NL80211_MNTR_FLAG_ACTIVE,
    WS_NL80211_MNTR_FLAG_SKIP_TX,
};

enum ws_nl80211_mesh_power_mode {
    WS_NL80211_MESH_POWER_UNKNOWN,
    WS_NL80211_MESH_POWER_ACTIVE,
    WS_NL80211_MESH_POWER_LIGHT_SLEEP,
    WS_NL80211_MESH_POWER_DEEP_SLEEP,
};

enum ws_nl80211_meshconf_params {
    WS___NL80211_MESHCONF_INVALID,
    WS_NL80211_MESHCONF_RETRY_TIMEOUT,
    WS_NL80211_MESHCONF_CONFIRM_TIMEOUT,
    WS_NL80211_MESHCONF_HOLDING_TIMEOUT,
    WS_NL80211_MESHCONF_MAX_PEER_LINKS,
    WS_NL80211_MESHCONF_MAX_RETRIES,
    WS_NL80211_MESHCONF_TTL,
    WS_NL80211_MESHCONF_AUTO_OPEN_PLINKS,
    WS_NL80211_MESHCONF_HWMP_MAX_PREQ_RETRIES,
    WS_NL80211_MESHCONF_PATH_REFRESH_TIME,
    WS_NL80211_MESHCONF_MIN_DISCOVERY_TIMEOUT,
    WS_NL80211_MESHCONF_HWMP_ACTIVE_PATH_TIMEOUT,
    WS_NL80211_MESHCONF_HWMP_PREQ_MIN_INTERVAL,
    WS_NL80211_MESHCONF_HWMP_NET_DIAM_TRVS_TIME,
    WS_NL80211_MESHCONF_HWMP_ROOTMODE,
    WS_NL80211_MESHCONF_ELEMENT_TTL,
    WS_NL80211_MESHCONF_HWMP_RANN_INTERVAL,
    WS_NL80211_MESHCONF_GATE_ANNOUNCEMENTS,
    WS_NL80211_MESHCONF_HWMP_PERR_MIN_INTERVAL,
    WS_NL80211_MESHCONF_FORWARDING,
    WS_NL80211_MESHCONF_RSSI_THRESHOLD,
    WS_NL80211_MESHCONF_SYNC_OFFSET_MAX_NEIGHBOR,
    WS_NL80211_MESHCONF_HT_OPMODE,
    WS_NL80211_MESHCONF_HWMP_PATH_TO_ROOT_TIMEOUT,
    WS_NL80211_MESHCONF_HWMP_ROOT_INTERVAL,
    WS_NL80211_MESHCONF_HWMP_CONFIRMATION_INTERVAL,
    WS_NL80211_MESHCONF_POWER_MODE,
    WS_NL80211_MESHCONF_AWAKE_WINDOW,
    WS_NL80211_MESHCONF_PLINK_TIMEOUT,
    WS_NL80211_MESHCONF_CONNECTED_TO_GATE,
    WS_NL80211_MESHCONF_NOLEARN,
    WS_NL80211_MESHCONF_CONNECTED_TO_AS,
};

enum ws_nl80211_mesh_setup_params {
    WS___NL80211_MESH_SETUP_INVALID,
    WS_NL80211_MESH_SETUP_ENABLE_VENDOR_PATH_SEL,
    WS_NL80211_MESH_SETUP_ENABLE_VENDOR_METRIC,
    WS_NL80211_MESH_SETUP_IE,
    WS_NL80211_MESH_SETUP_USERSPACE_AUTH,
    WS_NL80211_MESH_SETUP_USERSPACE_AMPE,
    WS_NL80211_MESH_SETUP_ENABLE_VENDOR_SYNC,
    WS_NL80211_MESH_SETUP_USERSPACE_MPM,
    WS_NL80211_MESH_SETUP_AUTH_PROTOCOL,
};

enum ws_nl80211_txq_attr {
    WS___NL80211_TXQ_ATTR_INVALID,
    WS_NL80211_TXQ_ATTR_AC,
    WS_NL80211_TXQ_ATTR_TXOP,
    WS_NL80211_TXQ_ATTR_CWMIN,
    WS_NL80211_TXQ_ATTR_CWMAX,
    WS_NL80211_TXQ_ATTR_AIFS,
};

enum ws_nl80211_ac {
    WS_NL80211_AC_VO,
    WS_NL80211_AC_VI,
    WS_NL80211_AC_BE,
    WS_NL80211_AC_BK,
    WS_NL80211_NUM_ACS,
};

enum ws_nl80211_channel_type {
    WS_NL80211_CHAN_NO_HT,
    WS_NL80211_CHAN_HT20,
    WS_NL80211_CHAN_HT40MINUS,
    WS_NL80211_CHAN_HT40PLUS,
};

enum ws_nl80211_key_mode {
    WS_NL80211_KEY_RX_TX,
    WS_NL80211_KEY_NO_TX,
    WS_NL80211_KEY_SET_TX,
};

enum ws_nl80211_chan_width {
    WS_NL80211_CHAN_WIDTH_20_NOHT,
    WS_NL80211_CHAN_WIDTH_20,
    WS_NL80211_CHAN_WIDTH_40,
    WS_NL80211_CHAN_WIDTH_80,
    WS_NL80211_CHAN_WIDTH_80P80,
    WS_NL80211_CHAN_WIDTH_160,
    WS_NL80211_CHAN_WIDTH_5,
    WS_NL80211_CHAN_WIDTH_10,
    WS_NL80211_CHAN_WIDTH_1,
    WS_NL80211_CHAN_WIDTH_2,
    WS_NL80211_CHAN_WIDTH_4,
    WS_NL80211_CHAN_WIDTH_8,
    WS_NL80211_CHAN_WIDTH_16,
    WS_NL80211_CHAN_WIDTH_320,
};

enum ws_nl80211_bss_scan_width {
    WS_NL80211_BSS_CHAN_WIDTH_20,
    WS_NL80211_BSS_CHAN_WIDTH_10,
    WS_NL80211_BSS_CHAN_WIDTH_5,
    WS_NL80211_BSS_CHAN_WIDTH_1,
    WS_NL80211_BSS_CHAN_WIDTH_2,
};

enum ws_nl80211_bss_use_for {
    WS_NL80211_BSS_USE_FOR_NORMAL = 1 << 0,
    WS_NL80211_BSS_USE_FOR_MLD_LINK = 1 << 1,
};

enum ws_nl80211_bss_cannot_use_reasons {
    WS_NL80211_BSS_CANNOT_USE_NSTR_NONPRIMARY = 1 << 0,
    WS_NL80211_BSS_CANNOT_USE_6GHZ_PWR_MISMATCH = 1 << 1,
};

enum ws_nl80211_bss {
    WS___NL80211_BSS_INVALID,
    WS_NL80211_BSS_BSSID,
    WS_NL80211_BSS_FREQUENCY,
    WS_NL80211_BSS_TSF,
    WS_NL80211_BSS_BEACON_INTERVAL,
    WS_NL80211_BSS_CAPABILITY,
    WS_NL80211_BSS_INFORMATION_ELEMENTS,
    WS_NL80211_BSS_SIGNAL_MBM,
    WS_NL80211_BSS_SIGNAL_UNSPEC,
    WS_NL80211_BSS_STATUS,
    WS_NL80211_BSS_SEEN_MS_AGO,
    WS_NL80211_BSS_BEACON_IES,
    WS_NL80211_BSS_CHAN_WIDTH,
    WS_NL80211_BSS_BEACON_TSF,
    WS_NL80211_BSS_PRESP_DATA,
    WS_NL80211_BSS_LAST_SEEN_BOOTTIME,
    WS_NL80211_BSS_PAD,
    WS_NL80211_BSS_PARENT_TSF,
    WS_NL80211_BSS_PARENT_BSSID,
    WS_NL80211_BSS_CHAIN_SIGNAL,
    WS_NL80211_BSS_FREQUENCY_OFFSET,
    WS_NL80211_BSS_MLO_LINK_ID,
    WS_NL80211_BSS_MLD_ADDR,
    WS_NL80211_BSS_USE_FOR,
    WS_NL80211_BSS_CANNOT_USE_REASONS,
};

enum ws_nl80211_bss_status {
    WS_NL80211_BSS_STATUS_AUTHENTICATED,
    WS_NL80211_BSS_STATUS_ASSOCIATED,
    WS_NL80211_BSS_STATUS_IBSS_JOINED,
};

enum ws_nl80211_auth_type {
    WS_NL80211_AUTHTYPE_OPEN_SYSTEM,
    WS_NL80211_AUTHTYPE_SHARED_KEY,
    WS_NL80211_AUTHTYPE_FT,
    WS_NL80211_AUTHTYPE_NETWORK_EAP,
    WS_NL80211_AUTHTYPE_SAE,
    WS_NL80211_AUTHTYPE_FILS_SK,
    WS_NL80211_AUTHTYPE_FILS_SK_PFS,
    WS_NL80211_AUTHTYPE_FILS_PK,
};

enum ws_nl80211_key_type {
    WS_NL80211_KEYTYPE_GROUP,
    WS_NL80211_KEYTYPE_PAIRWISE,
    WS_NL80211_KEYTYPE_PEERKEY,
};

enum ws_nl80211_mfp {
    WS_NL80211_MFP_NO,
    WS_NL80211_MFP_REQUIRED,
    WS_NL80211_MFP_OPTIONAL,
};

enum ws_nl80211_wpa_versions {
    WS_NL80211_WPA_VERSION_1 = 1 << 0,
    WS_NL80211_WPA_VERSION_2 = 1 << 1,
    WS_NL80211_WPA_VERSION_3 = 1 << 2,
};

enum ws_nl80211_key_default_types {
    WS___NL80211_KEY_DEFAULT_TYPE_INVALID,
    WS_NL80211_KEY_DEFAULT_TYPE_UNICAST,
    WS_NL80211_KEY_DEFAULT_TYPE_MULTICAST,
};

enum ws_nl80211_key_attributes {
    WS___NL80211_KEY_INVALID,
    WS_NL80211_KEY_DATA,
    WS_NL80211_KEY_IDX,
    WS_NL80211_KEY_CIPHER,
    WS_NL80211_KEY_SEQ,
    WS_NL80211_KEY_DEFAULT,
    WS_NL80211_KEY_DEFAULT_MGMT,
    WS_NL80211_KEY_TYPE,
    WS_NL80211_KEY_DEFAULT_TYPES,
    WS_NL80211_KEY_MODE,
    WS_NL80211_KEY_DEFAULT_BEACON,
};

enum ws_nl80211_tx_rate_attributes {
    WS___NL80211_TXRATE_INVALID,
    WS_NL80211_TXRATE_LEGACY,
    WS_NL80211_TXRATE_HT,
    WS_NL80211_TXRATE_VHT,
    WS_NL80211_TXRATE_GI,
    WS_NL80211_TXRATE_HE,
    WS_NL80211_TXRATE_HE_GI,
    WS_NL80211_TXRATE_HE_LTF,
    WS_NL80211_TXRATE_EHT,
    WS_NL80211_TXRATE_EHT_GI,
    WS_NL80211_TXRATE_EHT_LTF,
};

enum ws_nl80211_txrate_gi {
    WS_NL80211_TXRATE_DEFAULT_GI,
    WS_NL80211_TXRATE_FORCE_SGI,
    WS_NL80211_TXRATE_FORCE_LGI,
};

enum ws_nl80211_band {
    WS_NL80211_BAND_2GHZ,
    WS_NL80211_BAND_5GHZ,
    WS_NL80211_BAND_60GHZ,
    WS_NL80211_BAND_6GHZ,
    WS_NL80211_BAND_S1GHZ,
    WS_NL80211_BAND_LC,
};

enum ws_nl80211_ps_state {
    WS_NL80211_PS_DISABLED,
    WS_NL80211_PS_ENABLED,
};

enum ws_nl80211_attr_cqm {
    WS___NL80211_ATTR_CQM_INVALID,
    WS_NL80211_ATTR_CQM_RSSI_THOLD,
    WS_NL80211_ATTR_CQM_RSSI_HYST,
    WS_NL80211_ATTR_CQM_RSSI_THRESHOLD_EVENT,
    WS_NL80211_ATTR_CQM_PKT_LOSS_EVENT,
    WS_NL80211_ATTR_CQM_TXE_RATE,
    WS_NL80211_ATTR_CQM_TXE_PKTS,
    WS_NL80211_ATTR_CQM_TXE_INTVL,
    WS_NL80211_ATTR_CQM_BEACON_LOSS_EVENT,
    WS_NL80211_ATTR_CQM_RSSI_LEVEL,
};

enum ws_nl80211_cqm_rssi_threshold_event {
    WS_NL80211_CQM_RSSI_THRESHOLD_EVENT_LOW,
    WS_NL80211_CQM_RSSI_THRESHOLD_EVENT_HIGH,
    WS_NL80211_CQM_RSSI_BEACON_LOSS_EVENT,
};

enum ws_nl80211_tx_power_setting {
    WS_NL80211_TX_POWER_AUTOMATIC,
    WS_NL80211_TX_POWER_LIMITED,
    WS_NL80211_TX_POWER_FIXED,
};

enum ws_nl80211_tid_config {
    WS_NL80211_TID_CONFIG_ENABLE,
    WS_NL80211_TID_CONFIG_DISABLE,
};

enum ws_nl80211_tx_rate_setting {
    WS_NL80211_TX_RATE_AUTOMATIC,
    WS_NL80211_TX_RATE_LIMITED,
    WS_NL80211_TX_RATE_FIXED,
};

enum ws_nl80211_tid_config_attr {
    WS___NL80211_TID_CONFIG_ATTR_INVALID,
    WS_NL80211_TID_CONFIG_ATTR_PAD,
    WS_NL80211_TID_CONFIG_ATTR_VIF_SUPP,
    WS_NL80211_TID_CONFIG_ATTR_PEER_SUPP,
    WS_NL80211_TID_CONFIG_ATTR_OVERRIDE,
    WS_NL80211_TID_CONFIG_ATTR_TIDS,
    WS_NL80211_TID_CONFIG_ATTR_NOACK,
    WS_NL80211_TID_CONFIG_ATTR_RETRY_SHORT,
    WS_NL80211_TID_CONFIG_ATTR_RETRY_LONG,
    WS_NL80211_TID_CONFIG_ATTR_AMPDU_CTRL,
    WS_NL80211_TID_CONFIG_ATTR_RTSCTS_CTRL,
    WS_NL80211_TID_CONFIG_ATTR_AMSDU_CTRL,
    WS_NL80211_TID_CONFIG_ATTR_TX_RATE_TYPE,
    WS_NL80211_TID_CONFIG_ATTR_TX_RATE,
};

enum ws_nl80211_packet_pattern_attr {
    WS___NL80211_PKTPAT_INVALID,
    WS_NL80211_PKTPAT_MASK,
    WS_NL80211_PKTPAT_PATTERN,
    WS_NL80211_PKTPAT_OFFSET,
};

enum ws_nl80211_wowlan_triggers {
    WS___NL80211_WOWLAN_TRIG_INVALID,
    WS_NL80211_WOWLAN_TRIG_ANY,
    WS_NL80211_WOWLAN_TRIG_DISCONNECT,
    WS_NL80211_WOWLAN_TRIG_MAGIC_PKT,
    WS_NL80211_WOWLAN_TRIG_PKT_PATTERN,
    WS_NL80211_WOWLAN_TRIG_GTK_REKEY_SUPPORTED,
    WS_NL80211_WOWLAN_TRIG_GTK_REKEY_FAILURE,
    WS_NL80211_WOWLAN_TRIG_EAP_IDENT_REQUEST,
    WS_NL80211_WOWLAN_TRIG_4WAY_HANDSHAKE,
    WS_NL80211_WOWLAN_TRIG_RFKILL_RELEASE,
    WS_NL80211_WOWLAN_TRIG_WAKEUP_PKT_80211,
    WS_NL80211_WOWLAN_TRIG_WAKEUP_PKT_80211_LEN,
    WS_NL80211_WOWLAN_TRIG_WAKEUP_PKT_8023,
    WS_NL80211_WOWLAN_TRIG_WAKEUP_PKT_8023_LEN,
    WS_NL80211_WOWLAN_TRIG_TCP_CONNECTION,
    WS_NL80211_WOWLAN_TRIG_WAKEUP_TCP_MATCH,
    WS_NL80211_WOWLAN_TRIG_WAKEUP_TCP_CONNLOST,
    WS_NL80211_WOWLAN_TRIG_WAKEUP_TCP_NOMORETOKENS,
    WS_NL80211_WOWLAN_TRIG_NET_DETECT,
    WS_NL80211_WOWLAN_TRIG_NET_DETECT_RESULTS,
    WS_NL80211_WOWLAN_TRIG_UNPROTECTED_DEAUTH_DISASSOC,
};

enum ws_nl80211_wowlan_tcp_attrs {
    WS___NL80211_WOWLAN_TCP_INVALID,
    WS_NL80211_WOWLAN_TCP_SRC_IPV4,
    WS_NL80211_WOWLAN_TCP_DST_IPV4,
    WS_NL80211_WOWLAN_TCP_DST_MAC,
    WS_NL80211_WOWLAN_TCP_SRC_PORT,
    WS_NL80211_WOWLAN_TCP_DST_PORT,
    WS_NL80211_WOWLAN_TCP_DATA_PAYLOAD,
    WS_NL80211_WOWLAN_TCP_DATA_PAYLOAD_SEQ,
    WS_NL80211_WOWLAN_TCP_DATA_PAYLOAD_TOKEN,
    WS_NL80211_WOWLAN_TCP_DATA_INTERVAL,
    WS_NL80211_WOWLAN_TCP_WAKE_PAYLOAD,
    WS_NL80211_WOWLAN_TCP_WAKE_MASK,
};

enum ws_nl80211_attr_coalesce_rule {
    WS___NL80211_COALESCE_RULE_INVALID,
    WS_NL80211_ATTR_COALESCE_RULE_DELAY,
    WS_NL80211_ATTR_COALESCE_RULE_CONDITION,
    WS_NL80211_ATTR_COALESCE_RULE_PKT_PATTERN,
};

enum ws_nl80211_coalesce_condition {
    WS_NL80211_COALESCE_CONDITION_MATCH,
    WS_NL80211_COALESCE_CONDITION_NO_MATCH,
};

enum ws_nl80211_iface_limit_attrs {
    WS_NL80211_IFACE_LIMIT_UNSPEC,
    WS_NL80211_IFACE_LIMIT_MAX,
    WS_NL80211_IFACE_LIMIT_TYPES,
};

enum ws_nl80211_if_combination_attrs {
    WS_NL80211_IFACE_COMB_UNSPEC,
    WS_NL80211_IFACE_COMB_LIMITS,
    WS_NL80211_IFACE_COMB_MAXNUM,
    WS_NL80211_IFACE_COMB_STA_AP_BI_MATCH,
    WS_NL80211_IFACE_COMB_NUM_CHANNELS,
    WS_NL80211_IFACE_COMB_RADAR_DETECT_WIDTHS,
    WS_NL80211_IFACE_COMB_RADAR_DETECT_REGIONS,
    WS_NL80211_IFACE_COMB_BI_MIN_GCD,
};

enum ws_nl80211_plink_state {
    WS_NL80211_PLINK_LISTEN,
    WS_NL80211_PLINK_OPN_SNT,
    WS_NL80211_PLINK_OPN_RCVD,
    WS_NL80211_PLINK_CNF_RCVD,
    WS_NL80211_PLINK_ESTAB,
    WS_NL80211_PLINK_HOLDING,
    WS_NL80211_PLINK_BLOCKED,
};

enum ws_nl80211_plink_action {
    WS_NL80211_PLINK_ACTION_NO_ACTION,
    WS_NL80211_PLINK_ACTION_OPEN,
    WS_NL80211_PLINK_ACTION_BLOCK,
};

enum ws_nl80211_rekey_data {
    WS___NL80211_REKEY_DATA_INVALID,
    WS_NL80211_REKEY_DATA_KEK,
    WS_NL80211_REKEY_DATA_KCK,
    WS_NL80211_REKEY_DATA_REPLAY_CTR,
    WS_NL80211_REKEY_DATA_AKM,
};

enum ws_nl80211_hidden_ssid {
    WS_NL80211_HIDDEN_SSID_NOT_IN_USE,
    WS_NL80211_HIDDEN_SSID_ZERO_LEN,
    WS_NL80211_HIDDEN_SSID_ZERO_CONTENTS,
};

enum ws_nl80211_sta_wme_attr {
    WS___NL80211_STA_WME_INVALID,
    WS_NL80211_STA_WME_UAPSD_QUEUES,
    WS_NL80211_STA_WME_MAX_SP,
};

enum ws_nl80211_pmksa_candidate_attr {
    WS___NL80211_PMKSA_CANDIDATE_INVALID,
    WS_NL80211_PMKSA_CANDIDATE_INDEX,
    WS_NL80211_PMKSA_CANDIDATE_BSSID,
    WS_NL80211_PMKSA_CANDIDATE_PREAUTH,
};

enum ws_nl80211_tdls_operation {
    WS_NL80211_TDLS_DISCOVERY_REQ,
    WS_NL80211_TDLS_SETUP,
    WS_NL80211_TDLS_TEARDOWN,
    WS_NL80211_TDLS_ENABLE_LINK,
    WS_NL80211_TDLS_DISABLE_LINK,
};

enum ws_nl80211_ap_sme_features {
    WS_NL80211_AP_SME_SA_QUERY_OFFLOAD = 1 << 0,
};

enum ws_nl80211_feature_flags {
    WS_NL80211_FEATURE_SK_TX_STATUS = 1 << 0,
    WS_NL80211_FEATURE_HT_IBSS = 1 << 1,
    WS_NL80211_FEATURE_INACTIVITY_TIMER = 1 << 2,
    WS_NL80211_FEATURE_CELL_BASE_REG_HINTS = 1 << 3,
    WS_NL80211_FEATURE_P2P_DEVICE_NEEDS_CHANNEL = 1 << 4,
    WS_NL80211_FEATURE_SAE = 1 << 5,
    WS_NL80211_FEATURE_LOW_PRIORITY_SCAN = 1 << 6,
    WS_NL80211_FEATURE_SCAN_FLUSH = 1 << 7,
    WS_NL80211_FEATURE_AP_SCAN = 1 << 8,
    WS_NL80211_FEATURE_VIF_TXPOWER = 1 << 9,
    WS_NL80211_FEATURE_NEED_OBSS_SCAN = 1 << 10,
    WS_NL80211_FEATURE_P2P_GO_CTWIN = 1 << 11,
    WS_NL80211_FEATURE_P2P_GO_OPPPS = 1 << 12,
    WS_NL80211_FEATURE_ADVERTISE_CHAN_LIMITS = 1 << 14,
    WS_NL80211_FEATURE_FULL_AP_CLIENT_STATE = 1 << 15,
    WS_NL80211_FEATURE_USERSPACE_MPM = 1 << 16,
    WS_NL80211_FEATURE_ACTIVE_MONITOR = 1 << 17,
    WS_NL80211_FEATURE_AP_MODE_CHAN_WIDTH_CHANGE = 1 << 18,
    WS_NL80211_FEATURE_DS_PARAM_SET_IE_IN_PROBES = 1 << 19,
    WS_NL80211_FEATURE_WFA_TPC_IE_IN_PROBES = 1 << 20,
    WS_NL80211_FEATURE_QUIET = 1 << 21,
    WS_NL80211_FEATURE_TX_POWER_INSERTION = 1 << 22,
    WS_NL80211_FEATURE_ACKTO_ESTIMATION = 1 << 23,
    WS_NL80211_FEATURE_STATIC_SMPS = 1 << 24,
    WS_NL80211_FEATURE_DYNAMIC_SMPS = 1 << 25,
    WS_NL80211_FEATURE_SUPPORTS_WMM_ADMISSION = 1 << 26,
    WS_NL80211_FEATURE_MAC_ON_CREATE = 1 << 27,
    WS_NL80211_FEATURE_TDLS_CHANNEL_SWITCH = 1 << 28,
    WS_NL80211_FEATURE_SCAN_RANDOM_MAC_ADDR = 1 << 29,
    WS_NL80211_FEATURE_SCHED_SCAN_RANDOM_MAC_ADDR = 1 << 30,
    WS_NL80211_FEATURE_ND_RANDOM_MAC_ADDR = 1U << 31,
};

enum ws_nl80211_ext_feature_index {
    WS_NL80211_EXT_FEATURE_VHT_IBSS,
    WS_NL80211_EXT_FEATURE_RRM,
    WS_NL80211_EXT_FEATURE_MU_MIMO_AIR_SNIFFER,
    WS_NL80211_EXT_FEATURE_SCAN_START_TIME,
    WS_NL80211_EXT_FEATURE_BSS_PARENT_TSF,
    WS_NL80211_EXT_FEATURE_SET_SCAN_DWELL,
    WS_NL80211_EXT_FEATURE_BEACON_RATE_LEGACY,
    WS_NL80211_EXT_FEATURE_BEACON_RATE_HT,
    WS_NL80211_EXT_FEATURE_BEACON_RATE_VHT,
    WS_NL80211_EXT_FEATURE_FILS_STA,
    WS_NL80211_EXT_FEATURE_MGMT_TX_RANDOM_TA,
    WS_NL80211_EXT_FEATURE_MGMT_TX_RANDOM_TA_CONNECTED,
    WS_NL80211_EXT_FEATURE_SCHED_SCAN_RELATIVE_RSSI,
    WS_NL80211_EXT_FEATURE_CQM_RSSI_LIST,
    WS_NL80211_EXT_FEATURE_FILS_SK_OFFLOAD,
    WS_NL80211_EXT_FEATURE_4WAY_HANDSHAKE_STA_PSK,
    WS_NL80211_EXT_FEATURE_4WAY_HANDSHAKE_STA_1X,
    WS_NL80211_EXT_FEATURE_FILS_MAX_CHANNEL_TIME,
    WS_NL80211_EXT_FEATURE_ACCEPT_BCAST_PROBE_RESP,
    WS_NL80211_EXT_FEATURE_OCE_PROBE_REQ_HIGH_TX_RATE,
    WS_NL80211_EXT_FEATURE_OCE_PROBE_REQ_DEFERRAL_SUPPRESSION,
    WS_NL80211_EXT_FEATURE_MFP_OPTIONAL,
    WS_NL80211_EXT_FEATURE_LOW_SPAN_SCAN,
    WS_NL80211_EXT_FEATURE_LOW_POWER_SCAN,
    WS_NL80211_EXT_FEATURE_HIGH_ACCURACY_SCAN,
    WS_NL80211_EXT_FEATURE_DFS_OFFLOAD,
    WS_NL80211_EXT_FEATURE_CONTROL_PORT_OVER_NL80211,
    WS_NL80211_EXT_FEATURE_ACK_SIGNAL_SUPPORT,
    WS_NL80211_EXT_FEATURE_TXQS,
    WS_NL80211_EXT_FEATURE_SCAN_RANDOM_SN,
    WS_NL80211_EXT_FEATURE_SCAN_MIN_PREQ_CONTENT,
    WS_NL80211_EXT_FEATURE_CAN_REPLACE_PTK0,
    WS_NL80211_EXT_FEATURE_ENABLE_FTM_RESPONDER,
    WS_NL80211_EXT_FEATURE_AIRTIME_FAIRNESS,
    WS_NL80211_EXT_FEATURE_AP_PMKSA_CACHING,
    WS_NL80211_EXT_FEATURE_SCHED_SCAN_BAND_SPECIFIC_RSSI_THOLD,
    WS_NL80211_EXT_FEATURE_EXT_KEY_ID,
    WS_NL80211_EXT_FEATURE_STA_TX_PWR,
    WS_NL80211_EXT_FEATURE_SAE_OFFLOAD,
    WS_NL80211_EXT_FEATURE_VLAN_OFFLOAD,
    WS_NL80211_EXT_FEATURE_AQL,
    WS_NL80211_EXT_FEATURE_BEACON_PROTECTION,
    WS_NL80211_EXT_FEATURE_CONTROL_PORT_NO_PREAUTH,
    WS_NL80211_EXT_FEATURE_PROTECTED_TWT,
    WS_NL80211_EXT_FEATURE_DEL_IBSS_STA,
    WS_NL80211_EXT_FEATURE_MULTICAST_REGISTRATIONS,
    WS_NL80211_EXT_FEATURE_BEACON_PROTECTION_CLIENT,
    WS_NL80211_EXT_FEATURE_SCAN_FREQ_KHZ,
    WS_NL80211_EXT_FEATURE_CONTROL_PORT_OVER_NL80211_TX_STATUS,
    WS_NL80211_EXT_FEATURE_OPERATING_CHANNEL_VALIDATION,
    WS_NL80211_EXT_FEATURE_4WAY_HANDSHAKE_AP_PSK,
    WS_NL80211_EXT_FEATURE_SAE_OFFLOAD_AP,
    WS_NL80211_EXT_FEATURE_FILS_DISCOVERY,
    WS_NL80211_EXT_FEATURE_UNSOL_BCAST_PROBE_RESP,
    WS_NL80211_EXT_FEATURE_BEACON_RATE_HE,
    WS_NL80211_EXT_FEATURE_SECURE_LTF,
    WS_NL80211_EXT_FEATURE_SECURE_RTT,
    WS_NL80211_EXT_FEATURE_PROT_RANGE_NEGO_AND_MEASURE,
    WS_NL80211_EXT_FEATURE_BSS_COLOR,
    WS_NL80211_EXT_FEATURE_FILS_CRYPTO_OFFLOAD,
    WS_NL80211_EXT_FEATURE_RADAR_BACKGROUND,
    WS_NL80211_EXT_FEATURE_POWERED_ADDR_CHANGE,
    WS_NL80211_EXT_FEATURE_PUNCT,
    WS_NL80211_EXT_FEATURE_SECURE_NAN,
    WS_NL80211_EXT_FEATURE_AUTH_AND_DEAUTH_RANDOM_TA,
    WS_NL80211_EXT_FEATURE_OWE_OFFLOAD,
    WS_NL80211_EXT_FEATURE_OWE_OFFLOAD_AP,
    WS_NL80211_EXT_FEATURE_DFS_CONCURRENT,
    WS_NL80211_EXT_FEATURE_SPP_AMSDU_SUPPORT,
    WS_NL80211_EXT_FEATURE_BEACON_RATE_EHT,
};

enum ws_nl80211_probe_resp_offload_support_attr {
    WS_NL80211_PROBE_RESP_OFFLOAD_SUPPORT_WPS = 1<<0,
    WS_NL80211_PROBE_RESP_OFFLOAD_SUPPORT_WPS2 = 1<<1,
    WS_NL80211_PROBE_RESP_OFFLOAD_SUPPORT_P2P = 1<<2,
    WS_NL80211_PROBE_RESP_OFFLOAD_SUPPORT_80211U = 1<<3,
};

enum ws_nl80211_connect_failed_reason {
    WS_NL80211_CONN_FAIL_MAX_CLIENTS,
    WS_NL80211_CONN_FAIL_BLOCKED_CLIENT,
};

enum ws_nl80211_timeout_reason {
    WS_NL80211_TIMEOUT_UNSPECIFIED,
    WS_NL80211_TIMEOUT_SCAN,
    WS_NL80211_TIMEOUT_AUTH,
    WS_NL80211_TIMEOUT_ASSOC,
};

enum ws_nl80211_scan_flags {
    WS_NL80211_SCAN_FLAG_LOW_PRIORITY = 1<<0,
    WS_NL80211_SCAN_FLAG_FLUSH = 1<<1,
    WS_NL80211_SCAN_FLAG_AP = 1<<2,
    WS_NL80211_SCAN_FLAG_RANDOM_ADDR = 1<<3,
    WS_NL80211_SCAN_FLAG_FILS_MAX_CHANNEL_TIME = 1<<4,
    WS_NL80211_SCAN_FLAG_ACCEPT_BCAST_PROBE_RESP = 1<<5,
    WS_NL80211_SCAN_FLAG_OCE_PROBE_REQ_HIGH_TX_RATE = 1<<6,
    WS_NL80211_SCAN_FLAG_OCE_PROBE_REQ_DEFERRAL_SUPPRESSION = 1<<7,
    WS_NL80211_SCAN_FLAG_LOW_SPAN = 1<<8,
    WS_NL80211_SCAN_FLAG_LOW_POWER = 1<<9,
    WS_NL80211_SCAN_FLAG_HIGH_ACCURACY = 1<<10,
    WS_NL80211_SCAN_FLAG_RANDOM_SN = 1<<11,
    WS_NL80211_SCAN_FLAG_MIN_PREQ_CONTENT = 1<<12,
    WS_NL80211_SCAN_FLAG_FREQ_KHZ = 1<<13,
    WS_NL80211_SCAN_FLAG_COLOCATED_6GHZ = 1<<14,
};

enum ws_nl80211_acl_policy {
    WS_NL80211_ACL_POLICY_ACCEPT_UNLESS_LISTED,
    WS_NL80211_ACL_POLICY_DENY_UNLESS_LISTED,
};

enum ws_nl80211_smps_mode {
    WS_NL80211_SMPS_OFF,
    WS_NL80211_SMPS_STATIC,
    WS_NL80211_SMPS_DYNAMIC,
};

enum ws_nl80211_radar_event {
    WS_NL80211_RADAR_DETECTED,
    WS_NL80211_RADAR_CAC_FINISHED,
    WS_NL80211_RADAR_CAC_ABORTED,
    WS_NL80211_RADAR_NOP_FINISHED,
    WS_NL80211_RADAR_PRE_CAC_EXPIRED,
    WS_NL80211_RADAR_CAC_STARTED,
};

enum ws_nl80211_dfs_state {
    WS_NL80211_DFS_USABLE,
    WS_NL80211_DFS_UNAVAILABLE,
    WS_NL80211_DFS_AVAILABLE,
};

enum ws_nl80211_protocol_features {
    WS_NL80211_PROTOCOL_FEATURE_SPLIT_WIPHY_DUMP = 1 << 0,
};

enum ws_nl80211_crit_proto_id {
    WS_NL80211_CRIT_PROTO_UNSPEC,
    WS_NL80211_CRIT_PROTO_DHCP,
    WS_NL80211_CRIT_PROTO_EAPOL,
    WS_NL80211_CRIT_PROTO_APIPA,
};

enum ws_nl80211_rxmgmt_flags {
    WS_NL80211_RXMGMT_FLAG_ANSWERED = 1 << 0,
    WS_NL80211_RXMGMT_FLAG_EXTERNAL_AUTH = 1 << 1,
};

enum ws_nl80211_tdls_peer_capability {
    WS_NL80211_TDLS_PEER_HT = 1<<0,
    WS_NL80211_TDLS_PEER_VHT = 1<<1,
    WS_NL80211_TDLS_PEER_WMM = 1<<2,
    WS_NL80211_TDLS_PEER_HE = 1<<3,
};

enum ws_nl80211_sched_scan_plan {
    WS___NL80211_SCHED_SCAN_PLAN_INVALID,
    WS_NL80211_SCHED_SCAN_PLAN_INTERVAL,
    WS_NL80211_SCHED_SCAN_PLAN_ITERATIONS,
};

enum ws_nl80211_bss_select_attr {
    WS___NL80211_BSS_SELECT_ATTR_INVALID,
    WS_NL80211_BSS_SELECT_ATTR_RSSI,
    WS_NL80211_BSS_SELECT_ATTR_BAND_PREF,
    WS_NL80211_BSS_SELECT_ATTR_RSSI_ADJUST,
};

enum ws_nl80211_nan_function_type {
    WS_NL80211_NAN_FUNC_PUBLISH,
    WS_NL80211_NAN_FUNC_SUBSCRIBE,
    WS_NL80211_NAN_FUNC_FOLLOW_UP,
};

enum ws_nl80211_nan_publish_type {
    WS_NL80211_NAN_SOLICITED_PUBLISH = 1 << 0,
    WS_NL80211_NAN_UNSOLICITED_PUBLISH = 1 << 1,
};

enum ws_nl80211_nan_func_term_reason {
    WS_NL80211_NAN_FUNC_TERM_REASON_USER_REQUEST,
    WS_NL80211_NAN_FUNC_TERM_REASON_TTL_EXPIRED,
    WS_NL80211_NAN_FUNC_TERM_REASON_ERROR,
};

enum ws_nl80211_nan_func_attributes {
    WS___NL80211_NAN_FUNC_INVALID,
    WS_NL80211_NAN_FUNC_TYPE,
    WS_NL80211_NAN_FUNC_SERVICE_ID,
    WS_NL80211_NAN_FUNC_PUBLISH_TYPE,
    WS_NL80211_NAN_FUNC_PUBLISH_BCAST,
    WS_NL80211_NAN_FUNC_SUBSCRIBE_ACTIVE,
    WS_NL80211_NAN_FUNC_FOLLOW_UP_ID,
    WS_NL80211_NAN_FUNC_FOLLOW_UP_REQ_ID,
    WS_NL80211_NAN_FUNC_FOLLOW_UP_DEST,
    WS_NL80211_NAN_FUNC_CLOSE_RANGE,
    WS_NL80211_NAN_FUNC_TTL,
    WS_NL80211_NAN_FUNC_SERVICE_INFO,
    WS_NL80211_NAN_FUNC_SRF,
    WS_NL80211_NAN_FUNC_RX_MATCH_FILTER,
    WS_NL80211_NAN_FUNC_TX_MATCH_FILTER,
    WS_NL80211_NAN_FUNC_INSTANCE_ID,
    WS_NL80211_NAN_FUNC_TERM_REASON,
};

enum ws_nl80211_nan_srf_attributes {
    WS___NL80211_NAN_SRF_INVALID,
    WS_NL80211_NAN_SRF_INCLUDE,
    WS_NL80211_NAN_SRF_BF,
    WS_NL80211_NAN_SRF_BF_IDX,
    WS_NL80211_NAN_SRF_MAC_ADDRS,
};

enum ws_nl80211_nan_match_attributes {
    WS___NL80211_NAN_MATCH_INVALID,
    WS_NL80211_NAN_MATCH_FUNC_LOCAL,
    WS_NL80211_NAN_MATCH_FUNC_PEER,
};

enum ws_nl80211_external_auth_action {
    WS_NL80211_EXTERNAL_AUTH_START,
    WS_NL80211_EXTERNAL_AUTH_ABORT,
};

enum ws_nl80211_ftm_responder_attributes {
    WS___NL80211_FTM_RESP_ATTR_INVALID,
    WS_NL80211_FTM_RESP_ATTR_ENABLED,
    WS_NL80211_FTM_RESP_ATTR_LCI,
    WS_NL80211_FTM_RESP_ATTR_CIVICLOC,
};

enum ws_nl80211_ftm_responder_stats {
    WS___NL80211_FTM_STATS_INVALID,
    WS_NL80211_FTM_STATS_SUCCESS_NUM,
    WS_NL80211_FTM_STATS_PARTIAL_NUM,
    WS_NL80211_FTM_STATS_FAILED_NUM,
    WS_NL80211_FTM_STATS_ASAP_NUM,
    WS_NL80211_FTM_STATS_NON_ASAP_NUM,
    WS_NL80211_FTM_STATS_TOTAL_DURATION_MSEC,
    WS_NL80211_FTM_STATS_UNKNOWN_TRIGGERS_NUM,
    WS_NL80211_FTM_STATS_RESCHEDULE_REQUESTS_NUM,
    WS_NL80211_FTM_STATS_OUT_OF_WINDOW_TRIGGERS_NUM,
    WS_NL80211_FTM_STATS_PAD,
};

enum ws_nl80211_preamble {
    WS_NL80211_PREAMBLE_LEGACY,
    WS_NL80211_PREAMBLE_HT,
    WS_NL80211_PREAMBLE_VHT,
    WS_NL80211_PREAMBLE_DMG,
    WS_NL80211_PREAMBLE_HE,
};

enum ws_nl80211_peer_measurement_type {
    WS_NL80211_PMSR_TYPE_INVALID,
    WS_NL80211_PMSR_TYPE_FTM,
};

enum ws_nl80211_peer_measurement_status {
    WS_NL80211_PMSR_STATUS_SUCCESS,
    WS_NL80211_PMSR_STATUS_REFUSED,
    WS_NL80211_PMSR_STATUS_TIMEOUT,
    WS_NL80211_PMSR_STATUS_FAILURE,
};

enum ws_nl80211_peer_measurement_req {
    WS___NL80211_PMSR_REQ_ATTR_INVALID,
    WS_NL80211_PMSR_REQ_ATTR_DATA,
    WS_NL80211_PMSR_REQ_ATTR_GET_AP_TSF,
};

enum ws_nl80211_peer_measurement_resp {
    WS___NL80211_PMSR_RESP_ATTR_INVALID,
    WS_NL80211_PMSR_RESP_ATTR_DATA,
    WS_NL80211_PMSR_RESP_ATTR_STATUS,
    WS_NL80211_PMSR_RESP_ATTR_HOST_TIME,
    WS_NL80211_PMSR_RESP_ATTR_AP_TSF,
    WS_NL80211_PMSR_RESP_ATTR_FINAL,
    WS_NL80211_PMSR_RESP_ATTR_PAD,
};

enum ws_nl80211_peer_measurement_peer_attrs {
    WS___NL80211_PMSR_PEER_ATTR_INVALID,
    WS_NL80211_PMSR_PEER_ATTR_ADDR,
    WS_NL80211_PMSR_PEER_ATTR_CHAN,
    WS_NL80211_PMSR_PEER_ATTR_REQ,
    WS_NL80211_PMSR_PEER_ATTR_RESP,
};

enum ws_nl80211_peer_measurement_attrs {
    WS___NL80211_PMSR_ATTR_INVALID,
    WS_NL80211_PMSR_ATTR_MAX_PEERS,
    WS_NL80211_PMSR_ATTR_REPORT_AP_TSF,
    WS_NL80211_PMSR_ATTR_RANDOMIZE_MAC_ADDR,
    WS_NL80211_PMSR_ATTR_TYPE_CAPA,
    WS_NL80211_PMSR_ATTR_PEERS,
};

enum ws_nl80211_peer_measurement_ftm_capa {
    WS___NL80211_PMSR_FTM_CAPA_ATTR_INVALID,
    WS_NL80211_PMSR_FTM_CAPA_ATTR_ASAP,
    WS_NL80211_PMSR_FTM_CAPA_ATTR_NON_ASAP,
    WS_NL80211_PMSR_FTM_CAPA_ATTR_REQ_LCI,
    WS_NL80211_PMSR_FTM_CAPA_ATTR_REQ_CIVICLOC,
    WS_NL80211_PMSR_FTM_CAPA_ATTR_PREAMBLES,
    WS_NL80211_PMSR_FTM_CAPA_ATTR_BANDWIDTHS,
    WS_NL80211_PMSR_FTM_CAPA_ATTR_MAX_BURSTS_EXPONENT,
    WS_NL80211_PMSR_FTM_CAPA_ATTR_MAX_FTMS_PER_BURST,
    WS_NL80211_PMSR_FTM_CAPA_ATTR_TRIGGER_BASED,
    WS_NL80211_PMSR_FTM_CAPA_ATTR_NON_TRIGGER_BASED,
};

enum ws_nl80211_peer_measurement_ftm_req {
    WS___NL80211_PMSR_FTM_REQ_ATTR_INVALID,
    WS_NL80211_PMSR_FTM_REQ_ATTR_ASAP,
    WS_NL80211_PMSR_FTM_REQ_ATTR_PREAMBLE,
    WS_NL80211_PMSR_FTM_REQ_ATTR_NUM_BURSTS_EXP,
    WS_NL80211_PMSR_FTM_REQ_ATTR_BURST_PERIOD,
    WS_NL80211_PMSR_FTM_REQ_ATTR_BURST_DURATION,
    WS_NL80211_PMSR_FTM_REQ_ATTR_FTMS_PER_BURST,
    WS_NL80211_PMSR_FTM_REQ_ATTR_NUM_FTMR_RETRIES,
    WS_NL80211_PMSR_FTM_REQ_ATTR_REQUEST_LCI,
    WS_NL80211_PMSR_FTM_REQ_ATTR_REQUEST_CIVICLOC,
    WS_NL80211_PMSR_FTM_REQ_ATTR_TRIGGER_BASED,
    WS_NL80211_PMSR_FTM_REQ_ATTR_NON_TRIGGER_BASED,
    WS_NL80211_PMSR_FTM_REQ_ATTR_LMR_FEEDBACK,
    WS_NL80211_PMSR_FTM_REQ_ATTR_BSS_COLOR,
};

enum ws_nl80211_peer_measurement_ftm_failure_reasons {
    WS_NL80211_PMSR_FTM_FAILURE_UNSPECIFIED,
    WS_NL80211_PMSR_FTM_FAILURE_NO_RESPONSE,
    WS_NL80211_PMSR_FTM_FAILURE_REJECTED,
    WS_NL80211_PMSR_FTM_FAILURE_WRONG_CHANNEL,
    WS_NL80211_PMSR_FTM_FAILURE_PEER_NOT_CAPABLE,
    WS_NL80211_PMSR_FTM_FAILURE_INVALID_TIMESTAMP,
    WS_NL80211_PMSR_FTM_FAILURE_PEER_BUSY,
    WS_NL80211_PMSR_FTM_FAILURE_BAD_CHANGED_PARAMS,
};

enum ws_nl80211_peer_measurement_ftm_resp {
    WS___NL80211_PMSR_FTM_RESP_ATTR_INVALID,
    WS_NL80211_PMSR_FTM_RESP_ATTR_FAIL_REASON,
    WS_NL80211_PMSR_FTM_RESP_ATTR_BURST_INDEX,
    WS_NL80211_PMSR_FTM_RESP_ATTR_NUM_FTMR_ATTEMPTS,
    WS_NL80211_PMSR_FTM_RESP_ATTR_NUM_FTMR_SUCCESSES,
    WS_NL80211_PMSR_FTM_RESP_ATTR_BUSY_RETRY_TIME,
    WS_NL80211_PMSR_FTM_RESP_ATTR_NUM_BURSTS_EXP,
    WS_NL80211_PMSR_FTM_RESP_ATTR_BURST_DURATION,
    WS_NL80211_PMSR_FTM_RESP_ATTR_FTMS_PER_BURST,
    WS_NL80211_PMSR_FTM_RESP_ATTR_RSSI_AVG,
    WS_NL80211_PMSR_FTM_RESP_ATTR_RSSI_SPREAD,
    WS_NL80211_PMSR_FTM_RESP_ATTR_TX_RATE,
    WS_NL80211_PMSR_FTM_RESP_ATTR_RX_RATE,
    WS_NL80211_PMSR_FTM_RESP_ATTR_RTT_AVG,
    WS_NL80211_PMSR_FTM_RESP_ATTR_RTT_VARIANCE,
    WS_NL80211_PMSR_FTM_RESP_ATTR_RTT_SPREAD,
    WS_NL80211_PMSR_FTM_RESP_ATTR_DIST_AVG,
    WS_NL80211_PMSR_FTM_RESP_ATTR_DIST_VARIANCE,
    WS_NL80211_PMSR_FTM_RESP_ATTR_DIST_SPREAD,
    WS_NL80211_PMSR_FTM_RESP_ATTR_LCI,
    WS_NL80211_PMSR_FTM_RESP_ATTR_CIVICLOC,
    WS_NL80211_PMSR_FTM_RESP_ATTR_PAD,
};

enum ws_nl80211_obss_pd_attributes {
    WS___NL80211_HE_OBSS_PD_ATTR_INVALID,
    WS_NL80211_HE_OBSS_PD_ATTR_MIN_OFFSET,
    WS_NL80211_HE_OBSS_PD_ATTR_MAX_OFFSET,
    WS_NL80211_HE_OBSS_PD_ATTR_NON_SRG_MAX_OFFSET,
    WS_NL80211_HE_OBSS_PD_ATTR_BSS_COLOR_BITMAP,
    WS_NL80211_HE_OBSS_PD_ATTR_PARTIAL_BSSID_BITMAP,
    WS_NL80211_HE_OBSS_PD_ATTR_SR_CTRL,
};

enum ws_nl80211_bss_color_attributes {
    WS___NL80211_HE_BSS_COLOR_ATTR_INVALID,
    WS_NL80211_HE_BSS_COLOR_ATTR_COLOR,
    WS_NL80211_HE_BSS_COLOR_ATTR_DISABLED,
    WS_NL80211_HE_BSS_COLOR_ATTR_PARTIAL,
};

enum ws_nl80211_iftype_akm_attributes {
    WS___NL80211_IFTYPE_AKM_ATTR_INVALID,
    WS_NL80211_IFTYPE_AKM_ATTR_IFTYPES,
    WS_NL80211_IFTYPE_AKM_ATTR_SUITES,
};

enum ws_nl80211_fils_discovery_attributes {
    WS___NL80211_FILS_DISCOVERY_ATTR_INVALID,
    WS_NL80211_FILS_DISCOVERY_ATTR_INT_MIN,
    WS_NL80211_FILS_DISCOVERY_ATTR_INT_MAX,
    WS_NL80211_FILS_DISCOVERY_ATTR_TMPL,
};

enum ws_nl80211_unsol_bcast_probe_resp_attributes {
    WS___NL80211_UNSOL_BCAST_PROBE_RESP_ATTR_INVALID,
    WS_NL80211_UNSOL_BCAST_PROBE_RESP_ATTR_INT,
    WS_NL80211_UNSOL_BCAST_PROBE_RESP_ATTR_TMPL,
};

enum ws_nl80211_sae_pwe_mechanism {
    WS_NL80211_SAE_PWE_UNSPECIFIED,
    WS_NL80211_SAE_PWE_HUNT_AND_PECK,
    WS_NL80211_SAE_PWE_HASH_TO_ELEMENT,
    WS_NL80211_SAE_PWE_BOTH,
};

enum ws_nl80211_sar_type {
    WS_NL80211_SAR_TYPE_POWER,
};

enum ws_nl80211_sar_attrs {
    WS___NL80211_SAR_ATTR_INVALID,
    WS_NL80211_SAR_ATTR_TYPE,
    WS_NL80211_SAR_ATTR_SPECS,
};

enum ws_nl80211_sar_specs_attrs {
    WS___NL80211_SAR_ATTR_SPECS_INVALID,
    WS_NL80211_SAR_ATTR_SPECS_POWER,
    WS_NL80211_SAR_ATTR_SPECS_RANGE_INDEX,
    WS_NL80211_SAR_ATTR_SPECS_START_FREQ,
    WS_NL80211_SAR_ATTR_SPECS_END_FREQ,
};

enum ws_nl80211_mbssid_config_attributes {
    WS___NL80211_MBSSID_CONFIG_ATTR_INVALID,
    WS_NL80211_MBSSID_CONFIG_ATTR_MAX_INTERFACES,
    WS_NL80211_MBSSID_CONFIG_ATTR_MAX_EMA_PROFILE_PERIODICITY,
    WS_NL80211_MBSSID_CONFIG_ATTR_INDEX,
    WS_NL80211_MBSSID_CONFIG_ATTR_TX_IFINDEX,
    WS_NL80211_MBSSID_CONFIG_ATTR_EMA,
    WS_NL80211_MBSSID_CONFIG_ATTR_TX_LINK_ID,
};

enum ws_nl80211_ap_settings_flags {
    WS_NL80211_AP_SETTINGS_EXTERNAL_AUTH_SUPPORT = 1 << 0,
    WS_NL80211_AP_SETTINGS_SA_QUERY_OFFLOAD_SUPPORT = 1 << 1,
};

enum ws_nl80211_wiphy_radio_attrs {
    WS___NL80211_WIPHY_RADIO_ATTR_INVALID,
    WS_NL80211_WIPHY_RADIO_ATTR_INDEX,
    WS_NL80211_WIPHY_RADIO_ATTR_FREQ_RANGE,
    WS_NL80211_WIPHY_RADIO_ATTR_INTERFACE_COMBINATION,
    WS_NL80211_WIPHY_RADIO_ATTR_ANTENNA_MASK,
    WS_NL80211_WIPHY_RADIO_ATTR_RTS_THRESHOLD,
};

enum ws_nl80211_wiphy_radio_freq_range {
    WS___NL80211_WIPHY_RADIO_FREQ_ATTR_INVALID,
    WS_NL80211_WIPHY_RADIO_FREQ_ATTR_START,
    WS_NL80211_WIPHY_RADIO_FREQ_ATTR_END,
};

enum ws_nl80211_s1g_short_beacon_attrs {
    WS___NL80211_S1G_SHORT_BEACON_ATTR_INVALID,
    WS_NL80211_S1G_SHORT_BEACON_ATTR_HEAD,
    WS_NL80211_S1G_SHORT_BEACON_ATTR_TAIL,
};

enum ws_nl80211_nan_capabilities {
    WS___NL80211_NAN_CAPABILITIES_INVALID,
    WS_NL80211_NAN_CAPA_CONFIGURABLE_SYNC,
    WS_NL80211_NAN_CAPA_USERSPACE_DE,
    WS_NL80211_NAN_CAPA_OP_MODE,
    WS_NL80211_NAN_CAPA_NUM_ANTENNAS,
    WS_NL80211_NAN_CAPA_MAX_CHANNEL_SWITCH_TIME,
    WS_NL80211_NAN_CAPA_CAPABILITIES,
};

static const value_string ws_nl80211_commands_vals[] = {
    { WS_NL80211_CMD_UNSPEC,                "NL80211_CMD_UNSPEC" },
    { WS_NL80211_CMD_GET_WIPHY,             "NL80211_CMD_GET_WIPHY" },
    { WS_NL80211_CMD_SET_WIPHY,             "NL80211_CMD_SET_WIPHY" },
    { WS_NL80211_CMD_NEW_WIPHY,             "NL80211_CMD_NEW_WIPHY" },
    { WS_NL80211_CMD_DEL_WIPHY,             "NL80211_CMD_DEL_WIPHY" },
    { WS_NL80211_CMD_GET_INTERFACE,         "NL80211_CMD_GET_INTERFACE" },
    { WS_NL80211_CMD_SET_INTERFACE,         "NL80211_CMD_SET_INTERFACE" },
    { WS_NL80211_CMD_NEW_INTERFACE,         "NL80211_CMD_NEW_INTERFACE" },
    { WS_NL80211_CMD_DEL_INTERFACE,         "NL80211_CMD_DEL_INTERFACE" },
    { WS_NL80211_CMD_GET_KEY,               "NL80211_CMD_GET_KEY" },
    { WS_NL80211_CMD_SET_KEY,               "NL80211_CMD_SET_KEY" },
    { WS_NL80211_CMD_NEW_KEY,               "NL80211_CMD_NEW_KEY" },
    { WS_NL80211_CMD_DEL_KEY,               "NL80211_CMD_DEL_KEY" },
    { WS_NL80211_CMD_GET_BEACON,            "NL80211_CMD_GET_BEACON" },
    { WS_NL80211_CMD_SET_BEACON,            "NL80211_CMD_SET_BEACON" },
    { WS_NL80211_CMD_START_AP,              "NL80211_CMD_START_AP" },
    { WS_NL80211_CMD_STOP_AP,               "NL80211_CMD_STOP_AP" },
    { WS_NL80211_CMD_GET_STATION,           "NL80211_CMD_GET_STATION" },
    { WS_NL80211_CMD_SET_STATION,           "NL80211_CMD_SET_STATION" },
    { WS_NL80211_CMD_NEW_STATION,           "NL80211_CMD_NEW_STATION" },
    { WS_NL80211_CMD_DEL_STATION,           "NL80211_CMD_DEL_STATION" },
    { WS_NL80211_CMD_GET_MPATH,             "NL80211_CMD_GET_MPATH" },
    { WS_NL80211_CMD_SET_MPATH,             "NL80211_CMD_SET_MPATH" },
    { WS_NL80211_CMD_NEW_MPATH,             "NL80211_CMD_NEW_MPATH" },
    { WS_NL80211_CMD_DEL_MPATH,             "NL80211_CMD_DEL_MPATH" },
    { WS_NL80211_CMD_SET_BSS,               "NL80211_CMD_SET_BSS" },
    { WS_NL80211_CMD_SET_REG,               "NL80211_CMD_SET_REG" },
    { WS_NL80211_CMD_REQ_SET_REG,           "NL80211_CMD_REQ_SET_REG" },
    { WS_NL80211_CMD_GET_MESH_CONFIG,       "NL80211_CMD_GET_MESH_CONFIG" },
    { WS_NL80211_CMD_SET_MESH_CONFIG,       "NL80211_CMD_SET_MESH_CONFIG" },
    { WS_NL80211_CMD_SET_MGMT_EXTRA_IE,     "NL80211_CMD_SET_MGMT_EXTRA_IE" },
    { WS_NL80211_CMD_GET_REG,               "NL80211_CMD_GET_REG" },
    { WS_NL80211_CMD_GET_SCAN,              "NL80211_CMD_GET_SCAN" },
    { WS_NL80211_CMD_TRIGGER_SCAN,          "NL80211_CMD_TRIGGER_SCAN" },
    { WS_NL80211_CMD_NEW_SCAN_RESULTS,      "NL80211_CMD_NEW_SCAN_RESULTS" },
    { WS_NL80211_CMD_SCAN_ABORTED,          "NL80211_CMD_SCAN_ABORTED" },
    { WS_NL80211_CMD_REG_CHANGE,            "NL80211_CMD_REG_CHANGE" },
    { WS_NL80211_CMD_AUTHENTICATE,          "NL80211_CMD_AUTHENTICATE" },
    { WS_NL80211_CMD_ASSOCIATE,             "NL80211_CMD_ASSOCIATE" },
    { WS_NL80211_CMD_DEAUTHENTICATE,        "NL80211_CMD_DEAUTHENTICATE" },
    { WS_NL80211_CMD_DISASSOCIATE,          "NL80211_CMD_DISASSOCIATE" },
    { WS_NL80211_CMD_MICHAEL_MIC_FAILURE,   "NL80211_CMD_MICHAEL_MIC_FAILURE" },
    { WS_NL80211_CMD_REG_BEACON_HINT,       "NL80211_CMD_REG_BEACON_HINT" },
    { WS_NL80211_CMD_JOIN_IBSS,             "NL80211_CMD_JOIN_IBSS" },
    { WS_NL80211_CMD_LEAVE_IBSS,            "NL80211_CMD_LEAVE_IBSS" },
    { WS_NL80211_CMD_TESTMODE,              "NL80211_CMD_TESTMODE" },
    { WS_NL80211_CMD_CONNECT,               "NL80211_CMD_CONNECT" },
    { WS_NL80211_CMD_ROAM,                  "NL80211_CMD_ROAM" },
    { WS_NL80211_CMD_DISCONNECT,            "NL80211_CMD_DISCONNECT" },
    { WS_NL80211_CMD_SET_WIPHY_NETNS,       "NL80211_CMD_SET_WIPHY_NETNS" },
    { WS_NL80211_CMD_GET_SURVEY,            "NL80211_CMD_GET_SURVEY" },
    { WS_NL80211_CMD_NEW_SURVEY_RESULTS,    "NL80211_CMD_NEW_SURVEY_RESULTS" },
    { WS_NL80211_CMD_SET_PMKSA,             "NL80211_CMD_SET_PMKSA" },
    { WS_NL80211_CMD_DEL_PMKSA,             "NL80211_CMD_DEL_PMKSA" },
    { WS_NL80211_CMD_FLUSH_PMKSA,           "NL80211_CMD_FLUSH_PMKSA" },
    { WS_NL80211_CMD_REMAIN_ON_CHANNEL,     "NL80211_CMD_REMAIN_ON_CHANNEL" },
    { WS_NL80211_CMD_CANCEL_REMAIN_ON_CHANNEL, "NL80211_CMD_CANCEL_REMAIN_ON_CHANNEL" },
    { WS_NL80211_CMD_SET_TX_BITRATE_MASK,   "NL80211_CMD_SET_TX_BITRATE_MASK" },
    { WS_NL80211_CMD_REGISTER_FRAME,        "NL80211_CMD_REGISTER_FRAME" },
    { WS_NL80211_CMD_FRAME,                 "NL80211_CMD_FRAME" },
    { WS_NL80211_CMD_FRAME_TX_STATUS,       "NL80211_CMD_FRAME_TX_STATUS" },
    { WS_NL80211_CMD_SET_POWER_SAVE,        "NL80211_CMD_SET_POWER_SAVE" },
    { WS_NL80211_CMD_GET_POWER_SAVE,        "NL80211_CMD_GET_POWER_SAVE" },
    { WS_NL80211_CMD_SET_CQM,               "NL80211_CMD_SET_CQM" },
    { WS_NL80211_CMD_NOTIFY_CQM,            "NL80211_CMD_NOTIFY_CQM" },
    { WS_NL80211_CMD_SET_CHANNEL,           "NL80211_CMD_SET_CHANNEL" },
    { WS_NL80211_CMD_SET_WDS_PEER,          "NL80211_CMD_SET_WDS_PEER" },
    { WS_NL80211_CMD_FRAME_WAIT_CANCEL,     "NL80211_CMD_FRAME_WAIT_CANCEL" },
    { WS_NL80211_CMD_JOIN_MESH,             "NL80211_CMD_JOIN_MESH" },
    { WS_NL80211_CMD_LEAVE_MESH,            "NL80211_CMD_LEAVE_MESH" },
    { WS_NL80211_CMD_UNPROT_DEAUTHENTICATE, "NL80211_CMD_UNPROT_DEAUTHENTICATE" },
    { WS_NL80211_CMD_UNPROT_DISASSOCIATE,   "NL80211_CMD_UNPROT_DISASSOCIATE" },
    { WS_NL80211_CMD_NEW_PEER_CANDIDATE,    "NL80211_CMD_NEW_PEER_CANDIDATE" },
    { WS_NL80211_CMD_GET_WOWLAN,            "NL80211_CMD_GET_WOWLAN" },
    { WS_NL80211_CMD_SET_WOWLAN,            "NL80211_CMD_SET_WOWLAN" },
    { WS_NL80211_CMD_START_SCHED_SCAN,      "NL80211_CMD_START_SCHED_SCAN" },
    { WS_NL80211_CMD_STOP_SCHED_SCAN,       "NL80211_CMD_STOP_SCHED_SCAN" },
    { WS_NL80211_CMD_SCHED_SCAN_RESULTS,    "NL80211_CMD_SCHED_SCAN_RESULTS" },
    { WS_NL80211_CMD_SCHED_SCAN_STOPPED,    "NL80211_CMD_SCHED_SCAN_STOPPED" },
    { WS_NL80211_CMD_SET_REKEY_OFFLOAD,     "NL80211_CMD_SET_REKEY_OFFLOAD" },
    { WS_NL80211_CMD_PMKSA_CANDIDATE,       "NL80211_CMD_PMKSA_CANDIDATE" },
    { WS_NL80211_CMD_TDLS_OPER,             "NL80211_CMD_TDLS_OPER" },
    { WS_NL80211_CMD_TDLS_MGMT,             "NL80211_CMD_TDLS_MGMT" },
    { WS_NL80211_CMD_UNEXPECTED_FRAME,      "NL80211_CMD_UNEXPECTED_FRAME" },
    { WS_NL80211_CMD_PROBE_CLIENT,          "NL80211_CMD_PROBE_CLIENT" },
    { WS_NL80211_CMD_REGISTER_BEACONS,      "NL80211_CMD_REGISTER_BEACONS" },
    { WS_NL80211_CMD_UNEXPECTED_4ADDR_FRAME, "NL80211_CMD_UNEXPECTED_4ADDR_FRAME" },
    { WS_NL80211_CMD_SET_NOACK_MAP,         "NL80211_CMD_SET_NOACK_MAP" },
    { WS_NL80211_CMD_CH_SWITCH_NOTIFY,      "NL80211_CMD_CH_SWITCH_NOTIFY" },
    { WS_NL80211_CMD_START_P2P_DEVICE,      "NL80211_CMD_START_P2P_DEVICE" },
    { WS_NL80211_CMD_STOP_P2P_DEVICE,       "NL80211_CMD_STOP_P2P_DEVICE" },
    { WS_NL80211_CMD_CONN_FAILED,           "NL80211_CMD_CONN_FAILED" },
    { WS_NL80211_CMD_SET_MCAST_RATE,        "NL80211_CMD_SET_MCAST_RATE" },
    { WS_NL80211_CMD_SET_MAC_ACL,           "NL80211_CMD_SET_MAC_ACL" },
    { WS_NL80211_CMD_RADAR_DETECT,          "NL80211_CMD_RADAR_DETECT" },
    { WS_NL80211_CMD_GET_PROTOCOL_FEATURES, "NL80211_CMD_GET_PROTOCOL_FEATURES" },
    { WS_NL80211_CMD_UPDATE_FT_IES,         "NL80211_CMD_UPDATE_FT_IES" },
    { WS_NL80211_CMD_FT_EVENT,              "NL80211_CMD_FT_EVENT" },
    { WS_NL80211_CMD_CRIT_PROTOCOL_START,   "NL80211_CMD_CRIT_PROTOCOL_START" },
    { WS_NL80211_CMD_CRIT_PROTOCOL_STOP,    "NL80211_CMD_CRIT_PROTOCOL_STOP" },
    { WS_NL80211_CMD_GET_COALESCE,          "NL80211_CMD_GET_COALESCE" },
    { WS_NL80211_CMD_SET_COALESCE,          "NL80211_CMD_SET_COALESCE" },
    { WS_NL80211_CMD_CHANNEL_SWITCH,        "NL80211_CMD_CHANNEL_SWITCH" },
    { WS_NL80211_CMD_VENDOR,                "NL80211_CMD_VENDOR" },
    { WS_NL80211_CMD_SET_QOS_MAP,           "NL80211_CMD_SET_QOS_MAP" },
    { WS_NL80211_CMD_ADD_TX_TS,             "NL80211_CMD_ADD_TX_TS" },
    { WS_NL80211_CMD_DEL_TX_TS,             "NL80211_CMD_DEL_TX_TS" },
    { WS_NL80211_CMD_GET_MPP,               "NL80211_CMD_GET_MPP" },
    { WS_NL80211_CMD_JOIN_OCB,              "NL80211_CMD_JOIN_OCB" },
    { WS_NL80211_CMD_LEAVE_OCB,             "NL80211_CMD_LEAVE_OCB" },
    { WS_NL80211_CMD_CH_SWITCH_STARTED_NOTIFY, "NL80211_CMD_CH_SWITCH_STARTED_NOTIFY" },
    { WS_NL80211_CMD_TDLS_CHANNEL_SWITCH,   "NL80211_CMD_TDLS_CHANNEL_SWITCH" },
    { WS_NL80211_CMD_TDLS_CANCEL_CHANNEL_SWITCH, "NL80211_CMD_TDLS_CANCEL_CHANNEL_SWITCH" },
    { WS_NL80211_CMD_WIPHY_REG_CHANGE,      "NL80211_CMD_WIPHY_REG_CHANGE" },
    { WS_NL80211_CMD_ABORT_SCAN,            "NL80211_CMD_ABORT_SCAN" },
    { WS_NL80211_CMD_START_NAN,             "NL80211_CMD_START_NAN" },
    { WS_NL80211_CMD_STOP_NAN,              "NL80211_CMD_STOP_NAN" },
    { WS_NL80211_CMD_ADD_NAN_FUNCTION,      "NL80211_CMD_ADD_NAN_FUNCTION" },
    { WS_NL80211_CMD_DEL_NAN_FUNCTION,      "NL80211_CMD_DEL_NAN_FUNCTION" },
    { WS_NL80211_CMD_CHANGE_NAN_CONFIG,     "NL80211_CMD_CHANGE_NAN_CONFIG" },
    { WS_NL80211_CMD_NAN_MATCH,             "NL80211_CMD_NAN_MATCH" },
    { WS_NL80211_CMD_SET_MULTICAST_TO_UNICAST, "NL80211_CMD_SET_MULTICAST_TO_UNICAST" },
    { WS_NL80211_CMD_UPDATE_CONNECT_PARAMS, "NL80211_CMD_UPDATE_CONNECT_PARAMS" },
    { WS_NL80211_CMD_SET_PMK,               "NL80211_CMD_SET_PMK" },
    { WS_NL80211_CMD_DEL_PMK,               "NL80211_CMD_DEL_PMK" },
    { WS_NL80211_CMD_PORT_AUTHORIZED,       "NL80211_CMD_PORT_AUTHORIZED" },
    { WS_NL80211_CMD_RELOAD_REGDB,          "NL80211_CMD_RELOAD_REGDB" },
    { WS_NL80211_CMD_EXTERNAL_AUTH,         "NL80211_CMD_EXTERNAL_AUTH" },
    { WS_NL80211_CMD_STA_OPMODE_CHANGED,    "NL80211_CMD_STA_OPMODE_CHANGED" },
    { WS_NL80211_CMD_CONTROL_PORT_FRAME,    "NL80211_CMD_CONTROL_PORT_FRAME" },
    { WS_NL80211_CMD_GET_FTM_RESPONDER_STATS, "NL80211_CMD_GET_FTM_RESPONDER_STATS" },
    { WS_NL80211_CMD_PEER_MEASUREMENT_START, "NL80211_CMD_PEER_MEASUREMENT_START" },
    { WS_NL80211_CMD_PEER_MEASUREMENT_RESULT, "NL80211_CMD_PEER_MEASUREMENT_RESULT" },
    { WS_NL80211_CMD_PEER_MEASUREMENT_COMPLETE, "NL80211_CMD_PEER_MEASUREMENT_COMPLETE" },
    { WS_NL80211_CMD_NOTIFY_RADAR,          "NL80211_CMD_NOTIFY_RADAR" },
    { WS_NL80211_CMD_UPDATE_OWE_INFO,       "NL80211_CMD_UPDATE_OWE_INFO" },
    { WS_NL80211_CMD_PROBE_MESH_LINK,       "NL80211_CMD_PROBE_MESH_LINK" },
    { WS_NL80211_CMD_SET_TID_CONFIG,        "NL80211_CMD_SET_TID_CONFIG" },
    { WS_NL80211_CMD_UNPROT_BEACON,         "NL80211_CMD_UNPROT_BEACON" },
    { WS_NL80211_CMD_CONTROL_PORT_FRAME_TX_STATUS, "NL80211_CMD_CONTROL_PORT_FRAME_TX_STATUS" },
    { WS_NL80211_CMD_SET_SAR_SPECS,         "NL80211_CMD_SET_SAR_SPECS" },
    { WS_NL80211_CMD_OBSS_COLOR_COLLISION,  "NL80211_CMD_OBSS_COLOR_COLLISION" },
    { WS_NL80211_CMD_COLOR_CHANGE_REQUEST,  "NL80211_CMD_COLOR_CHANGE_REQUEST" },
    { WS_NL80211_CMD_COLOR_CHANGE_STARTED,  "NL80211_CMD_COLOR_CHANGE_STARTED" },
    { WS_NL80211_CMD_COLOR_CHANGE_ABORTED,  "NL80211_CMD_COLOR_CHANGE_ABORTED" },
    { WS_NL80211_CMD_COLOR_CHANGE_COMPLETED, "NL80211_CMD_COLOR_CHANGE_COMPLETED" },
    { WS_NL80211_CMD_SET_FILS_AAD,          "NL80211_CMD_SET_FILS_AAD" },
    { WS_NL80211_CMD_ASSOC_COMEBACK,        "NL80211_CMD_ASSOC_COMEBACK" },
    { WS_NL80211_CMD_ADD_LINK,              "NL80211_CMD_ADD_LINK" },
    { WS_NL80211_CMD_REMOVE_LINK,           "NL80211_CMD_REMOVE_LINK" },
    { WS_NL80211_CMD_ADD_LINK_STA,          "NL80211_CMD_ADD_LINK_STA" },
    { WS_NL80211_CMD_MODIFY_LINK_STA,       "NL80211_CMD_MODIFY_LINK_STA" },
    { WS_NL80211_CMD_REMOVE_LINK_STA,       "NL80211_CMD_REMOVE_LINK_STA" },
    { WS_NL80211_CMD_SET_HW_TIMESTAMP,      "NL80211_CMD_SET_HW_TIMESTAMP" },
    { WS_NL80211_CMD_LINKS_REMOVED,         "NL80211_CMD_LINKS_REMOVED" },
    { WS_NL80211_CMD_SET_TID_TO_LINK_MAPPING, "NL80211_CMD_SET_TID_TO_LINK_MAPPING" },
    { WS_NL80211_CMD_ASSOC_MLO_RECONF,      "NL80211_CMD_ASSOC_MLO_RECONF" },
    { WS_NL80211_CMD_EPCS_CFG,              "NL80211_CMD_EPCS_CFG" },
    { WS_NL80211_CMD_NAN_NEXT_DW_NOTIFICATION, "NL80211_CMD_NAN_NEXT_DW_NOTIFICATION" },
    { WS_NL80211_CMD_NAN_CLUSTER_JOINED,    "NL80211_CMD_NAN_CLUSTER_JOINED" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_commands_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_commands_vals);

static const value_string ws_nl80211_attrs_vals[] = {
    { WS_NL80211_ATTR_UNSPEC,               "NL80211_ATTR_UNSPEC" },
    { WS_NL80211_ATTR_WIPHY,                "NL80211_ATTR_WIPHY" },
    { WS_NL80211_ATTR_WIPHY_NAME,           "NL80211_ATTR_WIPHY_NAME" },
    { WS_NL80211_ATTR_IFINDEX,              "NL80211_ATTR_IFINDEX" },
    { WS_NL80211_ATTR_IFNAME,               "NL80211_ATTR_IFNAME" },
    { WS_NL80211_ATTR_IFTYPE,               "NL80211_ATTR_IFTYPE" },
    { WS_NL80211_ATTR_MAC,                  "NL80211_ATTR_MAC" },
    { WS_NL80211_ATTR_KEY_DATA,             "NL80211_ATTR_KEY_DATA" },
    { WS_NL80211_ATTR_KEY_IDX,              "NL80211_ATTR_KEY_IDX" },
    { WS_NL80211_ATTR_KEY_CIPHER,           "NL80211_ATTR_KEY_CIPHER" },
    { WS_NL80211_ATTR_KEY_SEQ,              "NL80211_ATTR_KEY_SEQ" },
    { WS_NL80211_ATTR_KEY_DEFAULT,          "NL80211_ATTR_KEY_DEFAULT" },
    { WS_NL80211_ATTR_BEACON_INTERVAL,      "NL80211_ATTR_BEACON_INTERVAL" },
    { WS_NL80211_ATTR_DTIM_PERIOD,          "NL80211_ATTR_DTIM_PERIOD" },
    { WS_NL80211_ATTR_BEACON_HEAD,          "NL80211_ATTR_BEACON_HEAD" },
    { WS_NL80211_ATTR_BEACON_TAIL,          "NL80211_ATTR_BEACON_TAIL" },
    { WS_NL80211_ATTR_STA_AID,              "NL80211_ATTR_STA_AID" },
    { WS_NL80211_ATTR_STA_FLAGS,            "NL80211_ATTR_STA_FLAGS" },
    { WS_NL80211_ATTR_STA_LISTEN_INTERVAL,  "NL80211_ATTR_STA_LISTEN_INTERVAL" },
    { WS_NL80211_ATTR_STA_SUPPORTED_RATES,  "NL80211_ATTR_STA_SUPPORTED_RATES" },
    { WS_NL80211_ATTR_STA_VLAN,             "NL80211_ATTR_STA_VLAN" },
    { WS_NL80211_ATTR_STA_INFO,             "NL80211_ATTR_STA_INFO" },
    { WS_NL80211_ATTR_WIPHY_BANDS,          "NL80211_ATTR_WIPHY_BANDS" },
    { WS_NL80211_ATTR_MNTR_FLAGS,           "NL80211_ATTR_MNTR_FLAGS" },
    { WS_NL80211_ATTR_MESH_ID,              "NL80211_ATTR_MESH_ID" },
    { WS_NL80211_ATTR_STA_PLINK_ACTION,     "NL80211_ATTR_STA_PLINK_ACTION" },
    { WS_NL80211_ATTR_MPATH_NEXT_HOP,       "NL80211_ATTR_MPATH_NEXT_HOP" },
    { WS_NL80211_ATTR_MPATH_INFO,           "NL80211_ATTR_MPATH_INFO" },
    { WS_NL80211_ATTR_BSS_CTS_PROT,         "NL80211_ATTR_BSS_CTS_PROT" },
    { WS_NL80211_ATTR_BSS_SHORT_PREAMBLE,   "NL80211_ATTR_BSS_SHORT_PREAMBLE" },
    { WS_NL80211_ATTR_BSS_SHORT_SLOT_TIME,  "NL80211_ATTR_BSS_SHORT_SLOT_TIME" },
    { WS_NL80211_ATTR_HT_CAPABILITY,        "NL80211_ATTR_HT_CAPABILITY" },
    { WS_NL80211_ATTR_SUPPORTED_IFTYPES,    "NL80211_ATTR_SUPPORTED_IFTYPES" },
    { WS_NL80211_ATTR_REG_ALPHA2,           "NL80211_ATTR_REG_ALPHA2" },
    { WS_NL80211_ATTR_REG_RULES,            "NL80211_ATTR_REG_RULES" },
    { WS_NL80211_ATTR_MESH_CONFIG,          "NL80211_ATTR_MESH_CONFIG" },
    { WS_NL80211_ATTR_BSS_BASIC_RATES,      "NL80211_ATTR_BSS_BASIC_RATES" },
    { WS_NL80211_ATTR_WIPHY_TXQ_PARAMS,     "NL80211_ATTR_WIPHY_TXQ_PARAMS" },
    { WS_NL80211_ATTR_WIPHY_FREQ,           "NL80211_ATTR_WIPHY_FREQ" },
    { WS_NL80211_ATTR_WIPHY_CHANNEL_TYPE,   "NL80211_ATTR_WIPHY_CHANNEL_TYPE" },
    { WS_NL80211_ATTR_KEY_DEFAULT_MGMT,     "NL80211_ATTR_KEY_DEFAULT_MGMT" },
    { WS_NL80211_ATTR_MGMT_SUBTYPE,         "NL80211_ATTR_MGMT_SUBTYPE" },
    { WS_NL80211_ATTR_IE,                   "NL80211_ATTR_IE" },
    { WS_NL80211_ATTR_MAX_NUM_SCAN_SSIDS,   "NL80211_ATTR_MAX_NUM_SCAN_SSIDS" },
    { WS_NL80211_ATTR_SCAN_FREQUENCIES,     "NL80211_ATTR_SCAN_FREQUENCIES" },
    { WS_NL80211_ATTR_SCAN_SSIDS,           "NL80211_ATTR_SCAN_SSIDS" },
    { WS_NL80211_ATTR_GENERATION,           "NL80211_ATTR_GENERATION" },
    { WS_NL80211_ATTR_BSS,                  "NL80211_ATTR_BSS" },
    { WS_NL80211_ATTR_REG_INITIATOR,        "NL80211_ATTR_REG_INITIATOR" },
    { WS_NL80211_ATTR_REG_TYPE,             "NL80211_ATTR_REG_TYPE" },
    { WS_NL80211_ATTR_SUPPORTED_COMMANDS,   "NL80211_ATTR_SUPPORTED_COMMANDS" },
    { WS_NL80211_ATTR_FRAME,                "NL80211_ATTR_FRAME" },
    { WS_NL80211_ATTR_SSID,                 "NL80211_ATTR_SSID" },
    { WS_NL80211_ATTR_AUTH_TYPE,            "NL80211_ATTR_AUTH_TYPE" },
    { WS_NL80211_ATTR_REASON_CODE,          "NL80211_ATTR_REASON_CODE" },
    { WS_NL80211_ATTR_KEY_TYPE,             "NL80211_ATTR_KEY_TYPE" },
    { WS_NL80211_ATTR_MAX_SCAN_IE_LEN,      "NL80211_ATTR_MAX_SCAN_IE_LEN" },
    { WS_NL80211_ATTR_CIPHER_SUITES,        "NL80211_ATTR_CIPHER_SUITES" },
    { WS_NL80211_ATTR_FREQ_BEFORE,          "NL80211_ATTR_FREQ_BEFORE" },
    { WS_NL80211_ATTR_FREQ_AFTER,           "NL80211_ATTR_FREQ_AFTER" },
    { WS_NL80211_ATTR_FREQ_FIXED,           "NL80211_ATTR_FREQ_FIXED" },
    { WS_NL80211_ATTR_WIPHY_RETRY_SHORT,    "NL80211_ATTR_WIPHY_RETRY_SHORT" },
    { WS_NL80211_ATTR_WIPHY_RETRY_LONG,     "NL80211_ATTR_WIPHY_RETRY_LONG" },
    { WS_NL80211_ATTR_WIPHY_FRAG_THRESHOLD, "NL80211_ATTR_WIPHY_FRAG_THRESHOLD" },
    { WS_NL80211_ATTR_WIPHY_RTS_THRESHOLD,  "NL80211_ATTR_WIPHY_RTS_THRESHOLD" },
    { WS_NL80211_ATTR_TIMED_OUT,            "NL80211_ATTR_TIMED_OUT" },
    { WS_NL80211_ATTR_USE_MFP,              "NL80211_ATTR_USE_MFP" },
    { WS_NL80211_ATTR_STA_FLAGS2,           "NL80211_ATTR_STA_FLAGS2" },
    { WS_NL80211_ATTR_CONTROL_PORT,         "NL80211_ATTR_CONTROL_PORT" },
    { WS_NL80211_ATTR_TESTDATA,             "NL80211_ATTR_TESTDATA" },
    { WS_NL80211_ATTR_PRIVACY,              "NL80211_ATTR_PRIVACY" },
    { WS_NL80211_ATTR_DISCONNECTED_BY_AP,   "NL80211_ATTR_DISCONNECTED_BY_AP" },
    { WS_NL80211_ATTR_STATUS_CODE,          "NL80211_ATTR_STATUS_CODE" },
    { WS_NL80211_ATTR_CIPHER_SUITES_PAIRWISE, "NL80211_ATTR_CIPHER_SUITES_PAIRWISE" },
    { WS_NL80211_ATTR_CIPHER_SUITE_GROUP,   "NL80211_ATTR_CIPHER_SUITE_GROUP" },
    { WS_NL80211_ATTR_WPA_VERSIONS,         "NL80211_ATTR_WPA_VERSIONS" },
    { WS_NL80211_ATTR_AKM_SUITES,           "NL80211_ATTR_AKM_SUITES" },
    { WS_NL80211_ATTR_REQ_IE,               "NL80211_ATTR_REQ_IE" },
    { WS_NL80211_ATTR_RESP_IE,              "NL80211_ATTR_RESP_IE" },
    { WS_NL80211_ATTR_PREV_BSSID,           "NL80211_ATTR_PREV_BSSID" },
    { WS_NL80211_ATTR_KEY,                  "NL80211_ATTR_KEY" },
    { WS_NL80211_ATTR_KEYS,                 "NL80211_ATTR_KEYS" },
    { WS_NL80211_ATTR_PID,                  "NL80211_ATTR_PID" },
    { WS_NL80211_ATTR_4ADDR,                "NL80211_ATTR_4ADDR" },
    { WS_NL80211_ATTR_SURVEY_INFO,          "NL80211_ATTR_SURVEY_INFO" },
    { WS_NL80211_ATTR_PMKID,                "NL80211_ATTR_PMKID" },
    { WS_NL80211_ATTR_MAX_NUM_PMKIDS,       "NL80211_ATTR_MAX_NUM_PMKIDS" },
    { WS_NL80211_ATTR_DURATION,             "NL80211_ATTR_DURATION" },
    { WS_NL80211_ATTR_COOKIE,               "NL80211_ATTR_COOKIE" },
    { WS_NL80211_ATTR_WIPHY_COVERAGE_CLASS, "NL80211_ATTR_WIPHY_COVERAGE_CLASS" },
    { WS_NL80211_ATTR_TX_RATES,             "NL80211_ATTR_TX_RATES" },
    { WS_NL80211_ATTR_FRAME_MATCH,          "NL80211_ATTR_FRAME_MATCH" },
    { WS_NL80211_ATTR_ACK,                  "NL80211_ATTR_ACK" },
    { WS_NL80211_ATTR_PS_STATE,             "NL80211_ATTR_PS_STATE" },
    { WS_NL80211_ATTR_CQM,                  "NL80211_ATTR_CQM" },
    { WS_NL80211_ATTR_LOCAL_STATE_CHANGE,   "NL80211_ATTR_LOCAL_STATE_CHANGE" },
    { WS_NL80211_ATTR_AP_ISOLATE,           "NL80211_ATTR_AP_ISOLATE" },
    { WS_NL80211_ATTR_WIPHY_TX_POWER_SETTING, "NL80211_ATTR_WIPHY_TX_POWER_SETTING" },
    { WS_NL80211_ATTR_WIPHY_TX_POWER_LEVEL, "NL80211_ATTR_WIPHY_TX_POWER_LEVEL" },
    { WS_NL80211_ATTR_TX_FRAME_TYPES,       "NL80211_ATTR_TX_FRAME_TYPES" },
    { WS_NL80211_ATTR_RX_FRAME_TYPES,       "NL80211_ATTR_RX_FRAME_TYPES" },
    { WS_NL80211_ATTR_FRAME_TYPE,           "NL80211_ATTR_FRAME_TYPE" },
    { WS_NL80211_ATTR_CONTROL_PORT_ETHERTYPE, "NL80211_ATTR_CONTROL_PORT_ETHERTYPE" },
    { WS_NL80211_ATTR_CONTROL_PORT_NO_ENCRYPT, "NL80211_ATTR_CONTROL_PORT_NO_ENCRYPT" },
    { WS_NL80211_ATTR_SUPPORT_IBSS_RSN,     "NL80211_ATTR_SUPPORT_IBSS_RSN" },
    { WS_NL80211_ATTR_WIPHY_ANTENNA_TX,     "NL80211_ATTR_WIPHY_ANTENNA_TX" },
    { WS_NL80211_ATTR_WIPHY_ANTENNA_RX,     "NL80211_ATTR_WIPHY_ANTENNA_RX" },
    { WS_NL80211_ATTR_MCAST_RATE,           "NL80211_ATTR_MCAST_RATE" },
    { WS_NL80211_ATTR_OFFCHANNEL_TX_OK,     "NL80211_ATTR_OFFCHANNEL_TX_OK" },
    { WS_NL80211_ATTR_BSS_HT_OPMODE,        "NL80211_ATTR_BSS_HT_OPMODE" },
    { WS_NL80211_ATTR_KEY_DEFAULT_TYPES,    "NL80211_ATTR_KEY_DEFAULT_TYPES" },
    { WS_NL80211_ATTR_MAX_REMAIN_ON_CHANNEL_DURATION, "NL80211_ATTR_MAX_REMAIN_ON_CHANNEL_DURATION" },
    { WS_NL80211_ATTR_MESH_SETUP,           "NL80211_ATTR_MESH_SETUP" },
    { WS_NL80211_ATTR_WIPHY_ANTENNA_AVAIL_TX, "NL80211_ATTR_WIPHY_ANTENNA_AVAIL_TX" },
    { WS_NL80211_ATTR_WIPHY_ANTENNA_AVAIL_RX, "NL80211_ATTR_WIPHY_ANTENNA_AVAIL_RX" },
    { WS_NL80211_ATTR_SUPPORT_MESH_AUTH,    "NL80211_ATTR_SUPPORT_MESH_AUTH" },
    { WS_NL80211_ATTR_STA_PLINK_STATE,      "NL80211_ATTR_STA_PLINK_STATE" },
    { WS_NL80211_ATTR_WOWLAN_TRIGGERS,      "NL80211_ATTR_WOWLAN_TRIGGERS" },
    { WS_NL80211_ATTR_WOWLAN_TRIGGERS_SUPPORTED, "NL80211_ATTR_WOWLAN_TRIGGERS_SUPPORTED" },
    { WS_NL80211_ATTR_SCHED_SCAN_INTERVAL,  "NL80211_ATTR_SCHED_SCAN_INTERVAL" },
    { WS_NL80211_ATTR_INTERFACE_COMBINATIONS, "NL80211_ATTR_INTERFACE_COMBINATIONS" },
    { WS_NL80211_ATTR_SOFTWARE_IFTYPES,     "NL80211_ATTR_SOFTWARE_IFTYPES" },
    { WS_NL80211_ATTR_REKEY_DATA,           "NL80211_ATTR_REKEY_DATA" },
    { WS_NL80211_ATTR_MAX_NUM_SCHED_SCAN_SSIDS, "NL80211_ATTR_MAX_NUM_SCHED_SCAN_SSIDS" },
    { WS_NL80211_ATTR_MAX_SCHED_SCAN_IE_LEN, "NL80211_ATTR_MAX_SCHED_SCAN_IE_LEN" },
    { WS_NL80211_ATTR_SCAN_SUPP_RATES,      "NL80211_ATTR_SCAN_SUPP_RATES" },
    { WS_NL80211_ATTR_HIDDEN_SSID,          "NL80211_ATTR_HIDDEN_SSID" },
    { WS_NL80211_ATTR_IE_PROBE_RESP,        "NL80211_ATTR_IE_PROBE_RESP" },
    { WS_NL80211_ATTR_IE_ASSOC_RESP,        "NL80211_ATTR_IE_ASSOC_RESP" },
    { WS_NL80211_ATTR_STA_WME,              "NL80211_ATTR_STA_WME" },
    { WS_NL80211_ATTR_SUPPORT_AP_UAPSD,     "NL80211_ATTR_SUPPORT_AP_UAPSD" },
    { WS_NL80211_ATTR_ROAM_SUPPORT,         "NL80211_ATTR_ROAM_SUPPORT" },
    { WS_NL80211_ATTR_SCHED_SCAN_MATCH,     "NL80211_ATTR_SCHED_SCAN_MATCH" },
    { WS_NL80211_ATTR_MAX_MATCH_SETS,       "NL80211_ATTR_MAX_MATCH_SETS" },
    { WS_NL80211_ATTR_PMKSA_CANDIDATE,      "NL80211_ATTR_PMKSA_CANDIDATE" },
    { WS_NL80211_ATTR_TX_NO_CCK_RATE,       "NL80211_ATTR_TX_NO_CCK_RATE" },
    { WS_NL80211_ATTR_TDLS_ACTION,          "NL80211_ATTR_TDLS_ACTION" },
    { WS_NL80211_ATTR_TDLS_DIALOG_TOKEN,    "NL80211_ATTR_TDLS_DIALOG_TOKEN" },
    { WS_NL80211_ATTR_TDLS_OPERATION,       "NL80211_ATTR_TDLS_OPERATION" },
    { WS_NL80211_ATTR_TDLS_SUPPORT,         "NL80211_ATTR_TDLS_SUPPORT" },
    { WS_NL80211_ATTR_TDLS_EXTERNAL_SETUP,  "NL80211_ATTR_TDLS_EXTERNAL_SETUP" },
    { WS_NL80211_ATTR_DEVICE_AP_SME,        "NL80211_ATTR_DEVICE_AP_SME" },
    { WS_NL80211_ATTR_DONT_WAIT_FOR_ACK,    "NL80211_ATTR_DONT_WAIT_FOR_ACK" },
    { WS_NL80211_ATTR_FEATURE_FLAGS,        "NL80211_ATTR_FEATURE_FLAGS" },
    { WS_NL80211_ATTR_PROBE_RESP_OFFLOAD,   "NL80211_ATTR_PROBE_RESP_OFFLOAD" },
    { WS_NL80211_ATTR_PROBE_RESP,           "NL80211_ATTR_PROBE_RESP" },
    { WS_NL80211_ATTR_DFS_REGION,           "NL80211_ATTR_DFS_REGION" },
    { WS_NL80211_ATTR_DISABLE_HT,           "NL80211_ATTR_DISABLE_HT" },
    { WS_NL80211_ATTR_HT_CAPABILITY_MASK,   "NL80211_ATTR_HT_CAPABILITY_MASK" },
    { WS_NL80211_ATTR_NOACK_MAP,            "NL80211_ATTR_NOACK_MAP" },
    { WS_NL80211_ATTR_INACTIVITY_TIMEOUT,   "NL80211_ATTR_INACTIVITY_TIMEOUT" },
    { WS_NL80211_ATTR_RX_SIGNAL_DBM,        "NL80211_ATTR_RX_SIGNAL_DBM" },
    { WS_NL80211_ATTR_BG_SCAN_PERIOD,       "NL80211_ATTR_BG_SCAN_PERIOD" },
    { WS_NL80211_ATTR_WDEV,                 "NL80211_ATTR_WDEV" },
    { WS_NL80211_ATTR_USER_REG_HINT_TYPE,   "NL80211_ATTR_USER_REG_HINT_TYPE" },
    { WS_NL80211_ATTR_CONN_FAILED_REASON,   "NL80211_ATTR_CONN_FAILED_REASON" },
    { WS_NL80211_ATTR_AUTH_DATA,            "NL80211_ATTR_AUTH_DATA" },
    { WS_NL80211_ATTR_VHT_CAPABILITY,       "NL80211_ATTR_VHT_CAPABILITY" },
    { WS_NL80211_ATTR_SCAN_FLAGS,           "NL80211_ATTR_SCAN_FLAGS" },
    { WS_NL80211_ATTR_CHANNEL_WIDTH,        "NL80211_ATTR_CHANNEL_WIDTH" },
    { WS_NL80211_ATTR_CENTER_FREQ1,         "NL80211_ATTR_CENTER_FREQ1" },
    { WS_NL80211_ATTR_CENTER_FREQ2,         "NL80211_ATTR_CENTER_FREQ2" },
    { WS_NL80211_ATTR_P2P_CTWINDOW,         "NL80211_ATTR_P2P_CTWINDOW" },
    { WS_NL80211_ATTR_P2P_OPPPS,            "NL80211_ATTR_P2P_OPPPS" },
    { WS_NL80211_ATTR_LOCAL_MESH_POWER_MODE, "NL80211_ATTR_LOCAL_MESH_POWER_MODE" },
    { WS_NL80211_ATTR_ACL_POLICY,           "NL80211_ATTR_ACL_POLICY" },
    { WS_NL80211_ATTR_MAC_ADDRS,            "NL80211_ATTR_MAC_ADDRS" },
    { WS_NL80211_ATTR_MAC_ACL_MAX,          "NL80211_ATTR_MAC_ACL_MAX" },
    { WS_NL80211_ATTR_RADAR_EVENT,          "NL80211_ATTR_RADAR_EVENT" },
    { WS_NL80211_ATTR_EXT_CAPA,             "NL80211_ATTR_EXT_CAPA" },
    { WS_NL80211_ATTR_EXT_CAPA_MASK,        "NL80211_ATTR_EXT_CAPA_MASK" },
    { WS_NL80211_ATTR_STA_CAPABILITY,       "NL80211_ATTR_STA_CAPABILITY" },
    { WS_NL80211_ATTR_STA_EXT_CAPABILITY,   "NL80211_ATTR_STA_EXT_CAPABILITY" },
    { WS_NL80211_ATTR_PROTOCOL_FEATURES,    "NL80211_ATTR_PROTOCOL_FEATURES" },
    { WS_NL80211_ATTR_SPLIT_WIPHY_DUMP,     "NL80211_ATTR_SPLIT_WIPHY_DUMP" },
    { WS_NL80211_ATTR_DISABLE_VHT,          "NL80211_ATTR_DISABLE_VHT" },
    { WS_NL80211_ATTR_VHT_CAPABILITY_MASK,  "NL80211_ATTR_VHT_CAPABILITY_MASK" },
    { WS_NL80211_ATTR_MDID,                 "NL80211_ATTR_MDID" },
    { WS_NL80211_ATTR_IE_RIC,               "NL80211_ATTR_IE_RIC" },
    { WS_NL80211_ATTR_CRIT_PROT_ID,         "NL80211_ATTR_CRIT_PROT_ID" },
    { WS_NL80211_ATTR_MAX_CRIT_PROT_DURATION, "NL80211_ATTR_MAX_CRIT_PROT_DURATION" },
    { WS_NL80211_ATTR_PEER_AID,             "NL80211_ATTR_PEER_AID" },
    { WS_NL80211_ATTR_COALESCE_RULE,        "NL80211_ATTR_COALESCE_RULE" },
    { WS_NL80211_ATTR_CH_SWITCH_COUNT,      "NL80211_ATTR_CH_SWITCH_COUNT" },
    { WS_NL80211_ATTR_CH_SWITCH_BLOCK_TX,   "NL80211_ATTR_CH_SWITCH_BLOCK_TX" },
    { WS_NL80211_ATTR_CSA_IES,              "NL80211_ATTR_CSA_IES" },
    { WS_NL80211_ATTR_CNTDWN_OFFS_BEACON,   "NL80211_ATTR_CNTDWN_OFFS_BEACON" },
    { WS_NL80211_ATTR_CNTDWN_OFFS_PRESP,    "NL80211_ATTR_CNTDWN_OFFS_PRESP" },
    { WS_NL80211_ATTR_RXMGMT_FLAGS,         "NL80211_ATTR_RXMGMT_FLAGS" },
    { WS_NL80211_ATTR_STA_SUPPORTED_CHANNELS, "NL80211_ATTR_STA_SUPPORTED_CHANNELS" },
    { WS_NL80211_ATTR_STA_SUPPORTED_OPER_CLASSES, "NL80211_ATTR_STA_SUPPORTED_OPER_CLASSES" },
    { WS_NL80211_ATTR_HANDLE_DFS,           "NL80211_ATTR_HANDLE_DFS" },
    { WS_NL80211_ATTR_SUPPORT_5_MHZ,        "NL80211_ATTR_SUPPORT_5_MHZ" },
    { WS_NL80211_ATTR_SUPPORT_10_MHZ,       "NL80211_ATTR_SUPPORT_10_MHZ" },
    { WS_NL80211_ATTR_OPMODE_NOTIF,         "NL80211_ATTR_OPMODE_NOTIF" },
    { WS_NL80211_ATTR_VENDOR_ID,            "NL80211_ATTR_VENDOR_ID" },
    { WS_NL80211_ATTR_VENDOR_SUBCMD,        "NL80211_ATTR_VENDOR_SUBCMD" },
    { WS_NL80211_ATTR_VENDOR_DATA,          "NL80211_ATTR_VENDOR_DATA" },
    { WS_NL80211_ATTR_VENDOR_EVENTS,        "NL80211_ATTR_VENDOR_EVENTS" },
    { WS_NL80211_ATTR_QOS_MAP,              "NL80211_ATTR_QOS_MAP" },
    { WS_NL80211_ATTR_MAC_HINT,             "NL80211_ATTR_MAC_HINT" },
    { WS_NL80211_ATTR_WIPHY_FREQ_HINT,      "NL80211_ATTR_WIPHY_FREQ_HINT" },
    { WS_NL80211_ATTR_MAX_AP_ASSOC_STA,     "NL80211_ATTR_MAX_AP_ASSOC_STA" },
    { WS_NL80211_ATTR_TDLS_PEER_CAPABILITY, "NL80211_ATTR_TDLS_PEER_CAPABILITY" },
    { WS_NL80211_ATTR_SOCKET_OWNER,         "NL80211_ATTR_SOCKET_OWNER" },
    { WS_NL80211_ATTR_CSA_C_OFFSETS_TX,     "NL80211_ATTR_CSA_C_OFFSETS_TX" },
    { WS_NL80211_ATTR_MAX_CSA_COUNTERS,     "NL80211_ATTR_MAX_CSA_COUNTERS" },
    { WS_NL80211_ATTR_TDLS_INITIATOR,       "NL80211_ATTR_TDLS_INITIATOR" },
    { WS_NL80211_ATTR_USE_RRM,              "NL80211_ATTR_USE_RRM" },
    { WS_NL80211_ATTR_WIPHY_DYN_ACK,        "NL80211_ATTR_WIPHY_DYN_ACK" },
    { WS_NL80211_ATTR_TSID,                 "NL80211_ATTR_TSID" },
    { WS_NL80211_ATTR_USER_PRIO,            "NL80211_ATTR_USER_PRIO" },
    { WS_NL80211_ATTR_ADMITTED_TIME,        "NL80211_ATTR_ADMITTED_TIME" },
    { WS_NL80211_ATTR_SMPS_MODE,            "NL80211_ATTR_SMPS_MODE" },
    { WS_NL80211_ATTR_OPER_CLASS,           "NL80211_ATTR_OPER_CLASS" },
    { WS_NL80211_ATTR_MAC_MASK,             "NL80211_ATTR_MAC_MASK" },
    { WS_NL80211_ATTR_WIPHY_SELF_MANAGED_REG, "NL80211_ATTR_WIPHY_SELF_MANAGED_REG" },
    { WS_NL80211_ATTR_EXT_FEATURES,         "NL80211_ATTR_EXT_FEATURES" },
    { WS_NL80211_ATTR_SURVEY_RADIO_STATS,   "NL80211_ATTR_SURVEY_RADIO_STATS" },
    { WS_NL80211_ATTR_NETNS_FD,             "NL80211_ATTR_NETNS_FD" },
    { WS_NL80211_ATTR_SCHED_SCAN_DELAY,     "NL80211_ATTR_SCHED_SCAN_DELAY" },
    { WS_NL80211_ATTR_REG_INDOOR,           "NL80211_ATTR_REG_INDOOR" },
    { WS_NL80211_ATTR_MAX_NUM_SCHED_SCAN_PLANS, "NL80211_ATTR_MAX_NUM_SCHED_SCAN_PLANS" },
    { WS_NL80211_ATTR_MAX_SCAN_PLAN_INTERVAL, "NL80211_ATTR_MAX_SCAN_PLAN_INTERVAL" },
    { WS_NL80211_ATTR_MAX_SCAN_PLAN_ITERATIONS, "NL80211_ATTR_MAX_SCAN_PLAN_ITERATIONS" },
    { WS_NL80211_ATTR_SCHED_SCAN_PLANS,     "NL80211_ATTR_SCHED_SCAN_PLANS" },
    { WS_NL80211_ATTR_PBSS,                 "NL80211_ATTR_PBSS" },
    { WS_NL80211_ATTR_BSS_SELECT,           "NL80211_ATTR_BSS_SELECT" },
    { WS_NL80211_ATTR_STA_SUPPORT_P2P_PS,   "NL80211_ATTR_STA_SUPPORT_P2P_PS" },
    { WS_NL80211_ATTR_PAD,                  "NL80211_ATTR_PAD" },
    { WS_NL80211_ATTR_IFTYPE_EXT_CAPA,      "NL80211_ATTR_IFTYPE_EXT_CAPA" },
    { WS_NL80211_ATTR_MU_MIMO_GROUP_DATA,   "NL80211_ATTR_MU_MIMO_GROUP_DATA" },
    { WS_NL80211_ATTR_MU_MIMO_FOLLOW_MAC_ADDR, "NL80211_ATTR_MU_MIMO_FOLLOW_MAC_ADDR" },
    { WS_NL80211_ATTR_SCAN_START_TIME_TSF,  "NL80211_ATTR_SCAN_START_TIME_TSF" },
    { WS_NL80211_ATTR_SCAN_START_TIME_TSF_BSSID, "NL80211_ATTR_SCAN_START_TIME_TSF_BSSID" },
    { WS_NL80211_ATTR_MEASUREMENT_DURATION, "NL80211_ATTR_MEASUREMENT_DURATION" },
    { WS_NL80211_ATTR_MEASUREMENT_DURATION_MANDATORY, "NL80211_ATTR_MEASUREMENT_DURATION_MANDATORY" },
    { WS_NL80211_ATTR_MESH_PEER_AID,        "NL80211_ATTR_MESH_PEER_AID" },
    { WS_NL80211_ATTR_NAN_MASTER_PREF,      "NL80211_ATTR_NAN_MASTER_PREF" },
    { WS_NL80211_ATTR_BANDS,                "NL80211_ATTR_BANDS" },
    { WS_NL80211_ATTR_NAN_FUNC,             "NL80211_ATTR_NAN_FUNC" },
    { WS_NL80211_ATTR_NAN_MATCH,            "NL80211_ATTR_NAN_MATCH" },
    { WS_NL80211_ATTR_FILS_KEK,             "NL80211_ATTR_FILS_KEK" },
    { WS_NL80211_ATTR_FILS_NONCES,          "NL80211_ATTR_FILS_NONCES" },
    { WS_NL80211_ATTR_MULTICAST_TO_UNICAST_ENABLED, "NL80211_ATTR_MULTICAST_TO_UNICAST_ENABLED" },
    { WS_NL80211_ATTR_BSSID,                "NL80211_ATTR_BSSID" },
    { WS_NL80211_ATTR_SCHED_SCAN_RELATIVE_RSSI, "NL80211_ATTR_SCHED_SCAN_RELATIVE_RSSI" },
    { WS_NL80211_ATTR_SCHED_SCAN_RSSI_ADJUST, "NL80211_ATTR_SCHED_SCAN_RSSI_ADJUST" },
    { WS_NL80211_ATTR_TIMEOUT_REASON,       "NL80211_ATTR_TIMEOUT_REASON" },
    { WS_NL80211_ATTR_FILS_ERP_USERNAME,    "NL80211_ATTR_FILS_ERP_USERNAME" },
    { WS_NL80211_ATTR_FILS_ERP_REALM,       "NL80211_ATTR_FILS_ERP_REALM" },
    { WS_NL80211_ATTR_FILS_ERP_NEXT_SEQ_NUM, "NL80211_ATTR_FILS_ERP_NEXT_SEQ_NUM" },
    { WS_NL80211_ATTR_FILS_ERP_RRK,         "NL80211_ATTR_FILS_ERP_RRK" },
    { WS_NL80211_ATTR_FILS_CACHE_ID,        "NL80211_ATTR_FILS_CACHE_ID" },
    { WS_NL80211_ATTR_PMK,                  "NL80211_ATTR_PMK" },
    { WS_NL80211_ATTR_SCHED_SCAN_MULTI,     "NL80211_ATTR_SCHED_SCAN_MULTI" },
    { WS_NL80211_ATTR_SCHED_SCAN_MAX_REQS,  "NL80211_ATTR_SCHED_SCAN_MAX_REQS" },
    { WS_NL80211_ATTR_WANT_1X_4WAY_HS,      "NL80211_ATTR_WANT_1X_4WAY_HS" },
    { WS_NL80211_ATTR_PMKR0_NAME,           "NL80211_ATTR_PMKR0_NAME" },
    { WS_NL80211_ATTR_PORT_AUTHORIZED,      "NL80211_ATTR_PORT_AUTHORIZED" },
    { WS_NL80211_ATTR_EXTERNAL_AUTH_ACTION, "NL80211_ATTR_EXTERNAL_AUTH_ACTION" },
    { WS_NL80211_ATTR_EXTERNAL_AUTH_SUPPORT, "NL80211_ATTR_EXTERNAL_AUTH_SUPPORT" },
    { WS_NL80211_ATTR_NSS,                  "NL80211_ATTR_NSS" },
    { WS_NL80211_ATTR_ACK_SIGNAL,           "NL80211_ATTR_ACK_SIGNAL" },
    { WS_NL80211_ATTR_CONTROL_PORT_OVER_NL80211, "NL80211_ATTR_CONTROL_PORT_OVER_NL80211" },
    { WS_NL80211_ATTR_TXQ_STATS,            "NL80211_ATTR_TXQ_STATS" },
    { WS_NL80211_ATTR_TXQ_LIMIT,            "NL80211_ATTR_TXQ_LIMIT" },
    { WS_NL80211_ATTR_TXQ_MEMORY_LIMIT,     "NL80211_ATTR_TXQ_MEMORY_LIMIT" },
    { WS_NL80211_ATTR_TXQ_QUANTUM,          "NL80211_ATTR_TXQ_QUANTUM" },
    { WS_NL80211_ATTR_HE_CAPABILITY,        "NL80211_ATTR_HE_CAPABILITY" },
    { WS_NL80211_ATTR_FTM_RESPONDER,        "NL80211_ATTR_FTM_RESPONDER" },
    { WS_NL80211_ATTR_FTM_RESPONDER_STATS,  "NL80211_ATTR_FTM_RESPONDER_STATS" },
    { WS_NL80211_ATTR_TIMEOUT,              "NL80211_ATTR_TIMEOUT" },
    { WS_NL80211_ATTR_PEER_MEASUREMENTS,    "NL80211_ATTR_PEER_MEASUREMENTS" },
    { WS_NL80211_ATTR_AIRTIME_WEIGHT,       "NL80211_ATTR_AIRTIME_WEIGHT" },
    { WS_NL80211_ATTR_STA_TX_POWER_SETTING, "NL80211_ATTR_STA_TX_POWER_SETTING" },
    { WS_NL80211_ATTR_STA_TX_POWER,         "NL80211_ATTR_STA_TX_POWER" },
    { WS_NL80211_ATTR_SAE_PASSWORD,         "NL80211_ATTR_SAE_PASSWORD" },
    { WS_NL80211_ATTR_TWT_RESPONDER,        "NL80211_ATTR_TWT_RESPONDER" },
    { WS_NL80211_ATTR_HE_OBSS_PD,           "NL80211_ATTR_HE_OBSS_PD" },
    { WS_NL80211_ATTR_WIPHY_EDMG_CHANNELS,  "NL80211_ATTR_WIPHY_EDMG_CHANNELS" },
    { WS_NL80211_ATTR_WIPHY_EDMG_BW_CONFIG, "NL80211_ATTR_WIPHY_EDMG_BW_CONFIG" },
    { WS_NL80211_ATTR_VLAN_ID,              "NL80211_ATTR_VLAN_ID" },
    { WS_NL80211_ATTR_HE_BSS_COLOR,         "NL80211_ATTR_HE_BSS_COLOR" },
    { WS_NL80211_ATTR_IFTYPE_AKM_SUITES,    "NL80211_ATTR_IFTYPE_AKM_SUITES" },
    { WS_NL80211_ATTR_TID_CONFIG,           "NL80211_ATTR_TID_CONFIG" },
    { WS_NL80211_ATTR_CONTROL_PORT_NO_PREAUTH, "NL80211_ATTR_CONTROL_PORT_NO_PREAUTH" },
    { WS_NL80211_ATTR_PMK_LIFETIME,         "NL80211_ATTR_PMK_LIFETIME" },
    { WS_NL80211_ATTR_PMK_REAUTH_THRESHOLD, "NL80211_ATTR_PMK_REAUTH_THRESHOLD" },
    { WS_NL80211_ATTR_RECEIVE_MULTICAST,    "NL80211_ATTR_RECEIVE_MULTICAST" },
    { WS_NL80211_ATTR_WIPHY_FREQ_OFFSET,    "NL80211_ATTR_WIPHY_FREQ_OFFSET" },
    { WS_NL80211_ATTR_CENTER_FREQ1_OFFSET,  "NL80211_ATTR_CENTER_FREQ1_OFFSET" },
    { WS_NL80211_ATTR_SCAN_FREQ_KHZ,        "NL80211_ATTR_SCAN_FREQ_KHZ" },
    { WS_NL80211_ATTR_HE_6GHZ_CAPABILITY,   "NL80211_ATTR_HE_6GHZ_CAPABILITY" },
    { WS_NL80211_ATTR_FILS_DISCOVERY,       "NL80211_ATTR_FILS_DISCOVERY" },
    { WS_NL80211_ATTR_UNSOL_BCAST_PROBE_RESP, "NL80211_ATTR_UNSOL_BCAST_PROBE_RESP" },
    { WS_NL80211_ATTR_S1G_CAPABILITY,       "NL80211_ATTR_S1G_CAPABILITY" },
    { WS_NL80211_ATTR_S1G_CAPABILITY_MASK,  "NL80211_ATTR_S1G_CAPABILITY_MASK" },
    { WS_NL80211_ATTR_SAE_PWE,              "NL80211_ATTR_SAE_PWE" },
    { WS_NL80211_ATTR_RECONNECT_REQUESTED,  "NL80211_ATTR_RECONNECT_REQUESTED" },
    { WS_NL80211_ATTR_SAR_SPEC,             "NL80211_ATTR_SAR_SPEC" },
    { WS_NL80211_ATTR_DISABLE_HE,           "NL80211_ATTR_DISABLE_HE" },
    { WS_NL80211_ATTR_OBSS_COLOR_BITMAP,    "NL80211_ATTR_OBSS_COLOR_BITMAP" },
    { WS_NL80211_ATTR_COLOR_CHANGE_COUNT,   "NL80211_ATTR_COLOR_CHANGE_COUNT" },
    { WS_NL80211_ATTR_COLOR_CHANGE_COLOR,   "NL80211_ATTR_COLOR_CHANGE_COLOR" },
    { WS_NL80211_ATTR_COLOR_CHANGE_ELEMS,   "NL80211_ATTR_COLOR_CHANGE_ELEMS" },
    { WS_NL80211_ATTR_MBSSID_CONFIG,        "NL80211_ATTR_MBSSID_CONFIG" },
    { WS_NL80211_ATTR_MBSSID_ELEMS,         "NL80211_ATTR_MBSSID_ELEMS" },
    { WS_NL80211_ATTR_RADAR_BACKGROUND,     "NL80211_ATTR_RADAR_BACKGROUND" },
    { WS_NL80211_ATTR_AP_SETTINGS_FLAGS,    "NL80211_ATTR_AP_SETTINGS_FLAGS" },
    { WS_NL80211_ATTR_EHT_CAPABILITY,       "NL80211_ATTR_EHT_CAPABILITY" },
    { WS_NL80211_ATTR_DISABLE_EHT,          "NL80211_ATTR_DISABLE_EHT" },
    { WS_NL80211_ATTR_MLO_LINKS,            "NL80211_ATTR_MLO_LINKS" },
    { WS_NL80211_ATTR_MLO_LINK_ID,          "NL80211_ATTR_MLO_LINK_ID" },
    { WS_NL80211_ATTR_MLD_ADDR,             "NL80211_ATTR_MLD_ADDR" },
    { WS_NL80211_ATTR_MLO_SUPPORT,          "NL80211_ATTR_MLO_SUPPORT" },
    { WS_NL80211_ATTR_MAX_NUM_AKM_SUITES,   "NL80211_ATTR_MAX_NUM_AKM_SUITES" },
    { WS_NL80211_ATTR_EML_CAPABILITY,       "NL80211_ATTR_EML_CAPABILITY" },
    { WS_NL80211_ATTR_MLD_CAPA_AND_OPS,     "NL80211_ATTR_MLD_CAPA_AND_OPS" },
    { WS_NL80211_ATTR_TX_HW_TIMESTAMP,      "NL80211_ATTR_TX_HW_TIMESTAMP" },
    { WS_NL80211_ATTR_RX_HW_TIMESTAMP,      "NL80211_ATTR_RX_HW_TIMESTAMP" },
    { WS_NL80211_ATTR_TD_BITMAP,            "NL80211_ATTR_TD_BITMAP" },
    { WS_NL80211_ATTR_PUNCT_BITMAP,         "NL80211_ATTR_PUNCT_BITMAP" },
    { WS_NL80211_ATTR_MAX_HW_TIMESTAMP_PEERS, "NL80211_ATTR_MAX_HW_TIMESTAMP_PEERS" },
    { WS_NL80211_ATTR_HW_TIMESTAMP_ENABLED, "NL80211_ATTR_HW_TIMESTAMP_ENABLED" },
    { WS_NL80211_ATTR_EMA_RNR_ELEMS,        "NL80211_ATTR_EMA_RNR_ELEMS" },
    { WS_NL80211_ATTR_MLO_LINK_DISABLED,    "NL80211_ATTR_MLO_LINK_DISABLED" },
    { WS_NL80211_ATTR_BSS_DUMP_INCLUDE_USE_DATA, "NL80211_ATTR_BSS_DUMP_INCLUDE_USE_DATA" },
    { WS_NL80211_ATTR_MLO_TTLM_DLINK,       "NL80211_ATTR_MLO_TTLM_DLINK" },
    { WS_NL80211_ATTR_MLO_TTLM_ULINK,       "NL80211_ATTR_MLO_TTLM_ULINK" },
    { WS_NL80211_ATTR_ASSOC_SPP_AMSDU,      "NL80211_ATTR_ASSOC_SPP_AMSDU" },
    { WS_NL80211_ATTR_WIPHY_RADIOS,         "NL80211_ATTR_WIPHY_RADIOS" },
    { WS_NL80211_ATTR_WIPHY_INTERFACE_COMBINATIONS, "NL80211_ATTR_WIPHY_INTERFACE_COMBINATIONS" },
    { WS_NL80211_ATTR_VIF_RADIO_MASK,       "NL80211_ATTR_VIF_RADIO_MASK" },
    { WS_NL80211_ATTR_SUPPORTED_SELECTORS,  "NL80211_ATTR_SUPPORTED_SELECTORS" },
    { WS_NL80211_ATTR_MLO_RECONF_REM_LINKS, "NL80211_ATTR_MLO_RECONF_REM_LINKS" },
    { WS_NL80211_ATTR_EPCS,                 "NL80211_ATTR_EPCS" },
    { WS_NL80211_ATTR_ASSOC_MLD_EXT_CAPA_OPS, "NL80211_ATTR_ASSOC_MLD_EXT_CAPA_OPS" },
    { WS_NL80211_ATTR_WIPHY_RADIO_INDEX,    "NL80211_ATTR_WIPHY_RADIO_INDEX" },
    { WS_NL80211_ATTR_S1G_LONG_BEACON_PERIOD, "NL80211_ATTR_S1G_LONG_BEACON_PERIOD" },
    { WS_NL80211_ATTR_S1G_SHORT_BEACON,     "NL80211_ATTR_S1G_SHORT_BEACON" },
    { WS_NL80211_ATTR_BSS_PARAM,            "NL80211_ATTR_BSS_PARAM" },
    { WS_NL80211_ATTR_NAN_CONFIG,           "NL80211_ATTR_NAN_CONFIG" },
    { WS_NL80211_ATTR_NAN_NEW_CLUSTER,      "NL80211_ATTR_NAN_NEW_CLUSTER" },
    { WS_NL80211_ATTR_NAN_CAPABILITIES,     "NL80211_ATTR_NAN_CAPABILITIES" },
    { WS_NL80211_ATTR_S1G_PRIMARY_2MHZ,     "NL80211_ATTR_S1G_PRIMARY_2MHZ" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_attrs_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_attrs_vals);

static const value_string ws_nl80211_iftype_vals[] = {
    { WS_NL80211_IFTYPE_UNSPECIFIED,        "NL80211_IFTYPE_UNSPECIFIED" },
    { WS_NL80211_IFTYPE_ADHOC,              "NL80211_IFTYPE_ADHOC" },
    { WS_NL80211_IFTYPE_STATION,            "NL80211_IFTYPE_STATION" },
    { WS_NL80211_IFTYPE_AP,                 "NL80211_IFTYPE_AP" },
    { WS_NL80211_IFTYPE_AP_VLAN,            "NL80211_IFTYPE_AP_VLAN" },
    { WS_NL80211_IFTYPE_WDS,                "NL80211_IFTYPE_WDS" },
    { WS_NL80211_IFTYPE_MONITOR,            "NL80211_IFTYPE_MONITOR" },
    { WS_NL80211_IFTYPE_MESH_POINT,         "NL80211_IFTYPE_MESH_POINT" },
    { WS_NL80211_IFTYPE_P2P_CLIENT,         "NL80211_IFTYPE_P2P_CLIENT" },
    { WS_NL80211_IFTYPE_P2P_GO,             "NL80211_IFTYPE_P2P_GO" },
    { WS_NL80211_IFTYPE_P2P_DEVICE,         "NL80211_IFTYPE_P2P_DEVICE" },
    { WS_NL80211_IFTYPE_OCB,                "NL80211_IFTYPE_OCB" },
    { WS_NL80211_IFTYPE_NAN,                "NL80211_IFTYPE_NAN" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_iftype_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_iftype_vals);

static const value_string ws_nl80211_sta_flags_vals[] = {
    { WS___NL80211_STA_FLAG_INVALID,        "__NL80211_STA_FLAG_INVALID" },
    { WS_NL80211_STA_FLAG_AUTHORIZED,       "NL80211_STA_FLAG_AUTHORIZED" },
    { WS_NL80211_STA_FLAG_SHORT_PREAMBLE,   "NL80211_STA_FLAG_SHORT_PREAMBLE" },
    { WS_NL80211_STA_FLAG_WME,              "NL80211_STA_FLAG_WME" },
    { WS_NL80211_STA_FLAG_MFP,              "NL80211_STA_FLAG_MFP" },
    { WS_NL80211_STA_FLAG_AUTHENTICATED,    "NL80211_STA_FLAG_AUTHENTICATED" },
    { WS_NL80211_STA_FLAG_TDLS_PEER,        "NL80211_STA_FLAG_TDLS_PEER" },
    { WS_NL80211_STA_FLAG_ASSOCIATED,       "NL80211_STA_FLAG_ASSOCIATED" },
    { WS_NL80211_STA_FLAG_SPP_AMSDU,        "NL80211_STA_FLAG_SPP_AMSDU" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_sta_flags_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_sta_flags_vals);

static const value_string ws_nl80211_sta_p2p_ps_status_vals[] = {
    { WS_NL80211_P2P_PS_UNSUPPORTED,        "NL80211_P2P_PS_UNSUPPORTED" },
    { WS_NL80211_P2P_PS_SUPPORTED,          "NL80211_P2P_PS_SUPPORTED" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_sta_p2p_ps_status_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_sta_p2p_ps_status_vals);

static const value_string ws_nl80211_he_gi_vals[] = {
    { WS_NL80211_RATE_INFO_HE_GI_0_8,       "NL80211_RATE_INFO_HE_GI_0_8" },
    { WS_NL80211_RATE_INFO_HE_GI_1_6,       "NL80211_RATE_INFO_HE_GI_1_6" },
    { WS_NL80211_RATE_INFO_HE_GI_3_2,       "NL80211_RATE_INFO_HE_GI_3_2" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_he_gi_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_he_gi_vals);

static const value_string ws_nl80211_he_ltf_vals[] = {
    { WS_NL80211_RATE_INFO_HE_1XLTF,        "NL80211_RATE_INFO_HE_1XLTF" },
    { WS_NL80211_RATE_INFO_HE_2XLTF,        "NL80211_RATE_INFO_HE_2XLTF" },
    { WS_NL80211_RATE_INFO_HE_4XLTF,        "NL80211_RATE_INFO_HE_4XLTF" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_he_ltf_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_he_ltf_vals);

static const value_string ws_nl80211_he_ru_alloc_vals[] = {
    { WS_NL80211_RATE_INFO_HE_RU_ALLOC_26,  "NL80211_RATE_INFO_HE_RU_ALLOC_26" },
    { WS_NL80211_RATE_INFO_HE_RU_ALLOC_52,  "NL80211_RATE_INFO_HE_RU_ALLOC_52" },
    { WS_NL80211_RATE_INFO_HE_RU_ALLOC_106, "NL80211_RATE_INFO_HE_RU_ALLOC_106" },
    { WS_NL80211_RATE_INFO_HE_RU_ALLOC_242, "NL80211_RATE_INFO_HE_RU_ALLOC_242" },
    { WS_NL80211_RATE_INFO_HE_RU_ALLOC_484, "NL80211_RATE_INFO_HE_RU_ALLOC_484" },
    { WS_NL80211_RATE_INFO_HE_RU_ALLOC_996, "NL80211_RATE_INFO_HE_RU_ALLOC_996" },
    { WS_NL80211_RATE_INFO_HE_RU_ALLOC_2x996, "NL80211_RATE_INFO_HE_RU_ALLOC_2x996" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_he_ru_alloc_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_he_ru_alloc_vals);

static const value_string ws_nl80211_eht_gi_vals[] = {
    { WS_NL80211_RATE_INFO_EHT_GI_0_8,      "NL80211_RATE_INFO_EHT_GI_0_8" },
    { WS_NL80211_RATE_INFO_EHT_GI_1_6,      "NL80211_RATE_INFO_EHT_GI_1_6" },
    { WS_NL80211_RATE_INFO_EHT_GI_3_2,      "NL80211_RATE_INFO_EHT_GI_3_2" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_eht_gi_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_eht_gi_vals);

static const value_string ws_nl80211_eht_ru_alloc_vals[] = {
    { WS_NL80211_RATE_INFO_EHT_RU_ALLOC_26, "NL80211_RATE_INFO_EHT_RU_ALLOC_26" },
    { WS_NL80211_RATE_INFO_EHT_RU_ALLOC_52, "NL80211_RATE_INFO_EHT_RU_ALLOC_52" },
    { WS_NL80211_RATE_INFO_EHT_RU_ALLOC_52P26, "NL80211_RATE_INFO_EHT_RU_ALLOC_52P26" },
    { WS_NL80211_RATE_INFO_EHT_RU_ALLOC_106, "NL80211_RATE_INFO_EHT_RU_ALLOC_106" },
    { WS_NL80211_RATE_INFO_EHT_RU_ALLOC_106P26, "NL80211_RATE_INFO_EHT_RU_ALLOC_106P26" },
    { WS_NL80211_RATE_INFO_EHT_RU_ALLOC_242, "NL80211_RATE_INFO_EHT_RU_ALLOC_242" },
    { WS_NL80211_RATE_INFO_EHT_RU_ALLOC_484, "NL80211_RATE_INFO_EHT_RU_ALLOC_484" },
    { WS_NL80211_RATE_INFO_EHT_RU_ALLOC_484P242, "NL80211_RATE_INFO_EHT_RU_ALLOC_484P242" },
    { WS_NL80211_RATE_INFO_EHT_RU_ALLOC_996, "NL80211_RATE_INFO_EHT_RU_ALLOC_996" },
    { WS_NL80211_RATE_INFO_EHT_RU_ALLOC_996P484, "NL80211_RATE_INFO_EHT_RU_ALLOC_996P484" },
    { WS_NL80211_RATE_INFO_EHT_RU_ALLOC_996P484P242, "NL80211_RATE_INFO_EHT_RU_ALLOC_996P484P242" },
    { WS_NL80211_RATE_INFO_EHT_RU_ALLOC_2x996, "NL80211_RATE_INFO_EHT_RU_ALLOC_2x996" },
    { WS_NL80211_RATE_INFO_EHT_RU_ALLOC_2x996P484, "NL80211_RATE_INFO_EHT_RU_ALLOC_2x996P484" },
    { WS_NL80211_RATE_INFO_EHT_RU_ALLOC_3x996, "NL80211_RATE_INFO_EHT_RU_ALLOC_3x996" },
    { WS_NL80211_RATE_INFO_EHT_RU_ALLOC_3x996P484, "NL80211_RATE_INFO_EHT_RU_ALLOC_3x996P484" },
    { WS_NL80211_RATE_INFO_EHT_RU_ALLOC_4x996, "NL80211_RATE_INFO_EHT_RU_ALLOC_4x996" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_eht_ru_alloc_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_eht_ru_alloc_vals);

static const value_string ws_nl80211_rate_info_vals[] = {
    { WS___NL80211_RATE_INFO_INVALID,       "__NL80211_RATE_INFO_INVALID" },
    { WS_NL80211_RATE_INFO_BITRATE,         "NL80211_RATE_INFO_BITRATE" },
    { WS_NL80211_RATE_INFO_MCS,             "NL80211_RATE_INFO_MCS" },
    { WS_NL80211_RATE_INFO_40_MHZ_WIDTH,    "NL80211_RATE_INFO_40_MHZ_WIDTH" },
    { WS_NL80211_RATE_INFO_SHORT_GI,        "NL80211_RATE_INFO_SHORT_GI" },
    { WS_NL80211_RATE_INFO_BITRATE32,       "NL80211_RATE_INFO_BITRATE32" },
    { WS_NL80211_RATE_INFO_VHT_MCS,         "NL80211_RATE_INFO_VHT_MCS" },
    { WS_NL80211_RATE_INFO_VHT_NSS,         "NL80211_RATE_INFO_VHT_NSS" },
    { WS_NL80211_RATE_INFO_80_MHZ_WIDTH,    "NL80211_RATE_INFO_80_MHZ_WIDTH" },
    { WS_NL80211_RATE_INFO_80P80_MHZ_WIDTH, "NL80211_RATE_INFO_80P80_MHZ_WIDTH" },
    { WS_NL80211_RATE_INFO_160_MHZ_WIDTH,   "NL80211_RATE_INFO_160_MHZ_WIDTH" },
    { WS_NL80211_RATE_INFO_10_MHZ_WIDTH,    "NL80211_RATE_INFO_10_MHZ_WIDTH" },
    { WS_NL80211_RATE_INFO_5_MHZ_WIDTH,     "NL80211_RATE_INFO_5_MHZ_WIDTH" },
    { WS_NL80211_RATE_INFO_HE_MCS,          "NL80211_RATE_INFO_HE_MCS" },
    { WS_NL80211_RATE_INFO_HE_NSS,          "NL80211_RATE_INFO_HE_NSS" },
    { WS_NL80211_RATE_INFO_HE_GI,           "NL80211_RATE_INFO_HE_GI" },
    { WS_NL80211_RATE_INFO_HE_DCM,          "NL80211_RATE_INFO_HE_DCM" },
    { WS_NL80211_RATE_INFO_HE_RU_ALLOC,     "NL80211_RATE_INFO_HE_RU_ALLOC" },
    { WS_NL80211_RATE_INFO_320_MHZ_WIDTH,   "NL80211_RATE_INFO_320_MHZ_WIDTH" },
    { WS_NL80211_RATE_INFO_EHT_MCS,         "NL80211_RATE_INFO_EHT_MCS" },
    { WS_NL80211_RATE_INFO_EHT_NSS,         "NL80211_RATE_INFO_EHT_NSS" },
    { WS_NL80211_RATE_INFO_EHT_GI,          "NL80211_RATE_INFO_EHT_GI" },
    { WS_NL80211_RATE_INFO_EHT_RU_ALLOC,    "NL80211_RATE_INFO_EHT_RU_ALLOC" },
    { WS_NL80211_RATE_INFO_S1G_MCS,         "NL80211_RATE_INFO_S1G_MCS" },
    { WS_NL80211_RATE_INFO_S1G_NSS,         "NL80211_RATE_INFO_S1G_NSS" },
    { WS_NL80211_RATE_INFO_1_MHZ_WIDTH,     "NL80211_RATE_INFO_1_MHZ_WIDTH" },
    { WS_NL80211_RATE_INFO_2_MHZ_WIDTH,     "NL80211_RATE_INFO_2_MHZ_WIDTH" },
    { WS_NL80211_RATE_INFO_4_MHZ_WIDTH,     "NL80211_RATE_INFO_4_MHZ_WIDTH" },
    { WS_NL80211_RATE_INFO_8_MHZ_WIDTH,     "NL80211_RATE_INFO_8_MHZ_WIDTH" },
    { WS_NL80211_RATE_INFO_16_MHZ_WIDTH,    "NL80211_RATE_INFO_16_MHZ_WIDTH" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_rate_info_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_rate_info_vals);

static const value_string ws_nl80211_sta_bss_param_vals[] = {
    { WS___NL80211_STA_BSS_PARAM_INVALID,   "__NL80211_STA_BSS_PARAM_INVALID" },
    { WS_NL80211_STA_BSS_PARAM_CTS_PROT,    "NL80211_STA_BSS_PARAM_CTS_PROT" },
    { WS_NL80211_STA_BSS_PARAM_SHORT_PREAMBLE, "NL80211_STA_BSS_PARAM_SHORT_PREAMBLE" },
    { WS_NL80211_STA_BSS_PARAM_SHORT_SLOT_TIME, "NL80211_STA_BSS_PARAM_SHORT_SLOT_TIME" },
    { WS_NL80211_STA_BSS_PARAM_DTIM_PERIOD, "NL80211_STA_BSS_PARAM_DTIM_PERIOD" },
    { WS_NL80211_STA_BSS_PARAM_BEACON_INTERVAL, "NL80211_STA_BSS_PARAM_BEACON_INTERVAL" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_sta_bss_param_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_sta_bss_param_vals);

static const value_string ws_nl80211_sta_info_vals[] = {
    { WS___NL80211_STA_INFO_INVALID,        "__NL80211_STA_INFO_INVALID" },
    { WS_NL80211_STA_INFO_INACTIVE_TIME,    "NL80211_STA_INFO_INACTIVE_TIME" },
    { WS_NL80211_STA_INFO_RX_BYTES,         "NL80211_STA_INFO_RX_BYTES" },
    { WS_NL80211_STA_INFO_TX_BYTES,         "NL80211_STA_INFO_TX_BYTES" },
    { WS_NL80211_STA_INFO_LLID,             "NL80211_STA_INFO_LLID" },
    { WS_NL80211_STA_INFO_PLID,             "NL80211_STA_INFO_PLID" },
    { WS_NL80211_STA_INFO_PLINK_STATE,      "NL80211_STA_INFO_PLINK_STATE" },
    { WS_NL80211_STA_INFO_SIGNAL,           "NL80211_STA_INFO_SIGNAL" },
    { WS_NL80211_STA_INFO_TX_BITRATE,       "NL80211_STA_INFO_TX_BITRATE" },
    { WS_NL80211_STA_INFO_RX_PACKETS,       "NL80211_STA_INFO_RX_PACKETS" },
    { WS_NL80211_STA_INFO_TX_PACKETS,       "NL80211_STA_INFO_TX_PACKETS" },
    { WS_NL80211_STA_INFO_TX_RETRIES,       "NL80211_STA_INFO_TX_RETRIES" },
    { WS_NL80211_STA_INFO_TX_FAILED,        "NL80211_STA_INFO_TX_FAILED" },
    { WS_NL80211_STA_INFO_SIGNAL_AVG,       "NL80211_STA_INFO_SIGNAL_AVG" },
    { WS_NL80211_STA_INFO_RX_BITRATE,       "NL80211_STA_INFO_RX_BITRATE" },
    { WS_NL80211_STA_INFO_BSS_PARAM,        "NL80211_STA_INFO_BSS_PARAM" },
    { WS_NL80211_STA_INFO_CONNECTED_TIME,   "NL80211_STA_INFO_CONNECTED_TIME" },
    { WS_NL80211_STA_INFO_STA_FLAGS,        "NL80211_STA_INFO_STA_FLAGS" },
    { WS_NL80211_STA_INFO_BEACON_LOSS,      "NL80211_STA_INFO_BEACON_LOSS" },
    { WS_NL80211_STA_INFO_T_OFFSET,         "NL80211_STA_INFO_T_OFFSET" },
    { WS_NL80211_STA_INFO_LOCAL_PM,         "NL80211_STA_INFO_LOCAL_PM" },
    { WS_NL80211_STA_INFO_PEER_PM,          "NL80211_STA_INFO_PEER_PM" },
    { WS_NL80211_STA_INFO_NONPEER_PM,       "NL80211_STA_INFO_NONPEER_PM" },
    { WS_NL80211_STA_INFO_RX_BYTES64,       "NL80211_STA_INFO_RX_BYTES64" },
    { WS_NL80211_STA_INFO_TX_BYTES64,       "NL80211_STA_INFO_TX_BYTES64" },
    { WS_NL80211_STA_INFO_CHAIN_SIGNAL,     "NL80211_STA_INFO_CHAIN_SIGNAL" },
    { WS_NL80211_STA_INFO_CHAIN_SIGNAL_AVG, "NL80211_STA_INFO_CHAIN_SIGNAL_AVG" },
    { WS_NL80211_STA_INFO_EXPECTED_THROUGHPUT, "NL80211_STA_INFO_EXPECTED_THROUGHPUT" },
    { WS_NL80211_STA_INFO_RX_DROP_MISC,     "NL80211_STA_INFO_RX_DROP_MISC" },
    { WS_NL80211_STA_INFO_BEACON_RX,        "NL80211_STA_INFO_BEACON_RX" },
    { WS_NL80211_STA_INFO_BEACON_SIGNAL_AVG, "NL80211_STA_INFO_BEACON_SIGNAL_AVG" },
    { WS_NL80211_STA_INFO_TID_STATS,        "NL80211_STA_INFO_TID_STATS" },
    { WS_NL80211_STA_INFO_RX_DURATION,      "NL80211_STA_INFO_RX_DURATION" },
    { WS_NL80211_STA_INFO_PAD,              "NL80211_STA_INFO_PAD" },
    { WS_NL80211_STA_INFO_ACK_SIGNAL,       "NL80211_STA_INFO_ACK_SIGNAL" },
    { WS_NL80211_STA_INFO_ACK_SIGNAL_AVG,   "NL80211_STA_INFO_ACK_SIGNAL_AVG" },
    { WS_NL80211_STA_INFO_RX_MPDUS,         "NL80211_STA_INFO_RX_MPDUS" },
    { WS_NL80211_STA_INFO_FCS_ERROR_COUNT,  "NL80211_STA_INFO_FCS_ERROR_COUNT" },
    { WS_NL80211_STA_INFO_CONNECTED_TO_GATE, "NL80211_STA_INFO_CONNECTED_TO_GATE" },
    { WS_NL80211_STA_INFO_TX_DURATION,      "NL80211_STA_INFO_TX_DURATION" },
    { WS_NL80211_STA_INFO_AIRTIME_WEIGHT,   "NL80211_STA_INFO_AIRTIME_WEIGHT" },
    { WS_NL80211_STA_INFO_AIRTIME_LINK_METRIC, "NL80211_STA_INFO_AIRTIME_LINK_METRIC" },
    { WS_NL80211_STA_INFO_ASSOC_AT_BOOTTIME, "NL80211_STA_INFO_ASSOC_AT_BOOTTIME" },
    { WS_NL80211_STA_INFO_CONNECTED_TO_AS,  "NL80211_STA_INFO_CONNECTED_TO_AS" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_sta_info_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_sta_info_vals);

static const value_string ws_nl80211_tid_stats_vals[] = {
    { WS___NL80211_TID_STATS_INVALID,       "__NL80211_TID_STATS_INVALID" },
    { WS_NL80211_TID_STATS_RX_MSDU,         "NL80211_TID_STATS_RX_MSDU" },
    { WS_NL80211_TID_STATS_TX_MSDU,         "NL80211_TID_STATS_TX_MSDU" },
    { WS_NL80211_TID_STATS_TX_MSDU_RETRIES, "NL80211_TID_STATS_TX_MSDU_RETRIES" },
    { WS_NL80211_TID_STATS_TX_MSDU_FAILED,  "NL80211_TID_STATS_TX_MSDU_FAILED" },
    { WS_NL80211_TID_STATS_PAD,             "NL80211_TID_STATS_PAD" },
    { WS_NL80211_TID_STATS_TXQ_STATS,       "NL80211_TID_STATS_TXQ_STATS" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_tid_stats_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_tid_stats_vals);

static const value_string ws_nl80211_txq_stats_vals[] = {
    { WS___NL80211_TXQ_STATS_INVALID,       "__NL80211_TXQ_STATS_INVALID" },
    { WS_NL80211_TXQ_STATS_BACKLOG_BYTES,   "NL80211_TXQ_STATS_BACKLOG_BYTES" },
    { WS_NL80211_TXQ_STATS_BACKLOG_PACKETS, "NL80211_TXQ_STATS_BACKLOG_PACKETS" },
    { WS_NL80211_TXQ_STATS_FLOWS,           "NL80211_TXQ_STATS_FLOWS" },
    { WS_NL80211_TXQ_STATS_DROPS,           "NL80211_TXQ_STATS_DROPS" },
    { WS_NL80211_TXQ_STATS_ECN_MARKS,       "NL80211_TXQ_STATS_ECN_MARKS" },
    { WS_NL80211_TXQ_STATS_OVERLIMIT,       "NL80211_TXQ_STATS_OVERLIMIT" },
    { WS_NL80211_TXQ_STATS_OVERMEMORY,      "NL80211_TXQ_STATS_OVERMEMORY" },
    { WS_NL80211_TXQ_STATS_COLLISIONS,      "NL80211_TXQ_STATS_COLLISIONS" },
    { WS_NL80211_TXQ_STATS_TX_BYTES,        "NL80211_TXQ_STATS_TX_BYTES" },
    { WS_NL80211_TXQ_STATS_TX_PACKETS,      "NL80211_TXQ_STATS_TX_PACKETS" },
    { WS_NL80211_TXQ_STATS_MAX_FLOWS,       "NL80211_TXQ_STATS_MAX_FLOWS" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_txq_stats_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_txq_stats_vals);

static const value_string ws_nl80211_mpath_flags_vals[] = {
    { WS_NL80211_MPATH_FLAG_ACTIVE,         "NL80211_MPATH_FLAG_ACTIVE" },
    { WS_NL80211_MPATH_FLAG_RESOLVING,      "NL80211_MPATH_FLAG_RESOLVING" },
    { WS_NL80211_MPATH_FLAG_SN_VALID,       "NL80211_MPATH_FLAG_SN_VALID" },
    { WS_NL80211_MPATH_FLAG_FIXED,          "NL80211_MPATH_FLAG_FIXED" },
    { WS_NL80211_MPATH_FLAG_RESOLVED,       "NL80211_MPATH_FLAG_RESOLVED" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_mpath_flags_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_mpath_flags_vals);

static const value_string ws_nl80211_mpath_info_vals[] = {
    { WS___NL80211_MPATH_INFO_INVALID,      "__NL80211_MPATH_INFO_INVALID" },
    { WS_NL80211_MPATH_INFO_FRAME_QLEN,     "NL80211_MPATH_INFO_FRAME_QLEN" },
    { WS_NL80211_MPATH_INFO_SN,             "NL80211_MPATH_INFO_SN" },
    { WS_NL80211_MPATH_INFO_METRIC,         "NL80211_MPATH_INFO_METRIC" },
    { WS_NL80211_MPATH_INFO_EXPTIME,        "NL80211_MPATH_INFO_EXPTIME" },
    { WS_NL80211_MPATH_INFO_FLAGS,          "NL80211_MPATH_INFO_FLAGS" },
    { WS_NL80211_MPATH_INFO_DISCOVERY_TIMEOUT, "NL80211_MPATH_INFO_DISCOVERY_TIMEOUT" },
    { WS_NL80211_MPATH_INFO_DISCOVERY_RETRIES, "NL80211_MPATH_INFO_DISCOVERY_RETRIES" },
    { WS_NL80211_MPATH_INFO_HOP_COUNT,      "NL80211_MPATH_INFO_HOP_COUNT" },
    { WS_NL80211_MPATH_INFO_PATH_CHANGE,    "NL80211_MPATH_INFO_PATH_CHANGE" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_mpath_info_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_mpath_info_vals);

static const value_string ws_nl80211_band_iftype_attr_vals[] = {
    { WS___NL80211_BAND_IFTYPE_ATTR_INVALID, "__NL80211_BAND_IFTYPE_ATTR_INVALID" },
    { WS_NL80211_BAND_IFTYPE_ATTR_IFTYPES,  "NL80211_BAND_IFTYPE_ATTR_IFTYPES" },
    { WS_NL80211_BAND_IFTYPE_ATTR_HE_CAP_MAC, "NL80211_BAND_IFTYPE_ATTR_HE_CAP_MAC" },
    { WS_NL80211_BAND_IFTYPE_ATTR_HE_CAP_PHY, "NL80211_BAND_IFTYPE_ATTR_HE_CAP_PHY" },
    { WS_NL80211_BAND_IFTYPE_ATTR_HE_CAP_MCS_SET, "NL80211_BAND_IFTYPE_ATTR_HE_CAP_MCS_SET" },
    { WS_NL80211_BAND_IFTYPE_ATTR_HE_CAP_PPE, "NL80211_BAND_IFTYPE_ATTR_HE_CAP_PPE" },
    { WS_NL80211_BAND_IFTYPE_ATTR_HE_6GHZ_CAPA, "NL80211_BAND_IFTYPE_ATTR_HE_6GHZ_CAPA" },
    { WS_NL80211_BAND_IFTYPE_ATTR_VENDOR_ELEMS, "NL80211_BAND_IFTYPE_ATTR_VENDOR_ELEMS" },
    { WS_NL80211_BAND_IFTYPE_ATTR_EHT_CAP_MAC, "NL80211_BAND_IFTYPE_ATTR_EHT_CAP_MAC" },
    { WS_NL80211_BAND_IFTYPE_ATTR_EHT_CAP_PHY, "NL80211_BAND_IFTYPE_ATTR_EHT_CAP_PHY" },
    { WS_NL80211_BAND_IFTYPE_ATTR_EHT_CAP_MCS_SET, "NL80211_BAND_IFTYPE_ATTR_EHT_CAP_MCS_SET" },
    { WS_NL80211_BAND_IFTYPE_ATTR_EHT_CAP_PPE, "NL80211_BAND_IFTYPE_ATTR_EHT_CAP_PPE" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_band_iftype_attr_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_band_iftype_attr_vals);

static const value_string ws_nl80211_band_attr_vals[] = {
    { WS___NL80211_BAND_ATTR_INVALID,       "__NL80211_BAND_ATTR_INVALID" },
    { WS_NL80211_BAND_ATTR_FREQS,           "NL80211_BAND_ATTR_FREQS" },
    { WS_NL80211_BAND_ATTR_RATES,           "NL80211_BAND_ATTR_RATES" },
    { WS_NL80211_BAND_ATTR_HT_MCS_SET,      "NL80211_BAND_ATTR_HT_MCS_SET" },
    { WS_NL80211_BAND_ATTR_HT_CAPA,         "NL80211_BAND_ATTR_HT_CAPA" },
    { WS_NL80211_BAND_ATTR_HT_AMPDU_FACTOR, "NL80211_BAND_ATTR_HT_AMPDU_FACTOR" },
    { WS_NL80211_BAND_ATTR_HT_AMPDU_DENSITY, "NL80211_BAND_ATTR_HT_AMPDU_DENSITY" },
    { WS_NL80211_BAND_ATTR_VHT_MCS_SET,     "NL80211_BAND_ATTR_VHT_MCS_SET" },
    { WS_NL80211_BAND_ATTR_VHT_CAPA,        "NL80211_BAND_ATTR_VHT_CAPA" },
    { WS_NL80211_BAND_ATTR_IFTYPE_DATA,     "NL80211_BAND_ATTR_IFTYPE_DATA" },
    { WS_NL80211_BAND_ATTR_EDMG_CHANNELS,   "NL80211_BAND_ATTR_EDMG_CHANNELS" },
    { WS_NL80211_BAND_ATTR_EDMG_BW_CONFIG,  "NL80211_BAND_ATTR_EDMG_BW_CONFIG" },
    { WS_NL80211_BAND_ATTR_S1G_MCS_NSS_SET, "NL80211_BAND_ATTR_S1G_MCS_NSS_SET" },
    { WS_NL80211_BAND_ATTR_S1G_CAPA,        "NL80211_BAND_ATTR_S1G_CAPA" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_band_attr_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_band_attr_vals);

static const value_string ws_nl80211_wmm_rule_vals[] = {
    { WS___NL80211_WMMR_INVALID,            "__NL80211_WMMR_INVALID" },
    { WS_NL80211_WMMR_CW_MIN,               "NL80211_WMMR_CW_MIN" },
    { WS_NL80211_WMMR_CW_MAX,               "NL80211_WMMR_CW_MAX" },
    { WS_NL80211_WMMR_AIFSN,                "NL80211_WMMR_AIFSN" },
    { WS_NL80211_WMMR_TXOP,                 "NL80211_WMMR_TXOP" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_wmm_rule_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_wmm_rule_vals);

static const value_string ws_nl80211_frequency_attr_vals[] = {
    { WS___NL80211_FREQUENCY_ATTR_INVALID,  "__NL80211_FREQUENCY_ATTR_INVALID" },
    { WS_NL80211_FREQUENCY_ATTR_FREQ,       "NL80211_FREQUENCY_ATTR_FREQ" },
    { WS_NL80211_FREQUENCY_ATTR_DISABLED,   "NL80211_FREQUENCY_ATTR_DISABLED" },
    { WS_NL80211_FREQUENCY_ATTR_NO_IR,      "NL80211_FREQUENCY_ATTR_NO_IR" },
    { WS___NL80211_FREQUENCY_ATTR_NO_IBSS,  "__NL80211_FREQUENCY_ATTR_NO_IBSS" },
    { WS_NL80211_FREQUENCY_ATTR_RADAR,      "NL80211_FREQUENCY_ATTR_RADAR" },
    { WS_NL80211_FREQUENCY_ATTR_MAX_TX_POWER, "NL80211_FREQUENCY_ATTR_MAX_TX_POWER" },
    { WS_NL80211_FREQUENCY_ATTR_DFS_STATE,  "NL80211_FREQUENCY_ATTR_DFS_STATE" },
    { WS_NL80211_FREQUENCY_ATTR_DFS_TIME,   "NL80211_FREQUENCY_ATTR_DFS_TIME" },
    { WS_NL80211_FREQUENCY_ATTR_NO_HT40_MINUS, "NL80211_FREQUENCY_ATTR_NO_HT40_MINUS" },
    { WS_NL80211_FREQUENCY_ATTR_NO_HT40_PLUS, "NL80211_FREQUENCY_ATTR_NO_HT40_PLUS" },
    { WS_NL80211_FREQUENCY_ATTR_NO_80MHZ,   "NL80211_FREQUENCY_ATTR_NO_80MHZ" },
    { WS_NL80211_FREQUENCY_ATTR_NO_160MHZ,  "NL80211_FREQUENCY_ATTR_NO_160MHZ" },
    { WS_NL80211_FREQUENCY_ATTR_DFS_CAC_TIME, "NL80211_FREQUENCY_ATTR_DFS_CAC_TIME" },
    { WS_NL80211_FREQUENCY_ATTR_INDOOR_ONLY, "NL80211_FREQUENCY_ATTR_INDOOR_ONLY" },
    { WS_NL80211_FREQUENCY_ATTR_IR_CONCURRENT, "NL80211_FREQUENCY_ATTR_IR_CONCURRENT" },
    { WS_NL80211_FREQUENCY_ATTR_NO_20MHZ,   "NL80211_FREQUENCY_ATTR_NO_20MHZ" },
    { WS_NL80211_FREQUENCY_ATTR_NO_10MHZ,   "NL80211_FREQUENCY_ATTR_NO_10MHZ" },
    { WS_NL80211_FREQUENCY_ATTR_WMM,        "NL80211_FREQUENCY_ATTR_WMM" },
    { WS_NL80211_FREQUENCY_ATTR_NO_HE,      "NL80211_FREQUENCY_ATTR_NO_HE" },
    { WS_NL80211_FREQUENCY_ATTR_OFFSET,     "NL80211_FREQUENCY_ATTR_OFFSET" },
    { WS_NL80211_FREQUENCY_ATTR_1MHZ,       "NL80211_FREQUENCY_ATTR_1MHZ" },
    { WS_NL80211_FREQUENCY_ATTR_2MHZ,       "NL80211_FREQUENCY_ATTR_2MHZ" },
    { WS_NL80211_FREQUENCY_ATTR_4MHZ,       "NL80211_FREQUENCY_ATTR_4MHZ" },
    { WS_NL80211_FREQUENCY_ATTR_8MHZ,       "NL80211_FREQUENCY_ATTR_8MHZ" },
    { WS_NL80211_FREQUENCY_ATTR_16MHZ,      "NL80211_FREQUENCY_ATTR_16MHZ" },
    { WS_NL80211_FREQUENCY_ATTR_NO_320MHZ,  "NL80211_FREQUENCY_ATTR_NO_320MHZ" },
    { WS_NL80211_FREQUENCY_ATTR_NO_EHT,     "NL80211_FREQUENCY_ATTR_NO_EHT" },
    { WS_NL80211_FREQUENCY_ATTR_PSD,        "NL80211_FREQUENCY_ATTR_PSD" },
    { WS_NL80211_FREQUENCY_ATTR_DFS_CONCURRENT, "NL80211_FREQUENCY_ATTR_DFS_CONCURRENT" },
    { WS_NL80211_FREQUENCY_ATTR_NO_6GHZ_VLP_CLIENT, "NL80211_FREQUENCY_ATTR_NO_6GHZ_VLP_CLIENT" },
    { WS_NL80211_FREQUENCY_ATTR_NO_6GHZ_AFC_CLIENT, "NL80211_FREQUENCY_ATTR_NO_6GHZ_AFC_CLIENT" },
    { WS_NL80211_FREQUENCY_ATTR_CAN_MONITOR, "NL80211_FREQUENCY_ATTR_CAN_MONITOR" },
    { WS_NL80211_FREQUENCY_ATTR_ALLOW_6GHZ_VLP_AP, "NL80211_FREQUENCY_ATTR_ALLOW_6GHZ_VLP_AP" },
    { WS_NL80211_FREQUENCY_ATTR_ALLOW_20MHZ_ACTIVITY, "NL80211_FREQUENCY_ATTR_ALLOW_20MHZ_ACTIVITY" },
    { WS_NL80211_FREQUENCY_ATTR_NO_4MHZ,    "NL80211_FREQUENCY_ATTR_NO_4MHZ" },
    { WS_NL80211_FREQUENCY_ATTR_NO_8MHZ,    "NL80211_FREQUENCY_ATTR_NO_8MHZ" },
    { WS_NL80211_FREQUENCY_ATTR_NO_16MHZ,   "NL80211_FREQUENCY_ATTR_NO_16MHZ" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_frequency_attr_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_frequency_attr_vals);

static const value_string ws_nl80211_bitrate_attr_vals[] = {
    { WS___NL80211_BITRATE_ATTR_INVALID,    "__NL80211_BITRATE_ATTR_INVALID" },
    { WS_NL80211_BITRATE_ATTR_RATE,         "NL80211_BITRATE_ATTR_RATE" },
    { WS_NL80211_BITRATE_ATTR_2GHZ_SHORTPREAMBLE, "NL80211_BITRATE_ATTR_2GHZ_SHORTPREAMBLE" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_bitrate_attr_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_bitrate_attr_vals);

static const value_string ws_nl80211_reg_initiator_vals[] = {
    { WS_NL80211_REGDOM_SET_BY_CORE,        "NL80211_REGDOM_SET_BY_CORE" },
    { WS_NL80211_REGDOM_SET_BY_USER,        "NL80211_REGDOM_SET_BY_USER" },
    { WS_NL80211_REGDOM_SET_BY_DRIVER,      "NL80211_REGDOM_SET_BY_DRIVER" },
    { WS_NL80211_REGDOM_SET_BY_COUNTRY_IE,  "NL80211_REGDOM_SET_BY_COUNTRY_IE" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_reg_initiator_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_reg_initiator_vals);

static const value_string ws_nl80211_reg_type_vals[] = {
    { WS_NL80211_REGDOM_TYPE_COUNTRY,       "NL80211_REGDOM_TYPE_COUNTRY" },
    { WS_NL80211_REGDOM_TYPE_WORLD,         "NL80211_REGDOM_TYPE_WORLD" },
    { WS_NL80211_REGDOM_TYPE_CUSTOM_WORLD,  "NL80211_REGDOM_TYPE_CUSTOM_WORLD" },
    { WS_NL80211_REGDOM_TYPE_INTERSECTION,  "NL80211_REGDOM_TYPE_INTERSECTION" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_reg_type_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_reg_type_vals);

static const value_string ws_nl80211_reg_rule_attr_vals[] = {
    { WS___NL80211_REG_RULE_ATTR_INVALID,   "__NL80211_REG_RULE_ATTR_INVALID" },
    { WS_NL80211_ATTR_REG_RULE_FLAGS,       "NL80211_ATTR_REG_RULE_FLAGS" },
    { WS_NL80211_ATTR_FREQ_RANGE_START,     "NL80211_ATTR_FREQ_RANGE_START" },
    { WS_NL80211_ATTR_FREQ_RANGE_END,       "NL80211_ATTR_FREQ_RANGE_END" },
    { WS_NL80211_ATTR_FREQ_RANGE_MAX_BW,    "NL80211_ATTR_FREQ_RANGE_MAX_BW" },
    { WS_NL80211_ATTR_POWER_RULE_MAX_ANT_GAIN, "NL80211_ATTR_POWER_RULE_MAX_ANT_GAIN" },
    { WS_NL80211_ATTR_POWER_RULE_MAX_EIRP,  "NL80211_ATTR_POWER_RULE_MAX_EIRP" },
    { WS_NL80211_ATTR_DFS_CAC_TIME,         "NL80211_ATTR_DFS_CAC_TIME" },
    { WS_NL80211_ATTR_POWER_RULE_PSD,       "NL80211_ATTR_POWER_RULE_PSD" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_reg_rule_attr_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_reg_rule_attr_vals);

static const value_string ws_nl80211_sched_scan_match_attr_vals[] = {
    { WS___NL80211_SCHED_SCAN_MATCH_ATTR_INVALID, "__NL80211_SCHED_SCAN_MATCH_ATTR_INVALID" },
    { WS_NL80211_SCHED_SCAN_MATCH_ATTR_SSID, "NL80211_SCHED_SCAN_MATCH_ATTR_SSID" },
    { WS_NL80211_SCHED_SCAN_MATCH_ATTR_RSSI, "NL80211_SCHED_SCAN_MATCH_ATTR_RSSI" },
    { WS_NL80211_SCHED_SCAN_MATCH_ATTR_RELATIVE_RSSI, "NL80211_SCHED_SCAN_MATCH_ATTR_RELATIVE_RSSI" },
    { WS_NL80211_SCHED_SCAN_MATCH_ATTR_RSSI_ADJUST, "NL80211_SCHED_SCAN_MATCH_ATTR_RSSI_ADJUST" },
    { WS_NL80211_SCHED_SCAN_MATCH_ATTR_BSSID, "NL80211_SCHED_SCAN_MATCH_ATTR_BSSID" },
    { WS_NL80211_SCHED_SCAN_MATCH_PER_BAND_RSSI, "NL80211_SCHED_SCAN_MATCH_PER_BAND_RSSI" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_sched_scan_match_attr_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_sched_scan_match_attr_vals);

static const value_string ws_nl80211_reg_rule_flags_vals[] = {
    { WS_NL80211_RRF_NO_OFDM,               "NL80211_RRF_NO_OFDM" },
    { WS_NL80211_RRF_NO_CCK,                "NL80211_RRF_NO_CCK" },
    { WS_NL80211_RRF_NO_INDOOR,             "NL80211_RRF_NO_INDOOR" },
    { WS_NL80211_RRF_NO_OUTDOOR,            "NL80211_RRF_NO_OUTDOOR" },
    { WS_NL80211_RRF_DFS,                   "NL80211_RRF_DFS" },
    { WS_NL80211_RRF_PTP_ONLY,              "NL80211_RRF_PTP_ONLY" },
    { WS_NL80211_RRF_PTMP_ONLY,             "NL80211_RRF_PTMP_ONLY" },
    { WS_NL80211_RRF_NO_IR,                 "NL80211_RRF_NO_IR" },
    { WS___NL80211_RRF_NO_IBSS,             "__NL80211_RRF_NO_IBSS" },
    { WS_NL80211_RRF_AUTO_BW,               "NL80211_RRF_AUTO_BW" },
    { WS_NL80211_RRF_IR_CONCURRENT,         "NL80211_RRF_IR_CONCURRENT" },
    { WS_NL80211_RRF_NO_HT40MINUS,          "NL80211_RRF_NO_HT40MINUS" },
    { WS_NL80211_RRF_NO_HT40PLUS,           "NL80211_RRF_NO_HT40PLUS" },
    { WS_NL80211_RRF_NO_80MHZ,              "NL80211_RRF_NO_80MHZ" },
    { WS_NL80211_RRF_NO_160MHZ,             "NL80211_RRF_NO_160MHZ" },
    { WS_NL80211_RRF_NO_HE,                 "NL80211_RRF_NO_HE" },
    { WS_NL80211_RRF_NO_320MHZ,             "NL80211_RRF_NO_320MHZ" },
    { WS_NL80211_RRF_NO_EHT,                "NL80211_RRF_NO_EHT" },
    { WS_NL80211_RRF_PSD,                   "NL80211_RRF_PSD" },
    { WS_NL80211_RRF_DFS_CONCURRENT,        "NL80211_RRF_DFS_CONCURRENT" },
    { WS_NL80211_RRF_NO_6GHZ_VLP_CLIENT,    "NL80211_RRF_NO_6GHZ_VLP_CLIENT" },
    { WS_NL80211_RRF_NO_6GHZ_AFC_CLIENT,    "NL80211_RRF_NO_6GHZ_AFC_CLIENT" },
    { WS_NL80211_RRF_ALLOW_6GHZ_VLP_AP,     "NL80211_RRF_ALLOW_6GHZ_VLP_AP" },
    { WS_NL80211_RRF_ALLOW_20MHZ_ACTIVITY,  "NL80211_RRF_ALLOW_20MHZ_ACTIVITY" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_reg_rule_flags_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_reg_rule_flags_vals);

static const value_string ws_nl80211_dfs_regions_vals[] = {
    { WS_NL80211_DFS_UNSET,                 "NL80211_DFS_UNSET" },
    { WS_NL80211_DFS_FCC,                   "NL80211_DFS_FCC" },
    { WS_NL80211_DFS_ETSI,                  "NL80211_DFS_ETSI" },
    { WS_NL80211_DFS_JP,                    "NL80211_DFS_JP" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_dfs_regions_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_dfs_regions_vals);

static const value_string ws_nl80211_user_reg_hint_type_vals[] = {
    { WS_NL80211_USER_REG_HINT_USER,        "NL80211_USER_REG_HINT_USER" },
    { WS_NL80211_USER_REG_HINT_CELL_BASE,   "NL80211_USER_REG_HINT_CELL_BASE" },
    { WS_NL80211_USER_REG_HINT_INDOOR,      "NL80211_USER_REG_HINT_INDOOR" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_user_reg_hint_type_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_user_reg_hint_type_vals);

static const value_string ws_nl80211_survey_info_vals[] = {
    { WS___NL80211_SURVEY_INFO_INVALID,     "__NL80211_SURVEY_INFO_INVALID" },
    { WS_NL80211_SURVEY_INFO_FREQUENCY,     "NL80211_SURVEY_INFO_FREQUENCY" },
    { WS_NL80211_SURVEY_INFO_NOISE,         "NL80211_SURVEY_INFO_NOISE" },
    { WS_NL80211_SURVEY_INFO_IN_USE,        "NL80211_SURVEY_INFO_IN_USE" },
    { WS_NL80211_SURVEY_INFO_TIME,          "NL80211_SURVEY_INFO_TIME" },
    { WS_NL80211_SURVEY_INFO_TIME_BUSY,     "NL80211_SURVEY_INFO_TIME_BUSY" },
    { WS_NL80211_SURVEY_INFO_TIME_EXT_BUSY, "NL80211_SURVEY_INFO_TIME_EXT_BUSY" },
    { WS_NL80211_SURVEY_INFO_TIME_RX,       "NL80211_SURVEY_INFO_TIME_RX" },
    { WS_NL80211_SURVEY_INFO_TIME_TX,       "NL80211_SURVEY_INFO_TIME_TX" },
    { WS_NL80211_SURVEY_INFO_TIME_SCAN,     "NL80211_SURVEY_INFO_TIME_SCAN" },
    { WS_NL80211_SURVEY_INFO_PAD,           "NL80211_SURVEY_INFO_PAD" },
    { WS_NL80211_SURVEY_INFO_TIME_BSS_RX,   "NL80211_SURVEY_INFO_TIME_BSS_RX" },
    { WS_NL80211_SURVEY_INFO_FREQUENCY_OFFSET, "NL80211_SURVEY_INFO_FREQUENCY_OFFSET" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_survey_info_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_survey_info_vals);

static const value_string ws_nl80211_mntr_flags_vals[] = {
    { WS___NL80211_MNTR_FLAG_INVALID,       "__NL80211_MNTR_FLAG_INVALID" },
    { WS_NL80211_MNTR_FLAG_FCSFAIL,         "NL80211_MNTR_FLAG_FCSFAIL" },
    { WS_NL80211_MNTR_FLAG_PLCPFAIL,        "NL80211_MNTR_FLAG_PLCPFAIL" },
    { WS_NL80211_MNTR_FLAG_CONTROL,         "NL80211_MNTR_FLAG_CONTROL" },
    { WS_NL80211_MNTR_FLAG_OTHER_BSS,       "NL80211_MNTR_FLAG_OTHER_BSS" },
    { WS_NL80211_MNTR_FLAG_COOK_FRAMES,     "NL80211_MNTR_FLAG_COOK_FRAMES" },
    { WS_NL80211_MNTR_FLAG_ACTIVE,          "NL80211_MNTR_FLAG_ACTIVE" },
    { WS_NL80211_MNTR_FLAG_SKIP_TX,         "NL80211_MNTR_FLAG_SKIP_TX" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_mntr_flags_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_mntr_flags_vals);

static const value_string ws_nl80211_mesh_power_mode_vals[] = {
    { WS_NL80211_MESH_POWER_UNKNOWN,        "NL80211_MESH_POWER_UNKNOWN" },
    { WS_NL80211_MESH_POWER_ACTIVE,         "NL80211_MESH_POWER_ACTIVE" },
    { WS_NL80211_MESH_POWER_LIGHT_SLEEP,    "NL80211_MESH_POWER_LIGHT_SLEEP" },
    { WS_NL80211_MESH_POWER_DEEP_SLEEP,     "NL80211_MESH_POWER_DEEP_SLEEP" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_mesh_power_mode_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_mesh_power_mode_vals);

static const value_string ws_nl80211_meshconf_params_vals[] = {
    { WS___NL80211_MESHCONF_INVALID,        "__NL80211_MESHCONF_INVALID" },
    { WS_NL80211_MESHCONF_RETRY_TIMEOUT,    "NL80211_MESHCONF_RETRY_TIMEOUT" },
    { WS_NL80211_MESHCONF_CONFIRM_TIMEOUT,  "NL80211_MESHCONF_CONFIRM_TIMEOUT" },
    { WS_NL80211_MESHCONF_HOLDING_TIMEOUT,  "NL80211_MESHCONF_HOLDING_TIMEOUT" },
    { WS_NL80211_MESHCONF_MAX_PEER_LINKS,   "NL80211_MESHCONF_MAX_PEER_LINKS" },
    { WS_NL80211_MESHCONF_MAX_RETRIES,      "NL80211_MESHCONF_MAX_RETRIES" },
    { WS_NL80211_MESHCONF_TTL,              "NL80211_MESHCONF_TTL" },
    { WS_NL80211_MESHCONF_AUTO_OPEN_PLINKS, "NL80211_MESHCONF_AUTO_OPEN_PLINKS" },
    { WS_NL80211_MESHCONF_HWMP_MAX_PREQ_RETRIES, "NL80211_MESHCONF_HWMP_MAX_PREQ_RETRIES" },
    { WS_NL80211_MESHCONF_PATH_REFRESH_TIME, "NL80211_MESHCONF_PATH_REFRESH_TIME" },
    { WS_NL80211_MESHCONF_MIN_DISCOVERY_TIMEOUT, "NL80211_MESHCONF_MIN_DISCOVERY_TIMEOUT" },
    { WS_NL80211_MESHCONF_HWMP_ACTIVE_PATH_TIMEOUT, "NL80211_MESHCONF_HWMP_ACTIVE_PATH_TIMEOUT" },
    { WS_NL80211_MESHCONF_HWMP_PREQ_MIN_INTERVAL, "NL80211_MESHCONF_HWMP_PREQ_MIN_INTERVAL" },
    { WS_NL80211_MESHCONF_HWMP_NET_DIAM_TRVS_TIME, "NL80211_MESHCONF_HWMP_NET_DIAM_TRVS_TIME" },
    { WS_NL80211_MESHCONF_HWMP_ROOTMODE,    "NL80211_MESHCONF_HWMP_ROOTMODE" },
    { WS_NL80211_MESHCONF_ELEMENT_TTL,      "NL80211_MESHCONF_ELEMENT_TTL" },
    { WS_NL80211_MESHCONF_HWMP_RANN_INTERVAL, "NL80211_MESHCONF_HWMP_RANN_INTERVAL" },
    { WS_NL80211_MESHCONF_GATE_ANNOUNCEMENTS, "NL80211_MESHCONF_GATE_ANNOUNCEMENTS" },
    { WS_NL80211_MESHCONF_HWMP_PERR_MIN_INTERVAL, "NL80211_MESHCONF_HWMP_PERR_MIN_INTERVAL" },
    { WS_NL80211_MESHCONF_FORWARDING,       "NL80211_MESHCONF_FORWARDING" },
    { WS_NL80211_MESHCONF_RSSI_THRESHOLD,   "NL80211_MESHCONF_RSSI_THRESHOLD" },
    { WS_NL80211_MESHCONF_SYNC_OFFSET_MAX_NEIGHBOR, "NL80211_MESHCONF_SYNC_OFFSET_MAX_NEIGHBOR" },
    { WS_NL80211_MESHCONF_HT_OPMODE,        "NL80211_MESHCONF_HT_OPMODE" },
    { WS_NL80211_MESHCONF_HWMP_PATH_TO_ROOT_TIMEOUT, "NL80211_MESHCONF_HWMP_PATH_TO_ROOT_TIMEOUT" },
    { WS_NL80211_MESHCONF_HWMP_ROOT_INTERVAL, "NL80211_MESHCONF_HWMP_ROOT_INTERVAL" },
    { WS_NL80211_MESHCONF_HWMP_CONFIRMATION_INTERVAL, "NL80211_MESHCONF_HWMP_CONFIRMATION_INTERVAL" },
    { WS_NL80211_MESHCONF_POWER_MODE,       "NL80211_MESHCONF_POWER_MODE" },
    { WS_NL80211_MESHCONF_AWAKE_WINDOW,     "NL80211_MESHCONF_AWAKE_WINDOW" },
    { WS_NL80211_MESHCONF_PLINK_TIMEOUT,    "NL80211_MESHCONF_PLINK_TIMEOUT" },
    { WS_NL80211_MESHCONF_CONNECTED_TO_GATE, "NL80211_MESHCONF_CONNECTED_TO_GATE" },
    { WS_NL80211_MESHCONF_NOLEARN,          "NL80211_MESHCONF_NOLEARN" },
    { WS_NL80211_MESHCONF_CONNECTED_TO_AS,  "NL80211_MESHCONF_CONNECTED_TO_AS" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_meshconf_params_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_meshconf_params_vals);

static const value_string ws_nl80211_mesh_setup_params_vals[] = {
    { WS___NL80211_MESH_SETUP_INVALID,      "__NL80211_MESH_SETUP_INVALID" },
    { WS_NL80211_MESH_SETUP_ENABLE_VENDOR_PATH_SEL, "NL80211_MESH_SETUP_ENABLE_VENDOR_PATH_SEL" },
    { WS_NL80211_MESH_SETUP_ENABLE_VENDOR_METRIC, "NL80211_MESH_SETUP_ENABLE_VENDOR_METRIC" },
    { WS_NL80211_MESH_SETUP_IE,             "NL80211_MESH_SETUP_IE" },
    { WS_NL80211_MESH_SETUP_USERSPACE_AUTH, "NL80211_MESH_SETUP_USERSPACE_AUTH" },
    { WS_NL80211_MESH_SETUP_USERSPACE_AMPE, "NL80211_MESH_SETUP_USERSPACE_AMPE" },
    { WS_NL80211_MESH_SETUP_ENABLE_VENDOR_SYNC, "NL80211_MESH_SETUP_ENABLE_VENDOR_SYNC" },
    { WS_NL80211_MESH_SETUP_USERSPACE_MPM,  "NL80211_MESH_SETUP_USERSPACE_MPM" },
    { WS_NL80211_MESH_SETUP_AUTH_PROTOCOL,  "NL80211_MESH_SETUP_AUTH_PROTOCOL" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_mesh_setup_params_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_mesh_setup_params_vals);

static const value_string ws_nl80211_txq_attr_vals[] = {
    { WS___NL80211_TXQ_ATTR_INVALID,        "__NL80211_TXQ_ATTR_INVALID" },
    { WS_NL80211_TXQ_ATTR_AC,               "NL80211_TXQ_ATTR_AC" },
    { WS_NL80211_TXQ_ATTR_TXOP,             "NL80211_TXQ_ATTR_TXOP" },
    { WS_NL80211_TXQ_ATTR_CWMIN,            "NL80211_TXQ_ATTR_CWMIN" },
    { WS_NL80211_TXQ_ATTR_CWMAX,            "NL80211_TXQ_ATTR_CWMAX" },
    { WS_NL80211_TXQ_ATTR_AIFS,             "NL80211_TXQ_ATTR_AIFS" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_txq_attr_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_txq_attr_vals);

static const value_string ws_nl80211_ac_vals[] = {
    { WS_NL80211_AC_VO,                     "NL80211_AC_VO" },
    { WS_NL80211_AC_VI,                     "NL80211_AC_VI" },
    { WS_NL80211_AC_BE,                     "NL80211_AC_BE" },
    { WS_NL80211_AC_BK,                     "NL80211_AC_BK" },
    { WS_NL80211_NUM_ACS,                   "NL80211_NUM_ACS" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_ac_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_ac_vals);

static const value_string ws_nl80211_channel_type_vals[] = {
    { WS_NL80211_CHAN_NO_HT,                "NL80211_CHAN_NO_HT" },
    { WS_NL80211_CHAN_HT20,                 "NL80211_CHAN_HT20" },
    { WS_NL80211_CHAN_HT40MINUS,            "NL80211_CHAN_HT40MINUS" },
    { WS_NL80211_CHAN_HT40PLUS,             "NL80211_CHAN_HT40PLUS" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_channel_type_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_channel_type_vals);

static const value_string ws_nl80211_key_mode_vals[] = {
    { WS_NL80211_KEY_RX_TX,                 "NL80211_KEY_RX_TX" },
    { WS_NL80211_KEY_NO_TX,                 "NL80211_KEY_NO_TX" },
    { WS_NL80211_KEY_SET_TX,                "NL80211_KEY_SET_TX" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_key_mode_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_key_mode_vals);

static const value_string ws_nl80211_chan_width_vals[] = {
    { WS_NL80211_CHAN_WIDTH_20_NOHT,        "NL80211_CHAN_WIDTH_20_NOHT" },
    { WS_NL80211_CHAN_WIDTH_20,             "NL80211_CHAN_WIDTH_20" },
    { WS_NL80211_CHAN_WIDTH_40,             "NL80211_CHAN_WIDTH_40" },
    { WS_NL80211_CHAN_WIDTH_80,             "NL80211_CHAN_WIDTH_80" },
    { WS_NL80211_CHAN_WIDTH_80P80,          "NL80211_CHAN_WIDTH_80P80" },
    { WS_NL80211_CHAN_WIDTH_160,            "NL80211_CHAN_WIDTH_160" },
    { WS_NL80211_CHAN_WIDTH_5,              "NL80211_CHAN_WIDTH_5" },
    { WS_NL80211_CHAN_WIDTH_10,             "NL80211_CHAN_WIDTH_10" },
    { WS_NL80211_CHAN_WIDTH_1,              "NL80211_CHAN_WIDTH_1" },
    { WS_NL80211_CHAN_WIDTH_2,              "NL80211_CHAN_WIDTH_2" },
    { WS_NL80211_CHAN_WIDTH_4,              "NL80211_CHAN_WIDTH_4" },
    { WS_NL80211_CHAN_WIDTH_8,              "NL80211_CHAN_WIDTH_8" },
    { WS_NL80211_CHAN_WIDTH_16,             "NL80211_CHAN_WIDTH_16" },
    { WS_NL80211_CHAN_WIDTH_320,            "NL80211_CHAN_WIDTH_320" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_chan_width_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_chan_width_vals);

static const value_string ws_nl80211_bss_scan_width_vals[] = {
    { WS_NL80211_BSS_CHAN_WIDTH_20,         "NL80211_BSS_CHAN_WIDTH_20" },
    { WS_NL80211_BSS_CHAN_WIDTH_10,         "NL80211_BSS_CHAN_WIDTH_10" },
    { WS_NL80211_BSS_CHAN_WIDTH_5,          "NL80211_BSS_CHAN_WIDTH_5" },
    { WS_NL80211_BSS_CHAN_WIDTH_1,          "NL80211_BSS_CHAN_WIDTH_1" },
    { WS_NL80211_BSS_CHAN_WIDTH_2,          "NL80211_BSS_CHAN_WIDTH_2" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_bss_scan_width_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_bss_scan_width_vals);

static const value_string ws_nl80211_bss_use_for_vals[] = {
    { WS_NL80211_BSS_USE_FOR_NORMAL,        "NL80211_BSS_USE_FOR_NORMAL" },
    { WS_NL80211_BSS_USE_FOR_MLD_LINK,      "NL80211_BSS_USE_FOR_MLD_LINK" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_bss_use_for_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_bss_use_for_vals);

static const value_string ws_nl80211_bss_cannot_use_reasons_vals[] = {
    { WS_NL80211_BSS_CANNOT_USE_NSTR_NONPRIMARY, "NL80211_BSS_CANNOT_USE_NSTR_NONPRIMARY" },
    { WS_NL80211_BSS_CANNOT_USE_6GHZ_PWR_MISMATCH, "NL80211_BSS_CANNOT_USE_6GHZ_PWR_MISMATCH" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_bss_cannot_use_reasons_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_bss_cannot_use_reasons_vals);

static const value_string ws_nl80211_bss_vals[] = {
    { WS___NL80211_BSS_INVALID,             "__NL80211_BSS_INVALID" },
    { WS_NL80211_BSS_BSSID,                 "NL80211_BSS_BSSID" },
    { WS_NL80211_BSS_FREQUENCY,             "NL80211_BSS_FREQUENCY" },
    { WS_NL80211_BSS_TSF,                   "NL80211_BSS_TSF" },
    { WS_NL80211_BSS_BEACON_INTERVAL,       "NL80211_BSS_BEACON_INTERVAL" },
    { WS_NL80211_BSS_CAPABILITY,            "NL80211_BSS_CAPABILITY" },
    { WS_NL80211_BSS_INFORMATION_ELEMENTS,  "NL80211_BSS_INFORMATION_ELEMENTS" },
    { WS_NL80211_BSS_SIGNAL_MBM,            "NL80211_BSS_SIGNAL_MBM" },
    { WS_NL80211_BSS_SIGNAL_UNSPEC,         "NL80211_BSS_SIGNAL_UNSPEC" },
    { WS_NL80211_BSS_STATUS,                "NL80211_BSS_STATUS" },
    { WS_NL80211_BSS_SEEN_MS_AGO,           "NL80211_BSS_SEEN_MS_AGO" },
    { WS_NL80211_BSS_BEACON_IES,            "NL80211_BSS_BEACON_IES" },
    { WS_NL80211_BSS_CHAN_WIDTH,            "NL80211_BSS_CHAN_WIDTH" },
    { WS_NL80211_BSS_BEACON_TSF,            "NL80211_BSS_BEACON_TSF" },
    { WS_NL80211_BSS_PRESP_DATA,            "NL80211_BSS_PRESP_DATA" },
    { WS_NL80211_BSS_LAST_SEEN_BOOTTIME,    "NL80211_BSS_LAST_SEEN_BOOTTIME" },
    { WS_NL80211_BSS_PAD,                   "NL80211_BSS_PAD" },
    { WS_NL80211_BSS_PARENT_TSF,            "NL80211_BSS_PARENT_TSF" },
    { WS_NL80211_BSS_PARENT_BSSID,          "NL80211_BSS_PARENT_BSSID" },
    { WS_NL80211_BSS_CHAIN_SIGNAL,          "NL80211_BSS_CHAIN_SIGNAL" },
    { WS_NL80211_BSS_FREQUENCY_OFFSET,      "NL80211_BSS_FREQUENCY_OFFSET" },
    { WS_NL80211_BSS_MLO_LINK_ID,           "NL80211_BSS_MLO_LINK_ID" },
    { WS_NL80211_BSS_MLD_ADDR,              "NL80211_BSS_MLD_ADDR" },
    { WS_NL80211_BSS_USE_FOR,               "NL80211_BSS_USE_FOR" },
    { WS_NL80211_BSS_CANNOT_USE_REASONS,    "NL80211_BSS_CANNOT_USE_REASONS" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_bss_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_bss_vals);

static const value_string ws_nl80211_bss_status_vals[] = {
    { WS_NL80211_BSS_STATUS_AUTHENTICATED,  "NL80211_BSS_STATUS_AUTHENTICATED" },
    { WS_NL80211_BSS_STATUS_ASSOCIATED,     "NL80211_BSS_STATUS_ASSOCIATED" },
    { WS_NL80211_BSS_STATUS_IBSS_JOINED,    "NL80211_BSS_STATUS_IBSS_JOINED" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_bss_status_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_bss_status_vals);

static const value_string ws_nl80211_auth_type_vals[] = {
    { WS_NL80211_AUTHTYPE_OPEN_SYSTEM,      "NL80211_AUTHTYPE_OPEN_SYSTEM" },
    { WS_NL80211_AUTHTYPE_SHARED_KEY,       "NL80211_AUTHTYPE_SHARED_KEY" },
    { WS_NL80211_AUTHTYPE_FT,               "NL80211_AUTHTYPE_FT" },
    { WS_NL80211_AUTHTYPE_NETWORK_EAP,      "NL80211_AUTHTYPE_NETWORK_EAP" },
    { WS_NL80211_AUTHTYPE_SAE,              "NL80211_AUTHTYPE_SAE" },
    { WS_NL80211_AUTHTYPE_FILS_SK,          "NL80211_AUTHTYPE_FILS_SK" },
    { WS_NL80211_AUTHTYPE_FILS_SK_PFS,      "NL80211_AUTHTYPE_FILS_SK_PFS" },
    { WS_NL80211_AUTHTYPE_FILS_PK,          "NL80211_AUTHTYPE_FILS_PK" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_auth_type_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_auth_type_vals);

static const value_string ws_nl80211_key_type_vals[] = {
    { WS_NL80211_KEYTYPE_GROUP,             "NL80211_KEYTYPE_GROUP" },
    { WS_NL80211_KEYTYPE_PAIRWISE,          "NL80211_KEYTYPE_PAIRWISE" },
    { WS_NL80211_KEYTYPE_PEERKEY,           "NL80211_KEYTYPE_PEERKEY" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_key_type_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_key_type_vals);

static const value_string ws_nl80211_mfp_vals[] = {
    { WS_NL80211_MFP_NO,                    "NL80211_MFP_NO" },
    { WS_NL80211_MFP_REQUIRED,              "NL80211_MFP_REQUIRED" },
    { WS_NL80211_MFP_OPTIONAL,              "NL80211_MFP_OPTIONAL" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_mfp_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_mfp_vals);

static const value_string ws_nl80211_wpa_versions_vals[] = {
    { WS_NL80211_WPA_VERSION_1,             "NL80211_WPA_VERSION_1" },
    { WS_NL80211_WPA_VERSION_2,             "NL80211_WPA_VERSION_2" },
    { WS_NL80211_WPA_VERSION_3,             "NL80211_WPA_VERSION_3" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_wpa_versions_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_wpa_versions_vals);

static const value_string ws_nl80211_key_default_types_vals[] = {
    { WS___NL80211_KEY_DEFAULT_TYPE_INVALID, "__NL80211_KEY_DEFAULT_TYPE_INVALID" },
    { WS_NL80211_KEY_DEFAULT_TYPE_UNICAST,  "NL80211_KEY_DEFAULT_TYPE_UNICAST" },
    { WS_NL80211_KEY_DEFAULT_TYPE_MULTICAST, "NL80211_KEY_DEFAULT_TYPE_MULTICAST" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_key_default_types_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_key_default_types_vals);

static const value_string ws_nl80211_key_attributes_vals[] = {
    { WS___NL80211_KEY_INVALID,             "__NL80211_KEY_INVALID" },
    { WS_NL80211_KEY_DATA,                  "NL80211_KEY_DATA" },
    { WS_NL80211_KEY_IDX,                   "NL80211_KEY_IDX" },
    { WS_NL80211_KEY_CIPHER,                "NL80211_KEY_CIPHER" },
    { WS_NL80211_KEY_SEQ,                   "NL80211_KEY_SEQ" },
    { WS_NL80211_KEY_DEFAULT,               "NL80211_KEY_DEFAULT" },
    { WS_NL80211_KEY_DEFAULT_MGMT,          "NL80211_KEY_DEFAULT_MGMT" },
    { WS_NL80211_KEY_TYPE,                  "NL80211_KEY_TYPE" },
    { WS_NL80211_KEY_DEFAULT_TYPES,         "NL80211_KEY_DEFAULT_TYPES" },
    { WS_NL80211_KEY_MODE,                  "NL80211_KEY_MODE" },
    { WS_NL80211_KEY_DEFAULT_BEACON,        "NL80211_KEY_DEFAULT_BEACON" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_key_attributes_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_key_attributes_vals);

static const value_string ws_nl80211_tx_rate_attributes_vals[] = {
    { WS___NL80211_TXRATE_INVALID,          "__NL80211_TXRATE_INVALID" },
    { WS_NL80211_TXRATE_LEGACY,             "NL80211_TXRATE_LEGACY" },
    { WS_NL80211_TXRATE_HT,                 "NL80211_TXRATE_HT" },
    { WS_NL80211_TXRATE_VHT,                "NL80211_TXRATE_VHT" },
    { WS_NL80211_TXRATE_GI,                 "NL80211_TXRATE_GI" },
    { WS_NL80211_TXRATE_HE,                 "NL80211_TXRATE_HE" },
    { WS_NL80211_TXRATE_HE_GI,              "NL80211_TXRATE_HE_GI" },
    { WS_NL80211_TXRATE_HE_LTF,             "NL80211_TXRATE_HE_LTF" },
    { WS_NL80211_TXRATE_EHT,                "NL80211_TXRATE_EHT" },
    { WS_NL80211_TXRATE_EHT_GI,             "NL80211_TXRATE_EHT_GI" },
    { WS_NL80211_TXRATE_EHT_LTF,            "NL80211_TXRATE_EHT_LTF" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_tx_rate_attributes_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_tx_rate_attributes_vals);

static const value_string ws_nl80211_txrate_gi_vals[] = {
    { WS_NL80211_TXRATE_DEFAULT_GI,         "NL80211_TXRATE_DEFAULT_GI" },
    { WS_NL80211_TXRATE_FORCE_SGI,          "NL80211_TXRATE_FORCE_SGI" },
    { WS_NL80211_TXRATE_FORCE_LGI,          "NL80211_TXRATE_FORCE_LGI" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_txrate_gi_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_txrate_gi_vals);

static const value_string ws_nl80211_band_vals[] = {
    { WS_NL80211_BAND_2GHZ,                 "NL80211_BAND_2GHZ" },
    { WS_NL80211_BAND_5GHZ,                 "NL80211_BAND_5GHZ" },
    { WS_NL80211_BAND_60GHZ,                "NL80211_BAND_60GHZ" },
    { WS_NL80211_BAND_6GHZ,                 "NL80211_BAND_6GHZ" },
    { WS_NL80211_BAND_S1GHZ,                "NL80211_BAND_S1GHZ" },
    { WS_NL80211_BAND_LC,                   "NL80211_BAND_LC" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_band_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_band_vals);

static const value_string ws_nl80211_ps_state_vals[] = {
    { WS_NL80211_PS_DISABLED,               "NL80211_PS_DISABLED" },
    { WS_NL80211_PS_ENABLED,                "NL80211_PS_ENABLED" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_ps_state_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_ps_state_vals);

static const value_string ws_nl80211_attr_cqm_vals[] = {
    { WS___NL80211_ATTR_CQM_INVALID,        "__NL80211_ATTR_CQM_INVALID" },
    { WS_NL80211_ATTR_CQM_RSSI_THOLD,       "NL80211_ATTR_CQM_RSSI_THOLD" },
    { WS_NL80211_ATTR_CQM_RSSI_HYST,        "NL80211_ATTR_CQM_RSSI_HYST" },
    { WS_NL80211_ATTR_CQM_RSSI_THRESHOLD_EVENT, "NL80211_ATTR_CQM_RSSI_THRESHOLD_EVENT" },
    { WS_NL80211_ATTR_CQM_PKT_LOSS_EVENT,   "NL80211_ATTR_CQM_PKT_LOSS_EVENT" },
    { WS_NL80211_ATTR_CQM_TXE_RATE,         "NL80211_ATTR_CQM_TXE_RATE" },
    { WS_NL80211_ATTR_CQM_TXE_PKTS,         "NL80211_ATTR_CQM_TXE_PKTS" },
    { WS_NL80211_ATTR_CQM_TXE_INTVL,        "NL80211_ATTR_CQM_TXE_INTVL" },
    { WS_NL80211_ATTR_CQM_BEACON_LOSS_EVENT, "NL80211_ATTR_CQM_BEACON_LOSS_EVENT" },
    { WS_NL80211_ATTR_CQM_RSSI_LEVEL,       "NL80211_ATTR_CQM_RSSI_LEVEL" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_attr_cqm_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_attr_cqm_vals);

static const value_string ws_nl80211_cqm_rssi_threshold_event_vals[] = {
    { WS_NL80211_CQM_RSSI_THRESHOLD_EVENT_LOW, "NL80211_CQM_RSSI_THRESHOLD_EVENT_LOW" },
    { WS_NL80211_CQM_RSSI_THRESHOLD_EVENT_HIGH, "NL80211_CQM_RSSI_THRESHOLD_EVENT_HIGH" },
    { WS_NL80211_CQM_RSSI_BEACON_LOSS_EVENT, "NL80211_CQM_RSSI_BEACON_LOSS_EVENT" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_cqm_rssi_threshold_event_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_cqm_rssi_threshold_event_vals);

static const value_string ws_nl80211_tx_power_setting_vals[] = {
    { WS_NL80211_TX_POWER_AUTOMATIC,        "NL80211_TX_POWER_AUTOMATIC" },
    { WS_NL80211_TX_POWER_LIMITED,          "NL80211_TX_POWER_LIMITED" },
    { WS_NL80211_TX_POWER_FIXED,            "NL80211_TX_POWER_FIXED" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_tx_power_setting_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_tx_power_setting_vals);

static const value_string ws_nl80211_tid_config_vals[] = {
    { WS_NL80211_TID_CONFIG_ENABLE,         "NL80211_TID_CONFIG_ENABLE" },
    { WS_NL80211_TID_CONFIG_DISABLE,        "NL80211_TID_CONFIG_DISABLE" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_tid_config_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_tid_config_vals);

static const value_string ws_nl80211_tx_rate_setting_vals[] = {
    { WS_NL80211_TX_RATE_AUTOMATIC,         "NL80211_TX_RATE_AUTOMATIC" },
    { WS_NL80211_TX_RATE_LIMITED,           "NL80211_TX_RATE_LIMITED" },
    { WS_NL80211_TX_RATE_FIXED,             "NL80211_TX_RATE_FIXED" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_tx_rate_setting_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_tx_rate_setting_vals);

static const value_string ws_nl80211_tid_config_attr_vals[] = {
    { WS___NL80211_TID_CONFIG_ATTR_INVALID, "__NL80211_TID_CONFIG_ATTR_INVALID" },
    { WS_NL80211_TID_CONFIG_ATTR_PAD,       "NL80211_TID_CONFIG_ATTR_PAD" },
    { WS_NL80211_TID_CONFIG_ATTR_VIF_SUPP,  "NL80211_TID_CONFIG_ATTR_VIF_SUPP" },
    { WS_NL80211_TID_CONFIG_ATTR_PEER_SUPP, "NL80211_TID_CONFIG_ATTR_PEER_SUPP" },
    { WS_NL80211_TID_CONFIG_ATTR_OVERRIDE,  "NL80211_TID_CONFIG_ATTR_OVERRIDE" },
    { WS_NL80211_TID_CONFIG_ATTR_TIDS,      "NL80211_TID_CONFIG_ATTR_TIDS" },
    { WS_NL80211_TID_CONFIG_ATTR_NOACK,     "NL80211_TID_CONFIG_ATTR_NOACK" },
    { WS_NL80211_TID_CONFIG_ATTR_RETRY_SHORT, "NL80211_TID_CONFIG_ATTR_RETRY_SHORT" },
    { WS_NL80211_TID_CONFIG_ATTR_RETRY_LONG, "NL80211_TID_CONFIG_ATTR_RETRY_LONG" },
    { WS_NL80211_TID_CONFIG_ATTR_AMPDU_CTRL, "NL80211_TID_CONFIG_ATTR_AMPDU_CTRL" },
    { WS_NL80211_TID_CONFIG_ATTR_RTSCTS_CTRL, "NL80211_TID_CONFIG_ATTR_RTSCTS_CTRL" },
    { WS_NL80211_TID_CONFIG_ATTR_AMSDU_CTRL, "NL80211_TID_CONFIG_ATTR_AMSDU_CTRL" },
    { WS_NL80211_TID_CONFIG_ATTR_TX_RATE_TYPE, "NL80211_TID_CONFIG_ATTR_TX_RATE_TYPE" },
    { WS_NL80211_TID_CONFIG_ATTR_TX_RATE,   "NL80211_TID_CONFIG_ATTR_TX_RATE" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_tid_config_attr_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_tid_config_attr_vals);

static const value_string ws_nl80211_packet_pattern_attr_vals[] = {
    { WS___NL80211_PKTPAT_INVALID,          "__NL80211_PKTPAT_INVALID" },
    { WS_NL80211_PKTPAT_MASK,               "NL80211_PKTPAT_MASK" },
    { WS_NL80211_PKTPAT_PATTERN,            "NL80211_PKTPAT_PATTERN" },
    { WS_NL80211_PKTPAT_OFFSET,             "NL80211_PKTPAT_OFFSET" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_packet_pattern_attr_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_packet_pattern_attr_vals);

static const value_string ws_nl80211_wowlan_triggers_vals[] = {
    { WS___NL80211_WOWLAN_TRIG_INVALID,     "__NL80211_WOWLAN_TRIG_INVALID" },
    { WS_NL80211_WOWLAN_TRIG_ANY,           "NL80211_WOWLAN_TRIG_ANY" },
    { WS_NL80211_WOWLAN_TRIG_DISCONNECT,    "NL80211_WOWLAN_TRIG_DISCONNECT" },
    { WS_NL80211_WOWLAN_TRIG_MAGIC_PKT,     "NL80211_WOWLAN_TRIG_MAGIC_PKT" },
    { WS_NL80211_WOWLAN_TRIG_PKT_PATTERN,   "NL80211_WOWLAN_TRIG_PKT_PATTERN" },
    { WS_NL80211_WOWLAN_TRIG_GTK_REKEY_SUPPORTED, "NL80211_WOWLAN_TRIG_GTK_REKEY_SUPPORTED" },
    { WS_NL80211_WOWLAN_TRIG_GTK_REKEY_FAILURE, "NL80211_WOWLAN_TRIG_GTK_REKEY_FAILURE" },
    { WS_NL80211_WOWLAN_TRIG_EAP_IDENT_REQUEST, "NL80211_WOWLAN_TRIG_EAP_IDENT_REQUEST" },
    { WS_NL80211_WOWLAN_TRIG_4WAY_HANDSHAKE, "NL80211_WOWLAN_TRIG_4WAY_HANDSHAKE" },
    { WS_NL80211_WOWLAN_TRIG_RFKILL_RELEASE, "NL80211_WOWLAN_TRIG_RFKILL_RELEASE" },
    { WS_NL80211_WOWLAN_TRIG_WAKEUP_PKT_80211, "NL80211_WOWLAN_TRIG_WAKEUP_PKT_80211" },
    { WS_NL80211_WOWLAN_TRIG_WAKEUP_PKT_80211_LEN, "NL80211_WOWLAN_TRIG_WAKEUP_PKT_80211_LEN" },
    { WS_NL80211_WOWLAN_TRIG_WAKEUP_PKT_8023, "NL80211_WOWLAN_TRIG_WAKEUP_PKT_8023" },
    { WS_NL80211_WOWLAN_TRIG_WAKEUP_PKT_8023_LEN, "NL80211_WOWLAN_TRIG_WAKEUP_PKT_8023_LEN" },
    { WS_NL80211_WOWLAN_TRIG_TCP_CONNECTION, "NL80211_WOWLAN_TRIG_TCP_CONNECTION" },
    { WS_NL80211_WOWLAN_TRIG_WAKEUP_TCP_MATCH, "NL80211_WOWLAN_TRIG_WAKEUP_TCP_MATCH" },
    { WS_NL80211_WOWLAN_TRIG_WAKEUP_TCP_CONNLOST, "NL80211_WOWLAN_TRIG_WAKEUP_TCP_CONNLOST" },
    { WS_NL80211_WOWLAN_TRIG_WAKEUP_TCP_NOMORETOKENS, "NL80211_WOWLAN_TRIG_WAKEUP_TCP_NOMORETOKENS" },
    { WS_NL80211_WOWLAN_TRIG_NET_DETECT,    "NL80211_WOWLAN_TRIG_NET_DETECT" },
    { WS_NL80211_WOWLAN_TRIG_NET_DETECT_RESULTS, "NL80211_WOWLAN_TRIG_NET_DETECT_RESULTS" },
    { WS_NL80211_WOWLAN_TRIG_UNPROTECTED_DEAUTH_DISASSOC, "NL80211_WOWLAN_TRIG_UNPROTECTED_DEAUTH_DISASSOC" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_wowlan_triggers_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_wowlan_triggers_vals);

static const value_string ws_nl80211_wowlan_tcp_attrs_vals[] = {
    { WS___NL80211_WOWLAN_TCP_INVALID,      "__NL80211_WOWLAN_TCP_INVALID" },
    { WS_NL80211_WOWLAN_TCP_SRC_IPV4,       "NL80211_WOWLAN_TCP_SRC_IPV4" },
    { WS_NL80211_WOWLAN_TCP_DST_IPV4,       "NL80211_WOWLAN_TCP_DST_IPV4" },
    { WS_NL80211_WOWLAN_TCP_DST_MAC,        "NL80211_WOWLAN_TCP_DST_MAC" },
    { WS_NL80211_WOWLAN_TCP_SRC_PORT,       "NL80211_WOWLAN_TCP_SRC_PORT" },
    { WS_NL80211_WOWLAN_TCP_DST_PORT,       "NL80211_WOWLAN_TCP_DST_PORT" },
    { WS_NL80211_WOWLAN_TCP_DATA_PAYLOAD,   "NL80211_WOWLAN_TCP_DATA_PAYLOAD" },
    { WS_NL80211_WOWLAN_TCP_DATA_PAYLOAD_SEQ, "NL80211_WOWLAN_TCP_DATA_PAYLOAD_SEQ" },
    { WS_NL80211_WOWLAN_TCP_DATA_PAYLOAD_TOKEN, "NL80211_WOWLAN_TCP_DATA_PAYLOAD_TOKEN" },
    { WS_NL80211_WOWLAN_TCP_DATA_INTERVAL,  "NL80211_WOWLAN_TCP_DATA_INTERVAL" },
    { WS_NL80211_WOWLAN_TCP_WAKE_PAYLOAD,   "NL80211_WOWLAN_TCP_WAKE_PAYLOAD" },
    { WS_NL80211_WOWLAN_TCP_WAKE_MASK,      "NL80211_WOWLAN_TCP_WAKE_MASK" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_wowlan_tcp_attrs_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_wowlan_tcp_attrs_vals);

static const value_string ws_nl80211_attr_coalesce_rule_vals[] = {
    { WS___NL80211_COALESCE_RULE_INVALID,   "__NL80211_COALESCE_RULE_INVALID" },
    { WS_NL80211_ATTR_COALESCE_RULE_DELAY,  "NL80211_ATTR_COALESCE_RULE_DELAY" },
    { WS_NL80211_ATTR_COALESCE_RULE_CONDITION, "NL80211_ATTR_COALESCE_RULE_CONDITION" },
    { WS_NL80211_ATTR_COALESCE_RULE_PKT_PATTERN, "NL80211_ATTR_COALESCE_RULE_PKT_PATTERN" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_attr_coalesce_rule_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_attr_coalesce_rule_vals);

static const value_string ws_nl80211_coalesce_condition_vals[] = {
    { WS_NL80211_COALESCE_CONDITION_MATCH,  "NL80211_COALESCE_CONDITION_MATCH" },
    { WS_NL80211_COALESCE_CONDITION_NO_MATCH, "NL80211_COALESCE_CONDITION_NO_MATCH" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_coalesce_condition_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_coalesce_condition_vals);

static const value_string ws_nl80211_iface_limit_attrs_vals[] = {
    { WS_NL80211_IFACE_LIMIT_UNSPEC,        "NL80211_IFACE_LIMIT_UNSPEC" },
    { WS_NL80211_IFACE_LIMIT_MAX,           "NL80211_IFACE_LIMIT_MAX" },
    { WS_NL80211_IFACE_LIMIT_TYPES,         "NL80211_IFACE_LIMIT_TYPES" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_iface_limit_attrs_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_iface_limit_attrs_vals);

static const value_string ws_nl80211_if_combination_attrs_vals[] = {
    { WS_NL80211_IFACE_COMB_UNSPEC,         "NL80211_IFACE_COMB_UNSPEC" },
    { WS_NL80211_IFACE_COMB_LIMITS,         "NL80211_IFACE_COMB_LIMITS" },
    { WS_NL80211_IFACE_COMB_MAXNUM,         "NL80211_IFACE_COMB_MAXNUM" },
    { WS_NL80211_IFACE_COMB_STA_AP_BI_MATCH, "NL80211_IFACE_COMB_STA_AP_BI_MATCH" },
    { WS_NL80211_IFACE_COMB_NUM_CHANNELS,   "NL80211_IFACE_COMB_NUM_CHANNELS" },
    { WS_NL80211_IFACE_COMB_RADAR_DETECT_WIDTHS, "NL80211_IFACE_COMB_RADAR_DETECT_WIDTHS" },
    { WS_NL80211_IFACE_COMB_RADAR_DETECT_REGIONS, "NL80211_IFACE_COMB_RADAR_DETECT_REGIONS" },
    { WS_NL80211_IFACE_COMB_BI_MIN_GCD,     "NL80211_IFACE_COMB_BI_MIN_GCD" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_if_combination_attrs_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_if_combination_attrs_vals);

static const value_string ws_nl80211_plink_state_vals[] = {
    { WS_NL80211_PLINK_LISTEN,              "NL80211_PLINK_LISTEN" },
    { WS_NL80211_PLINK_OPN_SNT,             "NL80211_PLINK_OPN_SNT" },
    { WS_NL80211_PLINK_OPN_RCVD,            "NL80211_PLINK_OPN_RCVD" },
    { WS_NL80211_PLINK_CNF_RCVD,            "NL80211_PLINK_CNF_RCVD" },
    { WS_NL80211_PLINK_ESTAB,               "NL80211_PLINK_ESTAB" },
    { WS_NL80211_PLINK_HOLDING,             "NL80211_PLINK_HOLDING" },
    { WS_NL80211_PLINK_BLOCKED,             "NL80211_PLINK_BLOCKED" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_plink_state_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_plink_state_vals);

static const value_string ws_nl80211_plink_action_vals[] = {
    { WS_NL80211_PLINK_ACTION_NO_ACTION,    "NL80211_PLINK_ACTION_NO_ACTION" },
    { WS_NL80211_PLINK_ACTION_OPEN,         "NL80211_PLINK_ACTION_OPEN" },
    { WS_NL80211_PLINK_ACTION_BLOCK,        "NL80211_PLINK_ACTION_BLOCK" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_plink_action_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_plink_action_vals);

static const value_string ws_nl80211_rekey_data_vals[] = {
    { WS___NL80211_REKEY_DATA_INVALID,      "__NL80211_REKEY_DATA_INVALID" },
    { WS_NL80211_REKEY_DATA_KEK,            "NL80211_REKEY_DATA_KEK" },
    { WS_NL80211_REKEY_DATA_KCK,            "NL80211_REKEY_DATA_KCK" },
    { WS_NL80211_REKEY_DATA_REPLAY_CTR,     "NL80211_REKEY_DATA_REPLAY_CTR" },
    { WS_NL80211_REKEY_DATA_AKM,            "NL80211_REKEY_DATA_AKM" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_rekey_data_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_rekey_data_vals);

static const value_string ws_nl80211_hidden_ssid_vals[] = {
    { WS_NL80211_HIDDEN_SSID_NOT_IN_USE,    "NL80211_HIDDEN_SSID_NOT_IN_USE" },
    { WS_NL80211_HIDDEN_SSID_ZERO_LEN,      "NL80211_HIDDEN_SSID_ZERO_LEN" },
    { WS_NL80211_HIDDEN_SSID_ZERO_CONTENTS, "NL80211_HIDDEN_SSID_ZERO_CONTENTS" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_hidden_ssid_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_hidden_ssid_vals);

static const value_string ws_nl80211_sta_wme_attr_vals[] = {
    { WS___NL80211_STA_WME_INVALID,         "__NL80211_STA_WME_INVALID" },
    { WS_NL80211_STA_WME_UAPSD_QUEUES,      "NL80211_STA_WME_UAPSD_QUEUES" },
    { WS_NL80211_STA_WME_MAX_SP,            "NL80211_STA_WME_MAX_SP" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_sta_wme_attr_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_sta_wme_attr_vals);

static const value_string ws_nl80211_pmksa_candidate_attr_vals[] = {
    { WS___NL80211_PMKSA_CANDIDATE_INVALID, "__NL80211_PMKSA_CANDIDATE_INVALID" },
    { WS_NL80211_PMKSA_CANDIDATE_INDEX,     "NL80211_PMKSA_CANDIDATE_INDEX" },
    { WS_NL80211_PMKSA_CANDIDATE_BSSID,     "NL80211_PMKSA_CANDIDATE_BSSID" },
    { WS_NL80211_PMKSA_CANDIDATE_PREAUTH,   "NL80211_PMKSA_CANDIDATE_PREAUTH" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_pmksa_candidate_attr_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_pmksa_candidate_attr_vals);

static const value_string ws_nl80211_tdls_operation_vals[] = {
    { WS_NL80211_TDLS_DISCOVERY_REQ,        "NL80211_TDLS_DISCOVERY_REQ" },
    { WS_NL80211_TDLS_SETUP,                "NL80211_TDLS_SETUP" },
    { WS_NL80211_TDLS_TEARDOWN,             "NL80211_TDLS_TEARDOWN" },
    { WS_NL80211_TDLS_ENABLE_LINK,          "NL80211_TDLS_ENABLE_LINK" },
    { WS_NL80211_TDLS_DISABLE_LINK,         "NL80211_TDLS_DISABLE_LINK" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_tdls_operation_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_tdls_operation_vals);

static const value_string ws_nl80211_ap_sme_features_vals[] = {
    { WS_NL80211_AP_SME_SA_QUERY_OFFLOAD,   "NL80211_AP_SME_SA_QUERY_OFFLOAD" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_ap_sme_features_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_ap_sme_features_vals);

static const value_string ws_nl80211_feature_flags_vals[] = {
    { WS_NL80211_FEATURE_SK_TX_STATUS,      "NL80211_FEATURE_SK_TX_STATUS" },
    { WS_NL80211_FEATURE_HT_IBSS,           "NL80211_FEATURE_HT_IBSS" },
    { WS_NL80211_FEATURE_INACTIVITY_TIMER,  "NL80211_FEATURE_INACTIVITY_TIMER" },
    { WS_NL80211_FEATURE_CELL_BASE_REG_HINTS, "NL80211_FEATURE_CELL_BASE_REG_HINTS" },
    { WS_NL80211_FEATURE_P2P_DEVICE_NEEDS_CHANNEL, "NL80211_FEATURE_P2P_DEVICE_NEEDS_CHANNEL" },
    { WS_NL80211_FEATURE_SAE,               "NL80211_FEATURE_SAE" },
    { WS_NL80211_FEATURE_LOW_PRIORITY_SCAN, "NL80211_FEATURE_LOW_PRIORITY_SCAN" },
    { WS_NL80211_FEATURE_SCAN_FLUSH,        "NL80211_FEATURE_SCAN_FLUSH" },
    { WS_NL80211_FEATURE_AP_SCAN,           "NL80211_FEATURE_AP_SCAN" },
    { WS_NL80211_FEATURE_VIF_TXPOWER,       "NL80211_FEATURE_VIF_TXPOWER" },
    { WS_NL80211_FEATURE_NEED_OBSS_SCAN,    "NL80211_FEATURE_NEED_OBSS_SCAN" },
    { WS_NL80211_FEATURE_P2P_GO_CTWIN,      "NL80211_FEATURE_P2P_GO_CTWIN" },
    { WS_NL80211_FEATURE_P2P_GO_OPPPS,      "NL80211_FEATURE_P2P_GO_OPPPS" },
    { WS_NL80211_FEATURE_ADVERTISE_CHAN_LIMITS, "NL80211_FEATURE_ADVERTISE_CHAN_LIMITS" },
    { WS_NL80211_FEATURE_FULL_AP_CLIENT_STATE, "NL80211_FEATURE_FULL_AP_CLIENT_STATE" },
    { WS_NL80211_FEATURE_USERSPACE_MPM,     "NL80211_FEATURE_USERSPACE_MPM" },
    { WS_NL80211_FEATURE_ACTIVE_MONITOR,    "NL80211_FEATURE_ACTIVE_MONITOR" },
    { WS_NL80211_FEATURE_AP_MODE_CHAN_WIDTH_CHANGE, "NL80211_FEATURE_AP_MODE_CHAN_WIDTH_CHANGE" },
    { WS_NL80211_FEATURE_DS_PARAM_SET_IE_IN_PROBES, "NL80211_FEATURE_DS_PARAM_SET_IE_IN_PROBES" },
    { WS_NL80211_FEATURE_WFA_TPC_IE_IN_PROBES, "NL80211_FEATURE_WFA_TPC_IE_IN_PROBES" },
    { WS_NL80211_FEATURE_QUIET,             "NL80211_FEATURE_QUIET" },
    { WS_NL80211_FEATURE_TX_POWER_INSERTION, "NL80211_FEATURE_TX_POWER_INSERTION" },
    { WS_NL80211_FEATURE_ACKTO_ESTIMATION,  "NL80211_FEATURE_ACKTO_ESTIMATION" },
    { WS_NL80211_FEATURE_STATIC_SMPS,       "NL80211_FEATURE_STATIC_SMPS" },
    { WS_NL80211_FEATURE_DYNAMIC_SMPS,      "NL80211_FEATURE_DYNAMIC_SMPS" },
    { WS_NL80211_FEATURE_SUPPORTS_WMM_ADMISSION, "NL80211_FEATURE_SUPPORTS_WMM_ADMISSION" },
    { WS_NL80211_FEATURE_MAC_ON_CREATE,     "NL80211_FEATURE_MAC_ON_CREATE" },
    { WS_NL80211_FEATURE_TDLS_CHANNEL_SWITCH, "NL80211_FEATURE_TDLS_CHANNEL_SWITCH" },
    { WS_NL80211_FEATURE_SCAN_RANDOM_MAC_ADDR, "NL80211_FEATURE_SCAN_RANDOM_MAC_ADDR" },
    { WS_NL80211_FEATURE_SCHED_SCAN_RANDOM_MAC_ADDR, "NL80211_FEATURE_SCHED_SCAN_RANDOM_MAC_ADDR" },
    { WS_NL80211_FEATURE_ND_RANDOM_MAC_ADDR, "NL80211_FEATURE_ND_RANDOM_MAC_ADDR" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_feature_flags_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_feature_flags_vals);

static const value_string ws_nl80211_ext_feature_index_vals[] = {
    { WS_NL80211_EXT_FEATURE_VHT_IBSS,      "NL80211_EXT_FEATURE_VHT_IBSS" },
    { WS_NL80211_EXT_FEATURE_RRM,           "NL80211_EXT_FEATURE_RRM" },
    { WS_NL80211_EXT_FEATURE_MU_MIMO_AIR_SNIFFER, "NL80211_EXT_FEATURE_MU_MIMO_AIR_SNIFFER" },
    { WS_NL80211_EXT_FEATURE_SCAN_START_TIME, "NL80211_EXT_FEATURE_SCAN_START_TIME" },
    { WS_NL80211_EXT_FEATURE_BSS_PARENT_TSF, "NL80211_EXT_FEATURE_BSS_PARENT_TSF" },
    { WS_NL80211_EXT_FEATURE_SET_SCAN_DWELL, "NL80211_EXT_FEATURE_SET_SCAN_DWELL" },
    { WS_NL80211_EXT_FEATURE_BEACON_RATE_LEGACY, "NL80211_EXT_FEATURE_BEACON_RATE_LEGACY" },
    { WS_NL80211_EXT_FEATURE_BEACON_RATE_HT, "NL80211_EXT_FEATURE_BEACON_RATE_HT" },
    { WS_NL80211_EXT_FEATURE_BEACON_RATE_VHT, "NL80211_EXT_FEATURE_BEACON_RATE_VHT" },
    { WS_NL80211_EXT_FEATURE_FILS_STA,      "NL80211_EXT_FEATURE_FILS_STA" },
    { WS_NL80211_EXT_FEATURE_MGMT_TX_RANDOM_TA, "NL80211_EXT_FEATURE_MGMT_TX_RANDOM_TA" },
    { WS_NL80211_EXT_FEATURE_MGMT_TX_RANDOM_TA_CONNECTED, "NL80211_EXT_FEATURE_MGMT_TX_RANDOM_TA_CONNECTED" },
    { WS_NL80211_EXT_FEATURE_SCHED_SCAN_RELATIVE_RSSI, "NL80211_EXT_FEATURE_SCHED_SCAN_RELATIVE_RSSI" },
    { WS_NL80211_EXT_FEATURE_CQM_RSSI_LIST, "NL80211_EXT_FEATURE_CQM_RSSI_LIST" },
    { WS_NL80211_EXT_FEATURE_FILS_SK_OFFLOAD, "NL80211_EXT_FEATURE_FILS_SK_OFFLOAD" },
    { WS_NL80211_EXT_FEATURE_4WAY_HANDSHAKE_STA_PSK, "NL80211_EXT_FEATURE_4WAY_HANDSHAKE_STA_PSK" },
    { WS_NL80211_EXT_FEATURE_4WAY_HANDSHAKE_STA_1X, "NL80211_EXT_FEATURE_4WAY_HANDSHAKE_STA_1X" },
    { WS_NL80211_EXT_FEATURE_FILS_MAX_CHANNEL_TIME, "NL80211_EXT_FEATURE_FILS_MAX_CHANNEL_TIME" },
    { WS_NL80211_EXT_FEATURE_ACCEPT_BCAST_PROBE_RESP, "NL80211_EXT_FEATURE_ACCEPT_BCAST_PROBE_RESP" },
    { WS_NL80211_EXT_FEATURE_OCE_PROBE_REQ_HIGH_TX_RATE, "NL80211_EXT_FEATURE_OCE_PROBE_REQ_HIGH_TX_RATE" },
    { WS_NL80211_EXT_FEATURE_OCE_PROBE_REQ_DEFERRAL_SUPPRESSION, "NL80211_EXT_FEATURE_OCE_PROBE_REQ_DEFERRAL_SUPPRESSION" },
    { WS_NL80211_EXT_FEATURE_MFP_OPTIONAL,  "NL80211_EXT_FEATURE_MFP_OPTIONAL" },
    { WS_NL80211_EXT_FEATURE_LOW_SPAN_SCAN, "NL80211_EXT_FEATURE_LOW_SPAN_SCAN" },
    { WS_NL80211_EXT_FEATURE_LOW_POWER_SCAN, "NL80211_EXT_FEATURE_LOW_POWER_SCAN" },
    { WS_NL80211_EXT_FEATURE_HIGH_ACCURACY_SCAN, "NL80211_EXT_FEATURE_HIGH_ACCURACY_SCAN" },
    { WS_NL80211_EXT_FEATURE_DFS_OFFLOAD,   "NL80211_EXT_FEATURE_DFS_OFFLOAD" },
    { WS_NL80211_EXT_FEATURE_CONTROL_PORT_OVER_NL80211, "NL80211_EXT_FEATURE_CONTROL_PORT_OVER_NL80211" },
    { WS_NL80211_EXT_FEATURE_ACK_SIGNAL_SUPPORT, "NL80211_EXT_FEATURE_ACK_SIGNAL_SUPPORT" },
    { WS_NL80211_EXT_FEATURE_TXQS,          "NL80211_EXT_FEATURE_TXQS" },
    { WS_NL80211_EXT_FEATURE_SCAN_RANDOM_SN, "NL80211_EXT_FEATURE_SCAN_RANDOM_SN" },
    { WS_NL80211_EXT_FEATURE_SCAN_MIN_PREQ_CONTENT, "NL80211_EXT_FEATURE_SCAN_MIN_PREQ_CONTENT" },
    { WS_NL80211_EXT_FEATURE_CAN_REPLACE_PTK0, "NL80211_EXT_FEATURE_CAN_REPLACE_PTK0" },
    { WS_NL80211_EXT_FEATURE_ENABLE_FTM_RESPONDER, "NL80211_EXT_FEATURE_ENABLE_FTM_RESPONDER" },
    { WS_NL80211_EXT_FEATURE_AIRTIME_FAIRNESS, "NL80211_EXT_FEATURE_AIRTIME_FAIRNESS" },
    { WS_NL80211_EXT_FEATURE_AP_PMKSA_CACHING, "NL80211_EXT_FEATURE_AP_PMKSA_CACHING" },
    { WS_NL80211_EXT_FEATURE_SCHED_SCAN_BAND_SPECIFIC_RSSI_THOLD, "NL80211_EXT_FEATURE_SCHED_SCAN_BAND_SPECIFIC_RSSI_THOLD" },
    { WS_NL80211_EXT_FEATURE_EXT_KEY_ID,    "NL80211_EXT_FEATURE_EXT_KEY_ID" },
    { WS_NL80211_EXT_FEATURE_STA_TX_PWR,    "NL80211_EXT_FEATURE_STA_TX_PWR" },
    { WS_NL80211_EXT_FEATURE_SAE_OFFLOAD,   "NL80211_EXT_FEATURE_SAE_OFFLOAD" },
    { WS_NL80211_EXT_FEATURE_VLAN_OFFLOAD,  "NL80211_EXT_FEATURE_VLAN_OFFLOAD" },
    { WS_NL80211_EXT_FEATURE_AQL,           "NL80211_EXT_FEATURE_AQL" },
    { WS_NL80211_EXT_FEATURE_BEACON_PROTECTION, "NL80211_EXT_FEATURE_BEACON_PROTECTION" },
    { WS_NL80211_EXT_FEATURE_CONTROL_PORT_NO_PREAUTH, "NL80211_EXT_FEATURE_CONTROL_PORT_NO_PREAUTH" },
    { WS_NL80211_EXT_FEATURE_PROTECTED_TWT, "NL80211_EXT_FEATURE_PROTECTED_TWT" },
    { WS_NL80211_EXT_FEATURE_DEL_IBSS_STA,  "NL80211_EXT_FEATURE_DEL_IBSS_STA" },
    { WS_NL80211_EXT_FEATURE_MULTICAST_REGISTRATIONS, "NL80211_EXT_FEATURE_MULTICAST_REGISTRATIONS" },
    { WS_NL80211_EXT_FEATURE_BEACON_PROTECTION_CLIENT, "NL80211_EXT_FEATURE_BEACON_PROTECTION_CLIENT" },
    { WS_NL80211_EXT_FEATURE_SCAN_FREQ_KHZ, "NL80211_EXT_FEATURE_SCAN_FREQ_KHZ" },
    { WS_NL80211_EXT_FEATURE_CONTROL_PORT_OVER_NL80211_TX_STATUS, "NL80211_EXT_FEATURE_CONTROL_PORT_OVER_NL80211_TX_STATUS" },
    { WS_NL80211_EXT_FEATURE_OPERATING_CHANNEL_VALIDATION, "NL80211_EXT_FEATURE_OPERATING_CHANNEL_VALIDATION" },
    { WS_NL80211_EXT_FEATURE_4WAY_HANDSHAKE_AP_PSK, "NL80211_EXT_FEATURE_4WAY_HANDSHAKE_AP_PSK" },
    { WS_NL80211_EXT_FEATURE_SAE_OFFLOAD_AP, "NL80211_EXT_FEATURE_SAE_OFFLOAD_AP" },
    { WS_NL80211_EXT_FEATURE_FILS_DISCOVERY, "NL80211_EXT_FEATURE_FILS_DISCOVERY" },
    { WS_NL80211_EXT_FEATURE_UNSOL_BCAST_PROBE_RESP, "NL80211_EXT_FEATURE_UNSOL_BCAST_PROBE_RESP" },
    { WS_NL80211_EXT_FEATURE_BEACON_RATE_HE, "NL80211_EXT_FEATURE_BEACON_RATE_HE" },
    { WS_NL80211_EXT_FEATURE_SECURE_LTF,    "NL80211_EXT_FEATURE_SECURE_LTF" },
    { WS_NL80211_EXT_FEATURE_SECURE_RTT,    "NL80211_EXT_FEATURE_SECURE_RTT" },
    { WS_NL80211_EXT_FEATURE_PROT_RANGE_NEGO_AND_MEASURE, "NL80211_EXT_FEATURE_PROT_RANGE_NEGO_AND_MEASURE" },
    { WS_NL80211_EXT_FEATURE_BSS_COLOR,     "NL80211_EXT_FEATURE_BSS_COLOR" },
    { WS_NL80211_EXT_FEATURE_FILS_CRYPTO_OFFLOAD, "NL80211_EXT_FEATURE_FILS_CRYPTO_OFFLOAD" },
    { WS_NL80211_EXT_FEATURE_RADAR_BACKGROUND, "NL80211_EXT_FEATURE_RADAR_BACKGROUND" },
    { WS_NL80211_EXT_FEATURE_POWERED_ADDR_CHANGE, "NL80211_EXT_FEATURE_POWERED_ADDR_CHANGE" },
    { WS_NL80211_EXT_FEATURE_PUNCT,         "NL80211_EXT_FEATURE_PUNCT" },
    { WS_NL80211_EXT_FEATURE_SECURE_NAN,    "NL80211_EXT_FEATURE_SECURE_NAN" },
    { WS_NL80211_EXT_FEATURE_AUTH_AND_DEAUTH_RANDOM_TA, "NL80211_EXT_FEATURE_AUTH_AND_DEAUTH_RANDOM_TA" },
    { WS_NL80211_EXT_FEATURE_OWE_OFFLOAD,   "NL80211_EXT_FEATURE_OWE_OFFLOAD" },
    { WS_NL80211_EXT_FEATURE_OWE_OFFLOAD_AP, "NL80211_EXT_FEATURE_OWE_OFFLOAD_AP" },
    { WS_NL80211_EXT_FEATURE_DFS_CONCURRENT, "NL80211_EXT_FEATURE_DFS_CONCURRENT" },
    { WS_NL80211_EXT_FEATURE_SPP_AMSDU_SUPPORT, "NL80211_EXT_FEATURE_SPP_AMSDU_SUPPORT" },
    { WS_NL80211_EXT_FEATURE_BEACON_RATE_EHT, "NL80211_EXT_FEATURE_BEACON_RATE_EHT" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_ext_feature_index_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_ext_feature_index_vals);

static const value_string ws_nl80211_probe_resp_offload_support_attr_vals[] = {
    { WS_NL80211_PROBE_RESP_OFFLOAD_SUPPORT_WPS, "NL80211_PROBE_RESP_OFFLOAD_SUPPORT_WPS" },
    { WS_NL80211_PROBE_RESP_OFFLOAD_SUPPORT_WPS2, "NL80211_PROBE_RESP_OFFLOAD_SUPPORT_WPS2" },
    { WS_NL80211_PROBE_RESP_OFFLOAD_SUPPORT_P2P, "NL80211_PROBE_RESP_OFFLOAD_SUPPORT_P2P" },
    { WS_NL80211_PROBE_RESP_OFFLOAD_SUPPORT_80211U, "NL80211_PROBE_RESP_OFFLOAD_SUPPORT_80211U" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_probe_resp_offload_support_attr_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_probe_resp_offload_support_attr_vals);

static const value_string ws_nl80211_connect_failed_reason_vals[] = {
    { WS_NL80211_CONN_FAIL_MAX_CLIENTS,     "NL80211_CONN_FAIL_MAX_CLIENTS" },
    { WS_NL80211_CONN_FAIL_BLOCKED_CLIENT,  "NL80211_CONN_FAIL_BLOCKED_CLIENT" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_connect_failed_reason_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_connect_failed_reason_vals);

static const value_string ws_nl80211_timeout_reason_vals[] = {
    { WS_NL80211_TIMEOUT_UNSPECIFIED,       "NL80211_TIMEOUT_UNSPECIFIED" },
    { WS_NL80211_TIMEOUT_SCAN,              "NL80211_TIMEOUT_SCAN" },
    { WS_NL80211_TIMEOUT_AUTH,              "NL80211_TIMEOUT_AUTH" },
    { WS_NL80211_TIMEOUT_ASSOC,             "NL80211_TIMEOUT_ASSOC" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_timeout_reason_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_timeout_reason_vals);

static const value_string ws_nl80211_scan_flags_vals[] = {
    { WS_NL80211_SCAN_FLAG_LOW_PRIORITY,    "NL80211_SCAN_FLAG_LOW_PRIORITY" },
    { WS_NL80211_SCAN_FLAG_FLUSH,           "NL80211_SCAN_FLAG_FLUSH" },
    { WS_NL80211_SCAN_FLAG_AP,              "NL80211_SCAN_FLAG_AP" },
    { WS_NL80211_SCAN_FLAG_RANDOM_ADDR,     "NL80211_SCAN_FLAG_RANDOM_ADDR" },
    { WS_NL80211_SCAN_FLAG_FILS_MAX_CHANNEL_TIME, "NL80211_SCAN_FLAG_FILS_MAX_CHANNEL_TIME" },
    { WS_NL80211_SCAN_FLAG_ACCEPT_BCAST_PROBE_RESP, "NL80211_SCAN_FLAG_ACCEPT_BCAST_PROBE_RESP" },
    { WS_NL80211_SCAN_FLAG_OCE_PROBE_REQ_HIGH_TX_RATE, "NL80211_SCAN_FLAG_OCE_PROBE_REQ_HIGH_TX_RATE" },
    { WS_NL80211_SCAN_FLAG_OCE_PROBE_REQ_DEFERRAL_SUPPRESSION, "NL80211_SCAN_FLAG_OCE_PROBE_REQ_DEFERRAL_SUPPRESSION" },
    { WS_NL80211_SCAN_FLAG_LOW_SPAN,        "NL80211_SCAN_FLAG_LOW_SPAN" },
    { WS_NL80211_SCAN_FLAG_LOW_POWER,       "NL80211_SCAN_FLAG_LOW_POWER" },
    { WS_NL80211_SCAN_FLAG_HIGH_ACCURACY,   "NL80211_SCAN_FLAG_HIGH_ACCURACY" },
    { WS_NL80211_SCAN_FLAG_RANDOM_SN,       "NL80211_SCAN_FLAG_RANDOM_SN" },
    { WS_NL80211_SCAN_FLAG_MIN_PREQ_CONTENT, "NL80211_SCAN_FLAG_MIN_PREQ_CONTENT" },
    { WS_NL80211_SCAN_FLAG_FREQ_KHZ,        "NL80211_SCAN_FLAG_FREQ_KHZ" },
    { WS_NL80211_SCAN_FLAG_COLOCATED_6GHZ,  "NL80211_SCAN_FLAG_COLOCATED_6GHZ" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_scan_flags_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_scan_flags_vals);

static const value_string ws_nl80211_acl_policy_vals[] = {
    { WS_NL80211_ACL_POLICY_ACCEPT_UNLESS_LISTED, "NL80211_ACL_POLICY_ACCEPT_UNLESS_LISTED" },
    { WS_NL80211_ACL_POLICY_DENY_UNLESS_LISTED, "NL80211_ACL_POLICY_DENY_UNLESS_LISTED" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_acl_policy_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_acl_policy_vals);

static const value_string ws_nl80211_smps_mode_vals[] = {
    { WS_NL80211_SMPS_OFF,                  "NL80211_SMPS_OFF" },
    { WS_NL80211_SMPS_STATIC,               "NL80211_SMPS_STATIC" },
    { WS_NL80211_SMPS_DYNAMIC,              "NL80211_SMPS_DYNAMIC" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_smps_mode_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_smps_mode_vals);

static const value_string ws_nl80211_radar_event_vals[] = {
    { WS_NL80211_RADAR_DETECTED,            "NL80211_RADAR_DETECTED" },
    { WS_NL80211_RADAR_CAC_FINISHED,        "NL80211_RADAR_CAC_FINISHED" },
    { WS_NL80211_RADAR_CAC_ABORTED,         "NL80211_RADAR_CAC_ABORTED" },
    { WS_NL80211_RADAR_NOP_FINISHED,        "NL80211_RADAR_NOP_FINISHED" },
    { WS_NL80211_RADAR_PRE_CAC_EXPIRED,     "NL80211_RADAR_PRE_CAC_EXPIRED" },
    { WS_NL80211_RADAR_CAC_STARTED,         "NL80211_RADAR_CAC_STARTED" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_radar_event_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_radar_event_vals);

static const value_string ws_nl80211_dfs_state_vals[] = {
    { WS_NL80211_DFS_USABLE,                "NL80211_DFS_USABLE" },
    { WS_NL80211_DFS_UNAVAILABLE,           "NL80211_DFS_UNAVAILABLE" },
    { WS_NL80211_DFS_AVAILABLE,             "NL80211_DFS_AVAILABLE" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_dfs_state_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_dfs_state_vals);

static const value_string ws_nl80211_protocol_features_vals[] = {
    { WS_NL80211_PROTOCOL_FEATURE_SPLIT_WIPHY_DUMP, "NL80211_PROTOCOL_FEATURE_SPLIT_WIPHY_DUMP" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_protocol_features_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_protocol_features_vals);

static const value_string ws_nl80211_crit_proto_id_vals[] = {
    { WS_NL80211_CRIT_PROTO_UNSPEC,         "NL80211_CRIT_PROTO_UNSPEC" },
    { WS_NL80211_CRIT_PROTO_DHCP,           "NL80211_CRIT_PROTO_DHCP" },
    { WS_NL80211_CRIT_PROTO_EAPOL,          "NL80211_CRIT_PROTO_EAPOL" },
    { WS_NL80211_CRIT_PROTO_APIPA,          "NL80211_CRIT_PROTO_APIPA" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_crit_proto_id_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_crit_proto_id_vals);

static const value_string ws_nl80211_rxmgmt_flags_vals[] = {
    { WS_NL80211_RXMGMT_FLAG_ANSWERED,      "NL80211_RXMGMT_FLAG_ANSWERED" },
    { WS_NL80211_RXMGMT_FLAG_EXTERNAL_AUTH, "NL80211_RXMGMT_FLAG_EXTERNAL_AUTH" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_rxmgmt_flags_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_rxmgmt_flags_vals);

static const value_string ws_nl80211_tdls_peer_capability_vals[] = {
    { WS_NL80211_TDLS_PEER_HT,              "NL80211_TDLS_PEER_HT" },
    { WS_NL80211_TDLS_PEER_VHT,             "NL80211_TDLS_PEER_VHT" },
    { WS_NL80211_TDLS_PEER_WMM,             "NL80211_TDLS_PEER_WMM" },
    { WS_NL80211_TDLS_PEER_HE,              "NL80211_TDLS_PEER_HE" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_tdls_peer_capability_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_tdls_peer_capability_vals);

static const value_string ws_nl80211_sched_scan_plan_vals[] = {
    { WS___NL80211_SCHED_SCAN_PLAN_INVALID, "__NL80211_SCHED_SCAN_PLAN_INVALID" },
    { WS_NL80211_SCHED_SCAN_PLAN_INTERVAL,  "NL80211_SCHED_SCAN_PLAN_INTERVAL" },
    { WS_NL80211_SCHED_SCAN_PLAN_ITERATIONS, "NL80211_SCHED_SCAN_PLAN_ITERATIONS" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_sched_scan_plan_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_sched_scan_plan_vals);

static const value_string ws_nl80211_bss_select_attr_vals[] = {
    { WS___NL80211_BSS_SELECT_ATTR_INVALID, "__NL80211_BSS_SELECT_ATTR_INVALID" },
    { WS_NL80211_BSS_SELECT_ATTR_RSSI,      "NL80211_BSS_SELECT_ATTR_RSSI" },
    { WS_NL80211_BSS_SELECT_ATTR_BAND_PREF, "NL80211_BSS_SELECT_ATTR_BAND_PREF" },
    { WS_NL80211_BSS_SELECT_ATTR_RSSI_ADJUST, "NL80211_BSS_SELECT_ATTR_RSSI_ADJUST" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_bss_select_attr_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_bss_select_attr_vals);

static const value_string ws_nl80211_nan_function_type_vals[] = {
    { WS_NL80211_NAN_FUNC_PUBLISH,          "NL80211_NAN_FUNC_PUBLISH" },
    { WS_NL80211_NAN_FUNC_SUBSCRIBE,        "NL80211_NAN_FUNC_SUBSCRIBE" },
    { WS_NL80211_NAN_FUNC_FOLLOW_UP,        "NL80211_NAN_FUNC_FOLLOW_UP" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_nan_function_type_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_nan_function_type_vals);

static const value_string ws_nl80211_nan_publish_type_vals[] = {
    { WS_NL80211_NAN_SOLICITED_PUBLISH,     "NL80211_NAN_SOLICITED_PUBLISH" },
    { WS_NL80211_NAN_UNSOLICITED_PUBLISH,   "NL80211_NAN_UNSOLICITED_PUBLISH" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_nan_publish_type_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_nan_publish_type_vals);

static const value_string ws_nl80211_nan_func_term_reason_vals[] = {
    { WS_NL80211_NAN_FUNC_TERM_REASON_USER_REQUEST, "NL80211_NAN_FUNC_TERM_REASON_USER_REQUEST" },
    { WS_NL80211_NAN_FUNC_TERM_REASON_TTL_EXPIRED, "NL80211_NAN_FUNC_TERM_REASON_TTL_EXPIRED" },
    { WS_NL80211_NAN_FUNC_TERM_REASON_ERROR, "NL80211_NAN_FUNC_TERM_REASON_ERROR" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_nan_func_term_reason_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_nan_func_term_reason_vals);

static const value_string ws_nl80211_nan_func_attributes_vals[] = {
    { WS___NL80211_NAN_FUNC_INVALID,        "__NL80211_NAN_FUNC_INVALID" },
    { WS_NL80211_NAN_FUNC_TYPE,             "NL80211_NAN_FUNC_TYPE" },
    { WS_NL80211_NAN_FUNC_SERVICE_ID,       "NL80211_NAN_FUNC_SERVICE_ID" },
    { WS_NL80211_NAN_FUNC_PUBLISH_TYPE,     "NL80211_NAN_FUNC_PUBLISH_TYPE" },
    { WS_NL80211_NAN_FUNC_PUBLISH_BCAST,    "NL80211_NAN_FUNC_PUBLISH_BCAST" },
    { WS_NL80211_NAN_FUNC_SUBSCRIBE_ACTIVE, "NL80211_NAN_FUNC_SUBSCRIBE_ACTIVE" },
    { WS_NL80211_NAN_FUNC_FOLLOW_UP_ID,     "NL80211_NAN_FUNC_FOLLOW_UP_ID" },
    { WS_NL80211_NAN_FUNC_FOLLOW_UP_REQ_ID, "NL80211_NAN_FUNC_FOLLOW_UP_REQ_ID" },
    { WS_NL80211_NAN_FUNC_FOLLOW_UP_DEST,   "NL80211_NAN_FUNC_FOLLOW_UP_DEST" },
    { WS_NL80211_NAN_FUNC_CLOSE_RANGE,      "NL80211_NAN_FUNC_CLOSE_RANGE" },
    { WS_NL80211_NAN_FUNC_TTL,              "NL80211_NAN_FUNC_TTL" },
    { WS_NL80211_NAN_FUNC_SERVICE_INFO,     "NL80211_NAN_FUNC_SERVICE_INFO" },
    { WS_NL80211_NAN_FUNC_SRF,              "NL80211_NAN_FUNC_SRF" },
    { WS_NL80211_NAN_FUNC_RX_MATCH_FILTER,  "NL80211_NAN_FUNC_RX_MATCH_FILTER" },
    { WS_NL80211_NAN_FUNC_TX_MATCH_FILTER,  "NL80211_NAN_FUNC_TX_MATCH_FILTER" },
    { WS_NL80211_NAN_FUNC_INSTANCE_ID,      "NL80211_NAN_FUNC_INSTANCE_ID" },
    { WS_NL80211_NAN_FUNC_TERM_REASON,      "NL80211_NAN_FUNC_TERM_REASON" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_nan_func_attributes_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_nan_func_attributes_vals);

static const value_string ws_nl80211_nan_srf_attributes_vals[] = {
    { WS___NL80211_NAN_SRF_INVALID,         "__NL80211_NAN_SRF_INVALID" },
    { WS_NL80211_NAN_SRF_INCLUDE,           "NL80211_NAN_SRF_INCLUDE" },
    { WS_NL80211_NAN_SRF_BF,                "NL80211_NAN_SRF_BF" },
    { WS_NL80211_NAN_SRF_BF_IDX,            "NL80211_NAN_SRF_BF_IDX" },
    { WS_NL80211_NAN_SRF_MAC_ADDRS,         "NL80211_NAN_SRF_MAC_ADDRS" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_nan_srf_attributes_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_nan_srf_attributes_vals);

static const value_string ws_nl80211_nan_match_attributes_vals[] = {
    { WS___NL80211_NAN_MATCH_INVALID,       "__NL80211_NAN_MATCH_INVALID" },
    { WS_NL80211_NAN_MATCH_FUNC_LOCAL,      "NL80211_NAN_MATCH_FUNC_LOCAL" },
    { WS_NL80211_NAN_MATCH_FUNC_PEER,       "NL80211_NAN_MATCH_FUNC_PEER" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_nan_match_attributes_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_nan_match_attributes_vals);

static const value_string ws_nl80211_external_auth_action_vals[] = {
    { WS_NL80211_EXTERNAL_AUTH_START,       "NL80211_EXTERNAL_AUTH_START" },
    { WS_NL80211_EXTERNAL_AUTH_ABORT,       "NL80211_EXTERNAL_AUTH_ABORT" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_external_auth_action_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_external_auth_action_vals);

static const value_string ws_nl80211_ftm_responder_attributes_vals[] = {
    { WS___NL80211_FTM_RESP_ATTR_INVALID,   "__NL80211_FTM_RESP_ATTR_INVALID" },
    { WS_NL80211_FTM_RESP_ATTR_ENABLED,     "NL80211_FTM_RESP_ATTR_ENABLED" },
    { WS_NL80211_FTM_RESP_ATTR_LCI,         "NL80211_FTM_RESP_ATTR_LCI" },
    { WS_NL80211_FTM_RESP_ATTR_CIVICLOC,    "NL80211_FTM_RESP_ATTR_CIVICLOC" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_ftm_responder_attributes_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_ftm_responder_attributes_vals);

static const value_string ws_nl80211_ftm_responder_stats_vals[] = {
    { WS___NL80211_FTM_STATS_INVALID,       "__NL80211_FTM_STATS_INVALID" },
    { WS_NL80211_FTM_STATS_SUCCESS_NUM,     "NL80211_FTM_STATS_SUCCESS_NUM" },
    { WS_NL80211_FTM_STATS_PARTIAL_NUM,     "NL80211_FTM_STATS_PARTIAL_NUM" },
    { WS_NL80211_FTM_STATS_FAILED_NUM,      "NL80211_FTM_STATS_FAILED_NUM" },
    { WS_NL80211_FTM_STATS_ASAP_NUM,        "NL80211_FTM_STATS_ASAP_NUM" },
    { WS_NL80211_FTM_STATS_NON_ASAP_NUM,    "NL80211_FTM_STATS_NON_ASAP_NUM" },
    { WS_NL80211_FTM_STATS_TOTAL_DURATION_MSEC, "NL80211_FTM_STATS_TOTAL_DURATION_MSEC" },
    { WS_NL80211_FTM_STATS_UNKNOWN_TRIGGERS_NUM, "NL80211_FTM_STATS_UNKNOWN_TRIGGERS_NUM" },
    { WS_NL80211_FTM_STATS_RESCHEDULE_REQUESTS_NUM, "NL80211_FTM_STATS_RESCHEDULE_REQUESTS_NUM" },
    { WS_NL80211_FTM_STATS_OUT_OF_WINDOW_TRIGGERS_NUM, "NL80211_FTM_STATS_OUT_OF_WINDOW_TRIGGERS_NUM" },
    { WS_NL80211_FTM_STATS_PAD,             "NL80211_FTM_STATS_PAD" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_ftm_responder_stats_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_ftm_responder_stats_vals);

static const value_string ws_nl80211_preamble_vals[] = {
    { WS_NL80211_PREAMBLE_LEGACY,           "NL80211_PREAMBLE_LEGACY" },
    { WS_NL80211_PREAMBLE_HT,               "NL80211_PREAMBLE_HT" },
    { WS_NL80211_PREAMBLE_VHT,              "NL80211_PREAMBLE_VHT" },
    { WS_NL80211_PREAMBLE_DMG,              "NL80211_PREAMBLE_DMG" },
    { WS_NL80211_PREAMBLE_HE,               "NL80211_PREAMBLE_HE" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_preamble_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_preamble_vals);

static const value_string ws_nl80211_peer_measurement_type_vals[] = {
    { WS_NL80211_PMSR_TYPE_INVALID,         "NL80211_PMSR_TYPE_INVALID" },
    { WS_NL80211_PMSR_TYPE_FTM,             "NL80211_PMSR_TYPE_FTM" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_peer_measurement_type_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_peer_measurement_type_vals);

static const value_string ws_nl80211_peer_measurement_status_vals[] = {
    { WS_NL80211_PMSR_STATUS_SUCCESS,       "NL80211_PMSR_STATUS_SUCCESS" },
    { WS_NL80211_PMSR_STATUS_REFUSED,       "NL80211_PMSR_STATUS_REFUSED" },
    { WS_NL80211_PMSR_STATUS_TIMEOUT,       "NL80211_PMSR_STATUS_TIMEOUT" },
    { WS_NL80211_PMSR_STATUS_FAILURE,       "NL80211_PMSR_STATUS_FAILURE" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_peer_measurement_status_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_peer_measurement_status_vals);

static const value_string ws_nl80211_peer_measurement_req_vals[] = {
    { WS___NL80211_PMSR_REQ_ATTR_INVALID,   "__NL80211_PMSR_REQ_ATTR_INVALID" },
    { WS_NL80211_PMSR_REQ_ATTR_DATA,        "NL80211_PMSR_REQ_ATTR_DATA" },
    { WS_NL80211_PMSR_REQ_ATTR_GET_AP_TSF,  "NL80211_PMSR_REQ_ATTR_GET_AP_TSF" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_peer_measurement_req_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_peer_measurement_req_vals);

static const value_string ws_nl80211_peer_measurement_resp_vals[] = {
    { WS___NL80211_PMSR_RESP_ATTR_INVALID,  "__NL80211_PMSR_RESP_ATTR_INVALID" },
    { WS_NL80211_PMSR_RESP_ATTR_DATA,       "NL80211_PMSR_RESP_ATTR_DATA" },
    { WS_NL80211_PMSR_RESP_ATTR_STATUS,     "NL80211_PMSR_RESP_ATTR_STATUS" },
    { WS_NL80211_PMSR_RESP_ATTR_HOST_TIME,  "NL80211_PMSR_RESP_ATTR_HOST_TIME" },
    { WS_NL80211_PMSR_RESP_ATTR_AP_TSF,     "NL80211_PMSR_RESP_ATTR_AP_TSF" },
    { WS_NL80211_PMSR_RESP_ATTR_FINAL,      "NL80211_PMSR_RESP_ATTR_FINAL" },
    { WS_NL80211_PMSR_RESP_ATTR_PAD,        "NL80211_PMSR_RESP_ATTR_PAD" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_peer_measurement_resp_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_peer_measurement_resp_vals);

static const value_string ws_nl80211_peer_measurement_peer_attrs_vals[] = {
    { WS___NL80211_PMSR_PEER_ATTR_INVALID,  "__NL80211_PMSR_PEER_ATTR_INVALID" },
    { WS_NL80211_PMSR_PEER_ATTR_ADDR,       "NL80211_PMSR_PEER_ATTR_ADDR" },
    { WS_NL80211_PMSR_PEER_ATTR_CHAN,       "NL80211_PMSR_PEER_ATTR_CHAN" },
    { WS_NL80211_PMSR_PEER_ATTR_REQ,        "NL80211_PMSR_PEER_ATTR_REQ" },
    { WS_NL80211_PMSR_PEER_ATTR_RESP,       "NL80211_PMSR_PEER_ATTR_RESP" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_peer_measurement_peer_attrs_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_peer_measurement_peer_attrs_vals);

static const value_string ws_nl80211_peer_measurement_attrs_vals[] = {
    { WS___NL80211_PMSR_ATTR_INVALID,       "__NL80211_PMSR_ATTR_INVALID" },
    { WS_NL80211_PMSR_ATTR_MAX_PEERS,       "NL80211_PMSR_ATTR_MAX_PEERS" },
    { WS_NL80211_PMSR_ATTR_REPORT_AP_TSF,   "NL80211_PMSR_ATTR_REPORT_AP_TSF" },
    { WS_NL80211_PMSR_ATTR_RANDOMIZE_MAC_ADDR, "NL80211_PMSR_ATTR_RANDOMIZE_MAC_ADDR" },
    { WS_NL80211_PMSR_ATTR_TYPE_CAPA,       "NL80211_PMSR_ATTR_TYPE_CAPA" },
    { WS_NL80211_PMSR_ATTR_PEERS,           "NL80211_PMSR_ATTR_PEERS" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_peer_measurement_attrs_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_peer_measurement_attrs_vals);

static const value_string ws_nl80211_peer_measurement_ftm_capa_vals[] = {
    { WS___NL80211_PMSR_FTM_CAPA_ATTR_INVALID, "__NL80211_PMSR_FTM_CAPA_ATTR_INVALID" },
    { WS_NL80211_PMSR_FTM_CAPA_ATTR_ASAP,   "NL80211_PMSR_FTM_CAPA_ATTR_ASAP" },
    { WS_NL80211_PMSR_FTM_CAPA_ATTR_NON_ASAP, "NL80211_PMSR_FTM_CAPA_ATTR_NON_ASAP" },
    { WS_NL80211_PMSR_FTM_CAPA_ATTR_REQ_LCI, "NL80211_PMSR_FTM_CAPA_ATTR_REQ_LCI" },
    { WS_NL80211_PMSR_FTM_CAPA_ATTR_REQ_CIVICLOC, "NL80211_PMSR_FTM_CAPA_ATTR_REQ_CIVICLOC" },
    { WS_NL80211_PMSR_FTM_CAPA_ATTR_PREAMBLES, "NL80211_PMSR_FTM_CAPA_ATTR_PREAMBLES" },
    { WS_NL80211_PMSR_FTM_CAPA_ATTR_BANDWIDTHS, "NL80211_PMSR_FTM_CAPA_ATTR_BANDWIDTHS" },
    { WS_NL80211_PMSR_FTM_CAPA_ATTR_MAX_BURSTS_EXPONENT, "NL80211_PMSR_FTM_CAPA_ATTR_MAX_BURSTS_EXPONENT" },
    { WS_NL80211_PMSR_FTM_CAPA_ATTR_MAX_FTMS_PER_BURST, "NL80211_PMSR_FTM_CAPA_ATTR_MAX_FTMS_PER_BURST" },
    { WS_NL80211_PMSR_FTM_CAPA_ATTR_TRIGGER_BASED, "NL80211_PMSR_FTM_CAPA_ATTR_TRIGGER_BASED" },
    { WS_NL80211_PMSR_FTM_CAPA_ATTR_NON_TRIGGER_BASED, "NL80211_PMSR_FTM_CAPA_ATTR_NON_TRIGGER_BASED" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_peer_measurement_ftm_capa_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_peer_measurement_ftm_capa_vals);

static const value_string ws_nl80211_peer_measurement_ftm_req_vals[] = {
    { WS___NL80211_PMSR_FTM_REQ_ATTR_INVALID, "__NL80211_PMSR_FTM_REQ_ATTR_INVALID" },
    { WS_NL80211_PMSR_FTM_REQ_ATTR_ASAP,    "NL80211_PMSR_FTM_REQ_ATTR_ASAP" },
    { WS_NL80211_PMSR_FTM_REQ_ATTR_PREAMBLE, "NL80211_PMSR_FTM_REQ_ATTR_PREAMBLE" },
    { WS_NL80211_PMSR_FTM_REQ_ATTR_NUM_BURSTS_EXP, "NL80211_PMSR_FTM_REQ_ATTR_NUM_BURSTS_EXP" },
    { WS_NL80211_PMSR_FTM_REQ_ATTR_BURST_PERIOD, "NL80211_PMSR_FTM_REQ_ATTR_BURST_PERIOD" },
    { WS_NL80211_PMSR_FTM_REQ_ATTR_BURST_DURATION, "NL80211_PMSR_FTM_REQ_ATTR_BURST_DURATION" },
    { WS_NL80211_PMSR_FTM_REQ_ATTR_FTMS_PER_BURST, "NL80211_PMSR_FTM_REQ_ATTR_FTMS_PER_BURST" },
    { WS_NL80211_PMSR_FTM_REQ_ATTR_NUM_FTMR_RETRIES, "NL80211_PMSR_FTM_REQ_ATTR_NUM_FTMR_RETRIES" },
    { WS_NL80211_PMSR_FTM_REQ_ATTR_REQUEST_LCI, "NL80211_PMSR_FTM_REQ_ATTR_REQUEST_LCI" },
    { WS_NL80211_PMSR_FTM_REQ_ATTR_REQUEST_CIVICLOC, "NL80211_PMSR_FTM_REQ_ATTR_REQUEST_CIVICLOC" },
    { WS_NL80211_PMSR_FTM_REQ_ATTR_TRIGGER_BASED, "NL80211_PMSR_FTM_REQ_ATTR_TRIGGER_BASED" },
    { WS_NL80211_PMSR_FTM_REQ_ATTR_NON_TRIGGER_BASED, "NL80211_PMSR_FTM_REQ_ATTR_NON_TRIGGER_BASED" },
    { WS_NL80211_PMSR_FTM_REQ_ATTR_LMR_FEEDBACK, "NL80211_PMSR_FTM_REQ_ATTR_LMR_FEEDBACK" },
    { WS_NL80211_PMSR_FTM_REQ_ATTR_BSS_COLOR, "NL80211_PMSR_FTM_REQ_ATTR_BSS_COLOR" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_peer_measurement_ftm_req_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_peer_measurement_ftm_req_vals);

static const value_string ws_nl80211_peer_measurement_ftm_failure_reasons_vals[] = {
    { WS_NL80211_PMSR_FTM_FAILURE_UNSPECIFIED, "NL80211_PMSR_FTM_FAILURE_UNSPECIFIED" },
    { WS_NL80211_PMSR_FTM_FAILURE_NO_RESPONSE, "NL80211_PMSR_FTM_FAILURE_NO_RESPONSE" },
    { WS_NL80211_PMSR_FTM_FAILURE_REJECTED, "NL80211_PMSR_FTM_FAILURE_REJECTED" },
    { WS_NL80211_PMSR_FTM_FAILURE_WRONG_CHANNEL, "NL80211_PMSR_FTM_FAILURE_WRONG_CHANNEL" },
    { WS_NL80211_PMSR_FTM_FAILURE_PEER_NOT_CAPABLE, "NL80211_PMSR_FTM_FAILURE_PEER_NOT_CAPABLE" },
    { WS_NL80211_PMSR_FTM_FAILURE_INVALID_TIMESTAMP, "NL80211_PMSR_FTM_FAILURE_INVALID_TIMESTAMP" },
    { WS_NL80211_PMSR_FTM_FAILURE_PEER_BUSY, "NL80211_PMSR_FTM_FAILURE_PEER_BUSY" },
    { WS_NL80211_PMSR_FTM_FAILURE_BAD_CHANGED_PARAMS, "NL80211_PMSR_FTM_FAILURE_BAD_CHANGED_PARAMS" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_peer_measurement_ftm_failure_reasons_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_peer_measurement_ftm_failure_reasons_vals);

static const value_string ws_nl80211_peer_measurement_ftm_resp_vals[] = {
    { WS___NL80211_PMSR_FTM_RESP_ATTR_INVALID, "__NL80211_PMSR_FTM_RESP_ATTR_INVALID" },
    { WS_NL80211_PMSR_FTM_RESP_ATTR_FAIL_REASON, "NL80211_PMSR_FTM_RESP_ATTR_FAIL_REASON" },
    { WS_NL80211_PMSR_FTM_RESP_ATTR_BURST_INDEX, "NL80211_PMSR_FTM_RESP_ATTR_BURST_INDEX" },
    { WS_NL80211_PMSR_FTM_RESP_ATTR_NUM_FTMR_ATTEMPTS, "NL80211_PMSR_FTM_RESP_ATTR_NUM_FTMR_ATTEMPTS" },
    { WS_NL80211_PMSR_FTM_RESP_ATTR_NUM_FTMR_SUCCESSES, "NL80211_PMSR_FTM_RESP_ATTR_NUM_FTMR_SUCCESSES" },
    { WS_NL80211_PMSR_FTM_RESP_ATTR_BUSY_RETRY_TIME, "NL80211_PMSR_FTM_RESP_ATTR_BUSY_RETRY_TIME" },
    { WS_NL80211_PMSR_FTM_RESP_ATTR_NUM_BURSTS_EXP, "NL80211_PMSR_FTM_RESP_ATTR_NUM_BURSTS_EXP" },
    { WS_NL80211_PMSR_FTM_RESP_ATTR_BURST_DURATION, "NL80211_PMSR_FTM_RESP_ATTR_BURST_DURATION" },
    { WS_NL80211_PMSR_FTM_RESP_ATTR_FTMS_PER_BURST, "NL80211_PMSR_FTM_RESP_ATTR_FTMS_PER_BURST" },
    { WS_NL80211_PMSR_FTM_RESP_ATTR_RSSI_AVG, "NL80211_PMSR_FTM_RESP_ATTR_RSSI_AVG" },
    { WS_NL80211_PMSR_FTM_RESP_ATTR_RSSI_SPREAD, "NL80211_PMSR_FTM_RESP_ATTR_RSSI_SPREAD" },
    { WS_NL80211_PMSR_FTM_RESP_ATTR_TX_RATE, "NL80211_PMSR_FTM_RESP_ATTR_TX_RATE" },
    { WS_NL80211_PMSR_FTM_RESP_ATTR_RX_RATE, "NL80211_PMSR_FTM_RESP_ATTR_RX_RATE" },
    { WS_NL80211_PMSR_FTM_RESP_ATTR_RTT_AVG, "NL80211_PMSR_FTM_RESP_ATTR_RTT_AVG" },
    { WS_NL80211_PMSR_FTM_RESP_ATTR_RTT_VARIANCE, "NL80211_PMSR_FTM_RESP_ATTR_RTT_VARIANCE" },
    { WS_NL80211_PMSR_FTM_RESP_ATTR_RTT_SPREAD, "NL80211_PMSR_FTM_RESP_ATTR_RTT_SPREAD" },
    { WS_NL80211_PMSR_FTM_RESP_ATTR_DIST_AVG, "NL80211_PMSR_FTM_RESP_ATTR_DIST_AVG" },
    { WS_NL80211_PMSR_FTM_RESP_ATTR_DIST_VARIANCE, "NL80211_PMSR_FTM_RESP_ATTR_DIST_VARIANCE" },
    { WS_NL80211_PMSR_FTM_RESP_ATTR_DIST_SPREAD, "NL80211_PMSR_FTM_RESP_ATTR_DIST_SPREAD" },
    { WS_NL80211_PMSR_FTM_RESP_ATTR_LCI,    "NL80211_PMSR_FTM_RESP_ATTR_LCI" },
    { WS_NL80211_PMSR_FTM_RESP_ATTR_CIVICLOC, "NL80211_PMSR_FTM_RESP_ATTR_CIVICLOC" },
    { WS_NL80211_PMSR_FTM_RESP_ATTR_PAD,    "NL80211_PMSR_FTM_RESP_ATTR_PAD" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_peer_measurement_ftm_resp_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_peer_measurement_ftm_resp_vals);

static const value_string ws_nl80211_obss_pd_attributes_vals[] = {
    { WS___NL80211_HE_OBSS_PD_ATTR_INVALID, "__NL80211_HE_OBSS_PD_ATTR_INVALID" },
    { WS_NL80211_HE_OBSS_PD_ATTR_MIN_OFFSET, "NL80211_HE_OBSS_PD_ATTR_MIN_OFFSET" },
    { WS_NL80211_HE_OBSS_PD_ATTR_MAX_OFFSET, "NL80211_HE_OBSS_PD_ATTR_MAX_OFFSET" },
    { WS_NL80211_HE_OBSS_PD_ATTR_NON_SRG_MAX_OFFSET, "NL80211_HE_OBSS_PD_ATTR_NON_SRG_MAX_OFFSET" },
    { WS_NL80211_HE_OBSS_PD_ATTR_BSS_COLOR_BITMAP, "NL80211_HE_OBSS_PD_ATTR_BSS_COLOR_BITMAP" },
    { WS_NL80211_HE_OBSS_PD_ATTR_PARTIAL_BSSID_BITMAP, "NL80211_HE_OBSS_PD_ATTR_PARTIAL_BSSID_BITMAP" },
    { WS_NL80211_HE_OBSS_PD_ATTR_SR_CTRL,   "NL80211_HE_OBSS_PD_ATTR_SR_CTRL" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_obss_pd_attributes_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_obss_pd_attributes_vals);

static const value_string ws_nl80211_bss_color_attributes_vals[] = {
    { WS___NL80211_HE_BSS_COLOR_ATTR_INVALID, "__NL80211_HE_BSS_COLOR_ATTR_INVALID" },
    { WS_NL80211_HE_BSS_COLOR_ATTR_COLOR,   "NL80211_HE_BSS_COLOR_ATTR_COLOR" },
    { WS_NL80211_HE_BSS_COLOR_ATTR_DISABLED, "NL80211_HE_BSS_COLOR_ATTR_DISABLED" },
    { WS_NL80211_HE_BSS_COLOR_ATTR_PARTIAL, "NL80211_HE_BSS_COLOR_ATTR_PARTIAL" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_bss_color_attributes_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_bss_color_attributes_vals);

static const value_string ws_nl80211_iftype_akm_attributes_vals[] = {
    { WS___NL80211_IFTYPE_AKM_ATTR_INVALID, "__NL80211_IFTYPE_AKM_ATTR_INVALID" },
    { WS_NL80211_IFTYPE_AKM_ATTR_IFTYPES,   "NL80211_IFTYPE_AKM_ATTR_IFTYPES" },
    { WS_NL80211_IFTYPE_AKM_ATTR_SUITES,    "NL80211_IFTYPE_AKM_ATTR_SUITES" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_iftype_akm_attributes_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_iftype_akm_attributes_vals);

static const value_string ws_nl80211_fils_discovery_attributes_vals[] = {
    { WS___NL80211_FILS_DISCOVERY_ATTR_INVALID, "__NL80211_FILS_DISCOVERY_ATTR_INVALID" },
    { WS_NL80211_FILS_DISCOVERY_ATTR_INT_MIN, "NL80211_FILS_DISCOVERY_ATTR_INT_MIN" },
    { WS_NL80211_FILS_DISCOVERY_ATTR_INT_MAX, "NL80211_FILS_DISCOVERY_ATTR_INT_MAX" },
    { WS_NL80211_FILS_DISCOVERY_ATTR_TMPL,  "NL80211_FILS_DISCOVERY_ATTR_TMPL" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_fils_discovery_attributes_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_fils_discovery_attributes_vals);

static const value_string ws_nl80211_unsol_bcast_probe_resp_attributes_vals[] = {
    { WS___NL80211_UNSOL_BCAST_PROBE_RESP_ATTR_INVALID, "__NL80211_UNSOL_BCAST_PROBE_RESP_ATTR_INVALID" },
    { WS_NL80211_UNSOL_BCAST_PROBE_RESP_ATTR_INT, "NL80211_UNSOL_BCAST_PROBE_RESP_ATTR_INT" },
    { WS_NL80211_UNSOL_BCAST_PROBE_RESP_ATTR_TMPL, "NL80211_UNSOL_BCAST_PROBE_RESP_ATTR_TMPL" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_unsol_bcast_probe_resp_attributes_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_unsol_bcast_probe_resp_attributes_vals);

static const value_string ws_nl80211_sae_pwe_mechanism_vals[] = {
    { WS_NL80211_SAE_PWE_UNSPECIFIED,       "NL80211_SAE_PWE_UNSPECIFIED" },
    { WS_NL80211_SAE_PWE_HUNT_AND_PECK,     "NL80211_SAE_PWE_HUNT_AND_PECK" },
    { WS_NL80211_SAE_PWE_HASH_TO_ELEMENT,   "NL80211_SAE_PWE_HASH_TO_ELEMENT" },
    { WS_NL80211_SAE_PWE_BOTH,              "NL80211_SAE_PWE_BOTH" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_sae_pwe_mechanism_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_sae_pwe_mechanism_vals);

static const value_string ws_nl80211_sar_type_vals[] = {
    { WS_NL80211_SAR_TYPE_POWER,            "NL80211_SAR_TYPE_POWER" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_sar_type_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_sar_type_vals);

static const value_string ws_nl80211_sar_attrs_vals[] = {
    { WS___NL80211_SAR_ATTR_INVALID,        "__NL80211_SAR_ATTR_INVALID" },
    { WS_NL80211_SAR_ATTR_TYPE,             "NL80211_SAR_ATTR_TYPE" },
    { WS_NL80211_SAR_ATTR_SPECS,            "NL80211_SAR_ATTR_SPECS" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_sar_attrs_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_sar_attrs_vals);

static const value_string ws_nl80211_sar_specs_attrs_vals[] = {
    { WS___NL80211_SAR_ATTR_SPECS_INVALID,  "__NL80211_SAR_ATTR_SPECS_INVALID" },
    { WS_NL80211_SAR_ATTR_SPECS_POWER,      "NL80211_SAR_ATTR_SPECS_POWER" },
    { WS_NL80211_SAR_ATTR_SPECS_RANGE_INDEX, "NL80211_SAR_ATTR_SPECS_RANGE_INDEX" },
    { WS_NL80211_SAR_ATTR_SPECS_START_FREQ, "NL80211_SAR_ATTR_SPECS_START_FREQ" },
    { WS_NL80211_SAR_ATTR_SPECS_END_FREQ,   "NL80211_SAR_ATTR_SPECS_END_FREQ" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_sar_specs_attrs_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_sar_specs_attrs_vals);

static const value_string ws_nl80211_mbssid_config_attributes_vals[] = {
    { WS___NL80211_MBSSID_CONFIG_ATTR_INVALID, "__NL80211_MBSSID_CONFIG_ATTR_INVALID" },
    { WS_NL80211_MBSSID_CONFIG_ATTR_MAX_INTERFACES, "NL80211_MBSSID_CONFIG_ATTR_MAX_INTERFACES" },
    { WS_NL80211_MBSSID_CONFIG_ATTR_MAX_EMA_PROFILE_PERIODICITY, "NL80211_MBSSID_CONFIG_ATTR_MAX_EMA_PROFILE_PERIODICITY" },
    { WS_NL80211_MBSSID_CONFIG_ATTR_INDEX,  "NL80211_MBSSID_CONFIG_ATTR_INDEX" },
    { WS_NL80211_MBSSID_CONFIG_ATTR_TX_IFINDEX, "NL80211_MBSSID_CONFIG_ATTR_TX_IFINDEX" },
    { WS_NL80211_MBSSID_CONFIG_ATTR_EMA,    "NL80211_MBSSID_CONFIG_ATTR_EMA" },
    { WS_NL80211_MBSSID_CONFIG_ATTR_TX_LINK_ID, "NL80211_MBSSID_CONFIG_ATTR_TX_LINK_ID" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_mbssid_config_attributes_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_mbssid_config_attributes_vals);

static const value_string ws_nl80211_ap_settings_flags_vals[] = {
    { WS_NL80211_AP_SETTINGS_EXTERNAL_AUTH_SUPPORT, "NL80211_AP_SETTINGS_EXTERNAL_AUTH_SUPPORT" },
    { WS_NL80211_AP_SETTINGS_SA_QUERY_OFFLOAD_SUPPORT, "NL80211_AP_SETTINGS_SA_QUERY_OFFLOAD_SUPPORT" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_ap_settings_flags_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_ap_settings_flags_vals);

static const value_string ws_nl80211_wiphy_radio_attrs_vals[] = {
    { WS___NL80211_WIPHY_RADIO_ATTR_INVALID, "__NL80211_WIPHY_RADIO_ATTR_INVALID" },
    { WS_NL80211_WIPHY_RADIO_ATTR_INDEX,    "NL80211_WIPHY_RADIO_ATTR_INDEX" },
    { WS_NL80211_WIPHY_RADIO_ATTR_FREQ_RANGE, "NL80211_WIPHY_RADIO_ATTR_FREQ_RANGE" },
    { WS_NL80211_WIPHY_RADIO_ATTR_INTERFACE_COMBINATION, "NL80211_WIPHY_RADIO_ATTR_INTERFACE_COMBINATION" },
    { WS_NL80211_WIPHY_RADIO_ATTR_ANTENNA_MASK, "NL80211_WIPHY_RADIO_ATTR_ANTENNA_MASK" },
    { WS_NL80211_WIPHY_RADIO_ATTR_RTS_THRESHOLD, "NL80211_WIPHY_RADIO_ATTR_RTS_THRESHOLD" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_wiphy_radio_attrs_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_wiphy_radio_attrs_vals);

static const value_string ws_nl80211_wiphy_radio_freq_range_vals[] = {
    { WS___NL80211_WIPHY_RADIO_FREQ_ATTR_INVALID, "__NL80211_WIPHY_RADIO_FREQ_ATTR_INVALID" },
    { WS_NL80211_WIPHY_RADIO_FREQ_ATTR_START, "NL80211_WIPHY_RADIO_FREQ_ATTR_START" },
    { WS_NL80211_WIPHY_RADIO_FREQ_ATTR_END, "NL80211_WIPHY_RADIO_FREQ_ATTR_END" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_wiphy_radio_freq_range_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_wiphy_radio_freq_range_vals);

static const value_string ws_nl80211_s1g_short_beacon_attrs_vals[] = {
    { WS___NL80211_S1G_SHORT_BEACON_ATTR_INVALID, "__NL80211_S1G_SHORT_BEACON_ATTR_INVALID" },
    { WS_NL80211_S1G_SHORT_BEACON_ATTR_HEAD, "NL80211_S1G_SHORT_BEACON_ATTR_HEAD" },
    { WS_NL80211_S1G_SHORT_BEACON_ATTR_TAIL, "NL80211_S1G_SHORT_BEACON_ATTR_TAIL" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_s1g_short_beacon_attrs_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_s1g_short_beacon_attrs_vals);

static const value_string ws_nl80211_nan_capabilities_vals[] = {
    { WS___NL80211_NAN_CAPABILITIES_INVALID, "__NL80211_NAN_CAPABILITIES_INVALID" },
    { WS_NL80211_NAN_CAPA_CONFIGURABLE_SYNC, "NL80211_NAN_CAPA_CONFIGURABLE_SYNC" },
    { WS_NL80211_NAN_CAPA_USERSPACE_DE,     "NL80211_NAN_CAPA_USERSPACE_DE" },
    { WS_NL80211_NAN_CAPA_OP_MODE,          "NL80211_NAN_CAPA_OP_MODE" },
    { WS_NL80211_NAN_CAPA_NUM_ANTENNAS,     "NL80211_NAN_CAPA_NUM_ANTENNAS" },
    { WS_NL80211_NAN_CAPA_MAX_CHANNEL_SWITCH_TIME, "NL80211_NAN_CAPA_MAX_CHANNEL_SWITCH_TIME" },
    { WS_NL80211_NAN_CAPA_CAPABILITIES,     "NL80211_NAN_CAPA_CAPABILITIES" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_nan_capabilities_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_nan_capabilities_vals);

static int hf_nl80211_commands;
static int hf_nl80211_attrs;
static int hf_nl80211_iftype;
static int hf_nl80211_sta_flags;
static int hf_nl80211_sta_p2p_ps_status;
static int hf_nl80211_he_gi;
static int hf_nl80211_he_ltf;
static int hf_nl80211_he_ru_alloc;
static int hf_nl80211_eht_gi;
static int hf_nl80211_eht_ru_alloc;
static int hf_nl80211_rate_info;
static int hf_nl80211_sta_bss_param;
static int hf_nl80211_sta_info;
static int hf_nl80211_tid_stats;
static int hf_nl80211_txq_stats;
static int hf_nl80211_mpath_flags;
static int hf_nl80211_mpath_info;
static int hf_nl80211_band_iftype_attr;
static int hf_nl80211_band_attr;
static int hf_nl80211_wmm_rule;
static int hf_nl80211_frequency_attr;
static int hf_nl80211_bitrate_attr;
static int hf_nl80211_reg_initiator;
static int hf_nl80211_reg_type;
static int hf_nl80211_reg_rule_attr;
static int hf_nl80211_sched_scan_match_attr;
static int hf_nl80211_reg_rule_flags;
static int hf_nl80211_dfs_regions;
static int hf_nl80211_user_reg_hint_type;
static int hf_nl80211_survey_info;
static int hf_nl80211_mntr_flags;
static int hf_nl80211_mesh_power_mode;
static int hf_nl80211_meshconf_params;
static int hf_nl80211_mesh_setup_params;
static int hf_nl80211_txq_attr;
static int hf_nl80211_ac;
static int hf_nl80211_channel_type;
static int hf_nl80211_key_mode;
static int hf_nl80211_chan_width;
static int hf_nl80211_bss_scan_width;
static int hf_nl80211_bss_use_for;
static int hf_nl80211_bss_cannot_use_reasons;
static int hf_nl80211_bss;
static int hf_nl80211_bss_status;
static int hf_nl80211_auth_type;
static int hf_nl80211_key_type;
static int hf_nl80211_mfp;
static int hf_nl80211_wpa_versions;
static int hf_nl80211_key_default_types;
static int hf_nl80211_key_attributes;
static int hf_nl80211_tx_rate_attributes;
static int hf_nl80211_txrate_gi;
static int hf_nl80211_band;
static int hf_nl80211_ps_state;
static int hf_nl80211_attr_cqm;
static int hf_nl80211_cqm_rssi_threshold_event;
static int hf_nl80211_tx_power_setting;
static int hf_nl80211_tid_config;
static int hf_nl80211_tx_rate_setting;
static int hf_nl80211_tid_config_attr;
static int hf_nl80211_packet_pattern_attr;
static int hf_nl80211_wowlan_triggers;
static int hf_nl80211_wowlan_tcp_attrs;
static int hf_nl80211_attr_coalesce_rule;
static int hf_nl80211_coalesce_condition;
static int hf_nl80211_iface_limit_attrs;
static int hf_nl80211_if_combination_attrs;
static int hf_nl80211_plink_state;
static int hf_nl80211_plink_action;
static int hf_nl80211_rekey_data;
static int hf_nl80211_hidden_ssid;
static int hf_nl80211_sta_wme_attr;
static int hf_nl80211_pmksa_candidate_attr;
static int hf_nl80211_tdls_operation;
static int hf_nl80211_ap_sme_features;
static int hf_nl80211_feature_flags;
static int hf_nl80211_ext_feature_index;
static int hf_nl80211_probe_resp_offload_support_attr;
static int hf_nl80211_connect_failed_reason;
static int hf_nl80211_timeout_reason;
static int hf_nl80211_scan_flags;
static int hf_nl80211_acl_policy;
static int hf_nl80211_smps_mode;
static int hf_nl80211_radar_event;
static int hf_nl80211_dfs_state;
static int hf_nl80211_protocol_features;
static int hf_nl80211_crit_proto_id;
static int hf_nl80211_rxmgmt_flags;
static int hf_nl80211_tdls_peer_capability;
static int hf_nl80211_sched_scan_plan;
static int hf_nl80211_bss_select_attr;
static int hf_nl80211_nan_function_type;
static int hf_nl80211_nan_publish_type;
static int hf_nl80211_nan_func_term_reason;
static int hf_nl80211_nan_func_attributes;
static int hf_nl80211_nan_srf_attributes;
static int hf_nl80211_nan_match_attributes;
static int hf_nl80211_external_auth_action;
static int hf_nl80211_ftm_responder_attributes;
static int hf_nl80211_ftm_responder_stats;
static int hf_nl80211_preamble;
static int hf_nl80211_peer_measurement_type;
static int hf_nl80211_peer_measurement_status;
static int hf_nl80211_peer_measurement_req;
static int hf_nl80211_peer_measurement_resp;
static int hf_nl80211_peer_measurement_peer_attrs;
static int hf_nl80211_peer_measurement_attrs;
static int hf_nl80211_peer_measurement_ftm_capa;
static int hf_nl80211_peer_measurement_ftm_req;
static int hf_nl80211_peer_measurement_ftm_failure_reasons;
static int hf_nl80211_peer_measurement_ftm_resp;
static int hf_nl80211_obss_pd_attributes;
static int hf_nl80211_bss_color_attributes;
static int hf_nl80211_iftype_akm_attributes;
static int hf_nl80211_fils_discovery_attributes;
static int hf_nl80211_unsol_bcast_probe_resp_attributes;
static int hf_nl80211_sae_pwe_mechanism;
static int hf_nl80211_sar_type;
static int hf_nl80211_sar_attrs;
static int hf_nl80211_sar_specs_attrs;
static int hf_nl80211_mbssid_config_attributes;
static int hf_nl80211_ap_settings_flags;
static int hf_nl80211_wiphy_radio_attrs;
static int hf_nl80211_wiphy_radio_freq_range;
static int hf_nl80211_s1g_short_beacon_attrs;
static int hf_nl80211_nan_capabilities;

static int ett_nl80211_commands;
static int ett_nl80211_attrs;
static int ett_nl80211_iftype;
static int ett_nl80211_sta_flags;
static int ett_nl80211_sta_p2p_ps_status;
static int ett_nl80211_he_gi;
static int ett_nl80211_he_ltf;
static int ett_nl80211_he_ru_alloc;
static int ett_nl80211_eht_gi;
static int ett_nl80211_eht_ru_alloc;
static int ett_nl80211_rate_info;
static int ett_nl80211_sta_bss_param;
static int ett_nl80211_sta_info;
static int ett_nl80211_tid_stats;
static int ett_nl80211_txq_stats;
static int ett_nl80211_mpath_flags;
static int ett_nl80211_mpath_info;
static int ett_nl80211_band_iftype_attr;
static int ett_nl80211_band_attr;
static int ett_nl80211_wmm_rule;
static int ett_nl80211_frequency_attr;
static int ett_nl80211_bitrate_attr;
static int ett_nl80211_reg_initiator;
static int ett_nl80211_reg_type;
static int ett_nl80211_reg_rule_attr;
static int ett_nl80211_sched_scan_match_attr;
static int ett_nl80211_reg_rule_flags;
static int ett_nl80211_dfs_regions;
static int ett_nl80211_user_reg_hint_type;
static int ett_nl80211_survey_info;
static int ett_nl80211_mntr_flags;
static int ett_nl80211_mesh_power_mode;
static int ett_nl80211_meshconf_params;
static int ett_nl80211_mesh_setup_params;
static int ett_nl80211_txq_attr;
static int ett_nl80211_ac;
static int ett_nl80211_channel_type;
static int ett_nl80211_key_mode;
static int ett_nl80211_chan_width;
static int ett_nl80211_bss_scan_width;
static int ett_nl80211_bss_use_for;
static int ett_nl80211_bss_cannot_use_reasons;
static int ett_nl80211_bss;
static int ett_nl80211_bss_status;
static int ett_nl80211_auth_type;
static int ett_nl80211_key_type;
static int ett_nl80211_mfp;
static int ett_nl80211_wpa_versions;
static int ett_nl80211_key_default_types;
static int ett_nl80211_key_attributes;
static int ett_nl80211_tx_rate_attributes;
static int ett_nl80211_txrate_gi;
static int ett_nl80211_band;
static int ett_nl80211_ps_state;
static int ett_nl80211_attr_cqm;
static int ett_nl80211_cqm_rssi_threshold_event;
static int ett_nl80211_tx_power_setting;
static int ett_nl80211_tid_config;
static int ett_nl80211_tx_rate_setting;
static int ett_nl80211_tid_config_attr;
static int ett_nl80211_packet_pattern_attr;
static int ett_nl80211_wowlan_triggers;
static int ett_nl80211_wowlan_tcp_attrs;
static int ett_nl80211_attr_coalesce_rule;
static int ett_nl80211_coalesce_condition;
static int ett_nl80211_iface_limit_attrs;
static int ett_nl80211_if_combination_attrs;
static int ett_nl80211_plink_state;
static int ett_nl80211_plink_action;
static int ett_nl80211_rekey_data;
static int ett_nl80211_hidden_ssid;
static int ett_nl80211_sta_wme_attr;
static int ett_nl80211_pmksa_candidate_attr;
static int ett_nl80211_tdls_operation;
static int ett_nl80211_ap_sme_features;
static int ett_nl80211_feature_flags;
static int ett_nl80211_ext_feature_index;
static int ett_nl80211_probe_resp_offload_support_attr;
static int ett_nl80211_connect_failed_reason;
static int ett_nl80211_timeout_reason;
static int ett_nl80211_scan_flags;
static int ett_nl80211_acl_policy;
static int ett_nl80211_smps_mode;
static int ett_nl80211_radar_event;
static int ett_nl80211_dfs_state;
static int ett_nl80211_protocol_features;
static int ett_nl80211_crit_proto_id;
static int ett_nl80211_rxmgmt_flags;
static int ett_nl80211_tdls_peer_capability;
static int ett_nl80211_sched_scan_plan;
static int ett_nl80211_bss_select_attr;
static int ett_nl80211_nan_function_type;
static int ett_nl80211_nan_publish_type;
static int ett_nl80211_nan_func_term_reason;
static int ett_nl80211_nan_func_attributes;
static int ett_nl80211_nan_srf_attributes;
static int ett_nl80211_nan_match_attributes;
static int ett_nl80211_external_auth_action;
static int ett_nl80211_ftm_responder_attributes;
static int ett_nl80211_ftm_responder_stats;
static int ett_nl80211_preamble;
static int ett_nl80211_peer_measurement_type;
static int ett_nl80211_peer_measurement_status;
static int ett_nl80211_peer_measurement_req;
static int ett_nl80211_peer_measurement_resp;
static int ett_nl80211_peer_measurement_peer_attrs;
static int ett_nl80211_peer_measurement_attrs;
static int ett_nl80211_peer_measurement_ftm_capa;
static int ett_nl80211_peer_measurement_ftm_req;
static int ett_nl80211_peer_measurement_ftm_failure_reasons;
static int ett_nl80211_peer_measurement_ftm_resp;
static int ett_nl80211_obss_pd_attributes;
static int ett_nl80211_bss_color_attributes;
static int ett_nl80211_iftype_akm_attributes;
static int ett_nl80211_fils_discovery_attributes;
static int ett_nl80211_unsol_bcast_probe_resp_attributes;
static int ett_nl80211_sae_pwe_mechanism;
static int ett_nl80211_sar_type;
static int ett_nl80211_sar_attrs;
static int ett_nl80211_sar_specs_attrs;
static int ett_nl80211_mbssid_config_attributes;
static int ett_nl80211_ap_settings_flags;
static int ett_nl80211_wiphy_radio_attrs;
static int ett_nl80211_wiphy_radio_freq_range;
static int ett_nl80211_s1g_short_beacon_attrs;
static int ett_nl80211_nan_capabilities;
/* }}} */


static int proto_netlink_nl80211;

static dissector_handle_t netlink_nl80211_handle;

static int hf_nl80211_attr_value;
static int hf_nl80211_attr_value16;
static int hf_nl80211_attr_value32;
static int hf_nl80211_attr_value64;
static int hf_nl80211_wiphy_name;
static int hf_nl80211_ifname;
static int hf_nl80211_mac;
static int hf_nl80211_alpha2;
static int hf_nl80211_dbm;
static int hf_nl80211_mlo_links;

static int ett_nl80211;
static int ett_nl80211_frame;
static int ett_nl80211_tag;
static int ett_nl80211_mlo_links;

static int
dissect_nl80211_generic(tvbuff_t *tvb, void *data _U_, struct packet_netlink_data *nl_data, proto_tree *tree, int nla_type _U_, int offset, int len)
{
    /*
     * No specific dissection available, apply arbitrary heuristics to
     * determine whether we have an u16 or u32 field and treat others as
     * opaque bytes.
     */
    if (len) {
        if (len == 2) {
            proto_tree_add_item(tree, hf_nl80211_attr_value16, tvb, offset, len, nl_data->encoding);
        } else if (len == 4) {
            proto_tree_add_item(tree, hf_nl80211_attr_value32, tvb, offset, len, nl_data->encoding);
        } else if (len == 8) {
            proto_tree_add_item(tree, hf_nl80211_attr_value64, tvb, offset, len, nl_data->encoding);
        } else {
            proto_tree_add_item(tree, hf_nl80211_attr_value, tvb, offset, len, nl_data->encoding);
        }
        offset += len;
    }
    return offset;
}

struct attr_lookup {
    unsigned int attr_type;
    int *hfptr;
    int* ett;
    int (*func)(tvbuff_t *tvb, void *data, struct packet_netlink_data *nl_data, proto_tree *tree, int nla_type, int offset, int len);
};

static int
dissect_nested_attr(tvbuff_t *tvb, void *data, struct packet_netlink_data *nl_data, proto_tree *tree, int nla_type, int offset, int len, const struct attr_lookup *nested)
{
    netlink_nl80211_info_t *info = (netlink_nl80211_info_t *)data;
    for (int i = 0; nested[i].attr_type != 0; i++) {
        if (nested[i].attr_type != (nla_type & NLA_TYPE_MASK)) {
            continue;
        }
        offset = dissect_netlink_attributes(tvb, *nested[i].hfptr, *nested[i].ett, info,
                                            nl_data, tree, offset, len,
                                            nested[i].func ? nested[i].func : dissect_nl80211_generic);
        break;
    }
    return offset;
}

static int
dissect_nested_attr_array(tvbuff_t *tvb, void *data, struct packet_netlink_data *nl_data, proto_tree *tree, int nla_type, int offset, int len, const struct attr_lookup *nested_arr)
{
    netlink_nl80211_info_t *info = (netlink_nl80211_info_t *)data;
    for (int i = 0; nested_arr[i].attr_type != 0; i++) {
        if (nested_arr[i].attr_type != (nla_type & NLA_TYPE_MASK)) {
            continue;
        }
        offset = dissect_netlink_attributes_array(tvb, *nested_arr[i].hfptr, *nested_arr[i].ett,
                                                  *nested_arr[i].ett, info,
                                                  nl_data, tree, offset, len,
                                                  nested_arr[i].func ?
                                                   nested_arr[i].func : dissect_nl80211_generic);
       break;
    }
    return offset;
}

static int
dissect_value(tvbuff_t *tvb, void *data _U_, struct packet_netlink_data *nl_data, proto_tree *tree, int nla_type, int offset, int len, const struct attr_lookup *values)
{
    for (int i = 0; values[i].attr_type != 0; i++) {
        if (values[i].attr_type != (nla_type & NLA_TYPE_MASK)) {
            continue;
        }
        proto_tree_add_item(tree, *values[i].hfptr, tvb, offset, len, nl_data->encoding);
        return offset + len;
    }
    return offset;
}

static int
dissect_tag(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, int offset, int len, uint8_t tag)
{
    proto_item *item;
    ieee80211_tagged_field_data_t field_data = { 0 };
    tvbuff_t *next_tvb = tvb_new_subset_length(tvb, offset, len);
    proto_tree *subtree = proto_tree_add_subtree(tree, next_tvb, 0, -1, ett_nl80211_tag, &item, "Attribute Value");
    dissector_try_uint_with_data(ieee80211_tag_dissector_table, tag, next_tvb, pinfo, subtree, false, &field_data);
    return offset +  len;
}

static int
dissect_information_elements(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, int offset, int len)
{
    int offset_end = offset + len;
    while (offset < offset_end) {
        int tlen = add_tagged_field(pinfo, tree, tvb, offset, 0, NULL, 0, NULL);
        if (tlen == 0) {
            break;
        }
        offset += tlen;
    }
    return offset;
}

static int
dissect_nl80211_frequency_attr(tvbuff_t *tvb, void *data, struct packet_netlink_data *nl_data, proto_tree *tree, int nla_type, int offset, int len)
{
    static const struct attr_lookup nested[] = {
        { WS_NL80211_FREQUENCY_ATTR_DFS_STATE, &hf_nl80211_dfs_state, &ett_nl80211_dfs_state, NULL},
        { WS_NL80211_FREQUENCY_ATTR_WMM, &hf_nl80211_wmm_rule, &ett_nl80211_wmm_rule, NULL},
        { 0, NULL, NULL, NULL }
    };
    enum ws_nl80211_frequency_attr type = (enum ws_nl80211_frequency_attr) nla_type & NLA_TYPE_MASK;
    int offset_end = offset + len;

    if (offset < offset_end) {
        offset = dissect_nested_attr(tvb, data, nl_data, tree, nla_type, offset, len, nested);
    }
    if (offset < offset_end) {
        switch (type) {
        default:
            offset = dissect_nl80211_generic(tvb, data, nl_data, tree, nla_type, offset, len);
            break;
        }
    }
    return offset;
}

static int
dissect_nl80211_band_attr(tvbuff_t *tvb, void *data, struct packet_netlink_data *nl_data, proto_tree *tree, int nla_type, int offset, int len)
{
    static const struct attr_lookup nested_arr[] = {
        { WS_NL80211_BAND_ATTR_FREQS, &hf_nl80211_frequency_attr, &ett_nl80211_frequency_attr, dissect_nl80211_frequency_attr },
        { WS_NL80211_BAND_ATTR_RATES, &hf_nl80211_bitrate_attr, &ett_nl80211_bitrate_attr, NULL },
        { WS_NL80211_BAND_ATTR_IFTYPE_DATA, &hf_nl80211_band_iftype_attr, &ett_nl80211_band_iftype_attr, NULL },
        { 0, NULL, NULL, NULL }
    };
    enum ws_nl80211_band_attr type = (enum ws_nl80211_band_attr) nla_type & NLA_TYPE_MASK;
    int offset_end = offset + len;

    if (offset < offset_end) {
        offset = dissect_nested_attr_array(tvb, data, nl_data, tree, nla_type, offset, len, nested_arr);
    }
    if (offset < offset_end) {
        switch (type) {
        /* TODO add more fields here? */
        default:
            offset = dissect_nl80211_generic(tvb, data, nl_data, tree, nla_type, offset, len);
            break;
        }
    }
    return offset;
}

static int
dissect_nl80211_bss(tvbuff_t *tvb, void *data, struct packet_netlink_data *nl_data, proto_tree *tree, int nla_type, int offset, int len)
{
    static const struct attr_lookup values[] = {
        { WS_NL80211_BSS_STATUS, &hf_nl80211_bss_status, NULL, NULL },
        { WS_NL80211_BSS_CHAN_WIDTH, &hf_nl80211_bss_scan_width, NULL, NULL },
        { 0, NULL, NULL, NULL }
    };
    enum ws_nl80211_bss type = (enum ws_nl80211_bss) nla_type & NLA_TYPE_MASK;
    int offset_end = offset + len;

    if (offset < offset_end) {
        offset = dissect_value(tvb, data, nl_data, tree, nla_type, offset, len, values);
    }
    if (offset < offset_end) {
        netlink_nl80211_info_t *info = (netlink_nl80211_info_t *)data;
        switch (type) {
        case WS_NL80211_BSS_INFORMATION_ELEMENTS:
        case WS_NL80211_BSS_BEACON_IES:
            offset = dissect_information_elements(tvb, info->pinfo, tree, offset, len);
            break;
        default:
            offset = dissect_nl80211_generic(tvb, data, nl_data, tree, nla_type, offset, len);
            break;
        }
    }
    return offset;
}

static int
dissect_nl80211_tid_stats(tvbuff_t *tvb, void *data, struct packet_netlink_data *nl_data, proto_tree *tree, int nla_type, int offset, int len)
{
    static const struct attr_lookup nested[] = {
        { WS_NL80211_TID_STATS_TXQ_STATS, &hf_nl80211_txq_stats, &ett_nl80211_txq_stats, NULL},
        { 0, NULL, NULL, NULL }
    };

    enum ws_nl80211_tid_stats type = (enum ws_nl80211_tid_stats) nla_type & NLA_TYPE_MASK;
    int offset_end = offset + len;
    if (offset < offset_end) {
        offset = dissect_nested_attr(tvb, data, nl_data, tree, nla_type, offset, len, nested);
    }
    if (offset < offset_end) {
        switch (type) {
        default:
            offset = dissect_nl80211_generic(tvb, data, nl_data, tree, nla_type, offset, len);
            break;
        }
    }
    return offset;
}

static int
dissect_nl80211_sta_info(tvbuff_t *tvb, void *data, struct packet_netlink_data *nl_data, proto_tree *tree, int nla_type, int offset, int len)
{
    static const struct attr_lookup nested[] = {
        { WS_NL80211_STA_INFO_TX_BITRATE, &hf_nl80211_rate_info, &ett_nl80211_rate_info, NULL},
        { WS_NL80211_STA_INFO_RX_BITRATE, &hf_nl80211_rate_info, &ett_nl80211_rate_info, NULL},
        { WS_NL80211_STA_INFO_BSS_PARAM, &hf_nl80211_sta_bss_param, &ett_nl80211_sta_bss_param, NULL },
        { 0, NULL, NULL, NULL }
    };
    static const struct attr_lookup nested_arr[] = {
        { WS_NL80211_STA_INFO_TID_STATS, &hf_nl80211_tid_stats, &ett_nl80211_tid_stats, dissect_nl80211_tid_stats},
        { 0, NULL, NULL, NULL }
    };

    enum ws_nl80211_sta_info type = (enum ws_nl80211_sta_info) nla_type & NLA_TYPE_MASK;
    int offset_end = offset + len;
    if (offset < offset_end) {
        offset = dissect_nested_attr(tvb, data, nl_data, tree, nla_type, offset, len, nested);
    }
    if (offset < offset_end) {
        offset = dissect_nested_attr_array(tvb, data, nl_data, tree, nla_type, offset, len, nested_arr);
    }
    if (offset < offset_end) {
        switch (type) {
        case WS_NL80211_STA_INFO_SIGNAL:
        case WS_NL80211_STA_INFO_SIGNAL_AVG:
        case WS_NL80211_STA_INFO_BEACON_SIGNAL_AVG:
        case WS_NL80211_STA_INFO_ACK_SIGNAL:
        case WS_NL80211_STA_INFO_ACK_SIGNAL_AVG:
            proto_tree_add_item(tree, hf_nl80211_dbm, tvb, offset, len, nl_data->encoding);
            offset += len;
            break;
        default:
            offset = dissect_nl80211_generic(tvb, data, nl_data, tree, nla_type, offset, len);
            break;
        }
    }
    return offset;
}


static int
dissect_nl80211_attrs(tvbuff_t *tvb, void *data, struct packet_netlink_data *nl_data, proto_tree *tree, int nla_type, int offset, int len)
{
    static const struct attr_lookup nested[] = {
        { WS_NL80211_ATTR_SUPPORTED_IFTYPES, &hf_nl80211_iftype, &ett_nl80211_iftype, NULL },
        { WS_NL80211_ATTR_STA_FLAGS, &hf_nl80211_sta_flags, &ett_nl80211_sta_flags, NULL },
        { WS_NL80211_ATTR_STA_INFO, &hf_nl80211_sta_info, &ett_nl80211_sta_info, dissect_nl80211_sta_info },
        { WS_NL80211_ATTR_MPATH_INFO, &hf_nl80211_mpath_info, &ett_nl80211_mpath_info, NULL },
        { WS_NL80211_ATTR_MNTR_FLAGS, &hf_nl80211_mntr_flags, &ett_nl80211_mntr_flags, NULL },
        { WS_NL80211_ATTR_BSS, &hf_nl80211_bss, &ett_nl80211_bss, dissect_nl80211_bss },
        { WS_NL80211_ATTR_KEY, &hf_nl80211_key_attributes, &ett_nl80211_key_attributes, NULL },
        { WS_NL80211_ATTR_SURVEY_INFO, &hf_nl80211_survey_info, &ett_nl80211_survey_info, NULL },
        { WS_NL80211_ATTR_FREQ_BEFORE, &hf_nl80211_frequency_attr, &ett_nl80211_frequency_attr, NULL },
        { WS_NL80211_ATTR_FREQ_AFTER, &hf_nl80211_frequency_attr, &ett_nl80211_frequency_attr, NULL },
        { WS_NL80211_ATTR_TX_RATES, &hf_nl80211_tx_rate_attributes, &ett_nl80211_tx_rate_attributes, NULL },
        { WS_NL80211_ATTR_CQM, &hf_nl80211_attr_cqm, &ett_nl80211_attr_cqm, NULL },
        { WS_NL80211_ATTR_KEY_DEFAULT_TYPES, &hf_nl80211_key_default_types, &ett_nl80211_key_default_types, NULL },
        { WS_NL80211_ATTR_MESH_SETUP, &hf_nl80211_mesh_setup_params, &ett_nl80211_mesh_setup_params, NULL },
        { WS_NL80211_ATTR_MESH_CONFIG, &hf_nl80211_meshconf_params, &ett_nl80211_meshconf_params, NULL },
        { WS_NL80211_ATTR_SCHED_SCAN_MATCH, &hf_nl80211_sched_scan_match_attr, &ett_nl80211_sched_scan_match_attr, NULL },
        { WS_NL80211_ATTR_INTERFACE_COMBINATIONS, &hf_nl80211_if_combination_attrs, &ett_nl80211_if_combination_attrs, NULL },
        { WS_NL80211_ATTR_REKEY_DATA, &hf_nl80211_rekey_data, &ett_nl80211_rekey_data, NULL },
        { WS_NL80211_ATTR_STA_WME, &hf_nl80211_sta_wme_attr, &ett_nl80211_sta_wme_attr, NULL },
        { WS_NL80211_ATTR_PMKSA_CANDIDATE, &hf_nl80211_pmksa_candidate_attr, &ett_nl80211_pmksa_candidate_attr, NULL },
        { WS_NL80211_ATTR_SCHED_SCAN_PLANS, &hf_nl80211_sched_scan_plan, &ett_nl80211_sched_scan_plan, NULL },
        { WS_NL80211_ATTR_BSS_SELECT, &hf_nl80211_bss_select_attr, &ett_nl80211_bss_select_attr, NULL },
        { WS_NL80211_ATTR_IFTYPE_EXT_CAPA, &hf_nl80211_attrs, &ett_nl80211_attrs, dissect_nl80211_attrs },
        { WS_NL80211_ATTR_NAN_FUNC, &hf_nl80211_nan_func_attributes, &ett_nl80211_nan_func_attributes, NULL },
        { WS_NL80211_ATTR_NAN_MATCH, &hf_nl80211_nan_match_attributes, &ett_nl80211_nan_match_attributes, NULL },
        { WS_NL80211_ATTR_TXQ_STATS, &hf_nl80211_txq_stats, &ett_nl80211_txq_stats, NULL },
        { 0, NULL, NULL, NULL }
    };
    static const struct attr_lookup nested_arr[] = {
        { WS_NL80211_ATTR_WIPHY_TXQ_PARAMS, &hf_nl80211_txq_attr, &ett_nl80211_txq_attr, NULL },
        { WS_NL80211_ATTR_WIPHY_BANDS, &hf_nl80211_band_attr, &ett_nl80211_band_attr, dissect_nl80211_band_attr },
        { WS_NL80211_ATTR_REG_RULES, &hf_nl80211_reg_rule_attr, &ett_nl80211_reg_rule_attr, NULL },
        { WS_NL80211_ATTR_MLO_LINKS, &hf_nl80211_mlo_links, &ett_nl80211_mlo_links, NULL },
        { 0, NULL, NULL, NULL }
    };
    static const struct attr_lookup values[] = {
        { WS_NL80211_ATTR_CHANNEL_WIDTH, &hf_nl80211_chan_width, NULL, NULL },
        { WS_NL80211_ATTR_WIPHY_NAME, &hf_nl80211_wiphy_name, NULL, NULL },
        { WS_NL80211_ATTR_WIPHY_CHANNEL_TYPE, &hf_nl80211_channel_type, NULL, NULL },
        { WS_NL80211_ATTR_IFNAME, &hf_nl80211_ifname, NULL, NULL },
        { WS_NL80211_ATTR_IFTYPE, &hf_nl80211_iftype, NULL, NULL },
        { WS_NL80211_ATTR_MAC, &hf_nl80211_mac, NULL, NULL },
        { WS_NL80211_ATTR_STA_PLINK_ACTION, &hf_nl80211_plink_action, NULL, NULL },
        { WS_NL80211_ATTR_MPATH_INFO, &hf_nl80211_mpath_info, NULL, NULL },
        { WS_NL80211_ATTR_REG_ALPHA2, &hf_nl80211_alpha2, NULL, NULL },
        { WS_NL80211_ATTR_REG_INITIATOR, &hf_nl80211_reg_initiator, NULL, NULL },
        { WS_NL80211_ATTR_REG_TYPE, &hf_nl80211_reg_type, NULL, NULL },
        { WS_NL80211_ATTR_AUTH_TYPE, &hf_nl80211_auth_type, NULL, NULL },
        { WS_NL80211_ATTR_KEY_TYPE, &hf_nl80211_key_type, NULL, NULL },
        { WS_NL80211_ATTR_USE_MFP, &hf_nl80211_mfp, NULL, NULL },
        { WS_NL80211_ATTR_PS_STATE, &hf_nl80211_ps_state, NULL, NULL },
        { WS_NL80211_ATTR_WIPHY_TX_POWER_SETTING, &hf_nl80211_tx_power_setting, NULL, NULL },
        { WS_NL80211_ATTR_STA_PLINK_STATE, &hf_nl80211_plink_state, NULL, NULL },
        { WS_NL80211_ATTR_TDLS_OPERATION, &hf_nl80211_tdls_operation, NULL, NULL },
        { WS_NL80211_ATTR_DFS_REGION, &hf_nl80211_dfs_regions, NULL, NULL },
        { WS_NL80211_ATTR_RX_SIGNAL_DBM, &hf_nl80211_dbm, NULL, NULL},
        { WS_NL80211_ATTR_USER_REG_HINT_TYPE, &hf_nl80211_user_reg_hint_type, NULL, NULL },
        { WS_NL80211_ATTR_CONN_FAILED_REASON, &hf_nl80211_connect_failed_reason, NULL, NULL },
        { WS_NL80211_ATTR_LOCAL_MESH_POWER_MODE, &hf_nl80211_mesh_power_mode, NULL, NULL },
        { WS_NL80211_ATTR_ACL_POLICY, &hf_nl80211_acl_policy, NULL, NULL },
        { WS_NL80211_ATTR_RADAR_EVENT, &hf_nl80211_radar_event, NULL, NULL },
        { WS_NL80211_ATTR_CRIT_PROT_ID, &hf_nl80211_crit_proto_id, NULL, NULL },
        { WS_NL80211_ATTR_SMPS_MODE, &hf_nl80211_smps_mode, NULL, NULL },
        { WS_NL80211_ATTR_STA_SUPPORT_P2P_PS, &hf_nl80211_sta_p2p_ps_status, NULL, NULL },
        { WS_NL80211_ATTR_TIMEOUT_REASON, &hf_nl80211_timeout_reason, NULL, NULL },
        { WS_NL80211_ATTR_EXTERNAL_AUTH_ACTION, &hf_nl80211_external_auth_action, NULL, NULL },
        { 0, NULL, NULL, NULL }
    };
    enum ws_nl80211_attrs type = (enum ws_nl80211_attrs) nla_type & NLA_TYPE_MASK;
    int offset_end = offset + len;
    if (offset < offset_end) {
        offset = dissect_nested_attr(tvb, data, nl_data, tree, nla_type, offset, len, nested);
    }
    if (offset < offset_end) {
        offset = dissect_nested_attr_array(tvb, data, nl_data, tree, nla_type, offset, len, nested_arr);
    }
    if (offset < offset_end) {
        offset = dissect_value(tvb, data, nl_data, tree, nla_type, offset, len, values);
    }
    if (offset < offset_end) {
        netlink_nl80211_info_t *info = (netlink_nl80211_info_t *)data;
        tvbuff_t *next_tvb;
        proto_tree *subtree;
        proto_item *item;

        switch (type) {
        case WS_NL80211_ATTR_IE:
        case WS_NL80211_ATTR_REQ_IE:
        case WS_NL80211_ATTR_RESP_IE:
        case WS_NL80211_ATTR_IE_PROBE_RESP:
        case WS_NL80211_ATTR_IE_ASSOC_RESP:
        case WS_NL80211_ATTR_CSA_IES:
            offset = dissect_information_elements(tvb, info->pinfo, tree, offset, len);
            break;
        case WS_NL80211_ATTR_HT_CAPABILITY:
            offset = dissect_tag(tvb, info->pinfo, tree, offset, len, 45 /* TAG_HT_CAPABILITY */);
            break;
        case WS_NL80211_ATTR_VHT_CAPABILITY:
            offset = dissect_tag(tvb, info->pinfo, tree, offset, len, 191 /* TAG_VHT_CAPABILITY */);
            break;
        case WS_NL80211_ATTR_FRAME:
            next_tvb = tvb_new_subset_length(tvb, offset, len);
            subtree = proto_tree_add_subtree(tree, next_tvb, 0, -1, ett_nl80211_frame,
                                             &item, "Attribute Value");
            call_dissector(ieee80211_handle, next_tvb, info->pinfo, subtree);
            break;
        /* TODO add more fields here? */
        default:
            offset = dissect_nl80211_generic(tvb, data, nl_data, tree, nla_type, offset, len);
            break;
        }
    }
    return offset;
}

static int
dissect_netlink_nl80211(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void *data)
{
    netlink_nl80211_info_t info;
    genl_info_t *genl_info = (genl_info_t *)data;
    proto_tree *nlmsg_tree;
    proto_item *pi;
    int offset;

    DISSECTOR_ASSERT(genl_info);

    col_set_str(pinfo->cinfo, COL_PROTOCOL, "nl80211");
    col_clear(pinfo->cinfo, COL_INFO);

    offset = dissect_genl_header(tvb, genl_info, genl_info->nl_data, hf_nl80211_commands);

    /* Return if command has no payload */
    if (!tvb_reported_length_remaining(tvb, offset))
            /* XXX If you do not set the protocol item, you cannot filter on these messages */
            return offset;

    info.pinfo = pinfo;

    pi = proto_tree_add_item(tree, proto_netlink_nl80211, tvb, offset, -1, ENC_NA);
    nlmsg_tree = proto_item_add_subtree(pi, ett_nl80211);

    offset = dissect_netlink_attributes_to_end(tvb, hf_nl80211_attrs, ett_nl80211_attrs, &info, genl_info->nl_data, nlmsg_tree, offset, dissect_nl80211_attrs);

    return offset;
}

void
proto_register_netlink_nl80211(void)
{
    static hf_register_info hf[] = {
        { &hf_nl80211_attr_value,
            { "Attribute Value", "nl80211.attr_value",
              FT_BYTES, BASE_NONE, NULL, 0x00,
              NULL, HFILL }
        },
        { &hf_nl80211_attr_value16,
            { "Attribute Value", "nl80211.attr_value16",
              FT_UINT16, BASE_HEX_DEC, NULL, 0x00,
              NULL, HFILL }
        },
        { &hf_nl80211_attr_value32,
            { "Attribute Value", "nl80211.attr_value32",
              FT_UINT32, BASE_HEX_DEC, NULL, 0x00,
              NULL, HFILL }
        },
        { &hf_nl80211_attr_value64,
            { "Attribute Value", "nl80211.attr_value64",
              FT_UINT64, BASE_HEX_DEC, NULL, 0x00,
              NULL, HFILL }
        },
        { &hf_nl80211_wiphy_name,
            { "Wiphy Name", "nl80211.wiphy_name",
              FT_STRINGZ, BASE_NONE, NULL, 0x00,
              NULL, HFILL }
        },
        { &hf_nl80211_ifname,
            { "Interface Name", "nl80211.ifname",
              FT_STRINGZ, BASE_NONE, NULL, 0x00,
              NULL, HFILL }
        },
        { &hf_nl80211_mac,
            { "MAC address", "nl80211.mac",
              FT_ETHER, BASE_NONE, NULL, 0x00,
              NULL, HFILL }
        },
        { &hf_nl80211_alpha2,
            { "Alpha2", "nl80211.alpha2",
              FT_STRINGZ, BASE_NONE, NULL, 0x00,
              NULL, HFILL }
        },
        { &hf_nl80211_dbm,
            { "dBm", "nl80211.dbm",
              FT_INT32, BASE_DEC, NULL, 0x00,
              NULL, HFILL }
        },
        { &hf_nl80211_mlo_links,
            { "Attribute Type", "nl80211.mlo_links",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_attrs_vals_ext), 0x00,
              NULL, HFILL },
        },
/* Extracted using tools/generate-nl80211-fields.py */
/* Definitions from linux/nl80211.h {{{ */
        { &hf_nl80211_commands,
            { "Command", "nl80211.cmd",
              FT_UINT8, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_commands_vals_ext), 0x00,
              "Generic Netlink Command", HFILL },
        },
        { &hf_nl80211_attrs,
            { "Attribute Type", "nl80211.attr_type",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_attrs_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_iftype,
            { "Attribute Type", "nl80211.iftype",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_iftype_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_sta_flags,
            { "Attribute Type", "nl80211.sta_flags",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_sta_flags_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_sta_p2p_ps_status,
            { "Attribute Value", "nl80211.sta_p2p_ps_status",
              FT_UINT8, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_sta_p2p_ps_status_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_he_gi,
            { "Attribute Type", "nl80211.he_gi",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_he_gi_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_he_ltf,
            { "Attribute Type", "nl80211.he_ltf",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_he_ltf_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_he_ru_alloc,
            { "Attribute Type", "nl80211.he_ru_alloc",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_he_ru_alloc_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_eht_gi,
            { "Attribute Type", "nl80211.eht_gi",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_eht_gi_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_eht_ru_alloc,
            { "Attribute Type", "nl80211.eht_ru_alloc",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_eht_ru_alloc_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_rate_info,
            { "Attribute Type", "nl80211.rate_info",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_rate_info_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_sta_bss_param,
            { "Attribute Type", "nl80211.sta_bss_param",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_sta_bss_param_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_sta_info,
            { "Attribute Type", "nl80211.sta_info",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_sta_info_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_tid_stats,
            { "Attribute Type", "nl80211.tid_stats",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_tid_stats_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_txq_stats,
            { "Attribute Type", "nl80211.txq_stats",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_txq_stats_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_mpath_flags,
            { "Attribute Type", "nl80211.mpath_flags",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_mpath_flags_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_mpath_info,
            { "Attribute Type", "nl80211.mpath_info",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_mpath_info_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_band_iftype_attr,
            { "Attribute Type", "nl80211.band_iftype_attr",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_band_iftype_attr_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_band_attr,
            { "Attribute Type", "nl80211.band_attr",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_band_attr_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_wmm_rule,
            { "Attribute Type", "nl80211.wmm_rule",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_wmm_rule_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_frequency_attr,
            { "Attribute Type", "nl80211.frequency_attr",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_frequency_attr_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_bitrate_attr,
            { "Attribute Type", "nl80211.bitrate_attr",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_bitrate_attr_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_reg_initiator,
            { "Attribute Value", "nl80211.reg_initiator",
              FT_UINT8, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_reg_initiator_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_reg_type,
            { "Attribute Value", "nl80211.reg_type",
              FT_UINT8, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_reg_type_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_reg_rule_attr,
            { "Attribute Type", "nl80211.reg_rule_attr",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_reg_rule_attr_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_sched_scan_match_attr,
            { "Attribute Type", "nl80211.sched_scan_match_attr",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_sched_scan_match_attr_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_reg_rule_flags,
            { "Attribute Type", "nl80211.reg_rule_flags",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_reg_rule_flags_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_dfs_regions,
            { "Attribute Value", "nl80211.dfs_regions",
              FT_UINT8, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_dfs_regions_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_user_reg_hint_type,
            { "Attribute Value", "nl80211.user_reg_hint_type",
              FT_UINT32, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_user_reg_hint_type_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_survey_info,
            { "Attribute Type", "nl80211.survey_info",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_survey_info_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_mntr_flags,
            { "Attribute Type", "nl80211.mntr_flags",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_mntr_flags_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_mesh_power_mode,
            { "Attribute Value", "nl80211.mesh_power_mode",
              FT_UINT32, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_mesh_power_mode_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_meshconf_params,
            { "Attribute Type", "nl80211.meshconf_params",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_meshconf_params_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_mesh_setup_params,
            { "Attribute Type", "nl80211.mesh_setup_params",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_mesh_setup_params_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_txq_attr,
            { "Attribute Type", "nl80211.txq_attr",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_txq_attr_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_ac,
            { "Attribute Type", "nl80211.ac",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_ac_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_channel_type,
            { "Attribute Value", "nl80211.channel_type",
              FT_UINT32, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_channel_type_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_key_mode,
            { "Attribute Type", "nl80211.key_mode",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_key_mode_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_chan_width,
            { "Attribute Value", "nl80211.chan_width",
              FT_UINT32, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_chan_width_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_bss_scan_width,
            { "Attribute Value", "nl80211.bss_scan_width",
              FT_UINT32, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_bss_scan_width_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_bss_use_for,
            { "Attribute Type", "nl80211.bss_use_for",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_bss_use_for_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_bss_cannot_use_reasons,
            { "Attribute Type", "nl80211.bss_cannot_use_reasons",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_bss_cannot_use_reasons_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_bss,
            { "Attribute Type", "nl80211.bss",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_bss_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_bss_status,
            { "Attribute Value", "nl80211.bss_status",
              FT_UINT32, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_bss_status_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_auth_type,
            { "Attribute Value", "nl80211.auth_type",
              FT_UINT32, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_auth_type_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_key_type,
            { "Attribute Value", "nl80211.key_type",
              FT_UINT32, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_key_type_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_mfp,
            { "Attribute Value", "nl80211.mfp",
              FT_UINT32, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_mfp_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_wpa_versions,
            { "Attribute Type", "nl80211.wpa_versions",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_wpa_versions_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_key_default_types,
            { "Attribute Type", "nl80211.key_default_types",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_key_default_types_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_key_attributes,
            { "Attribute Type", "nl80211.key_attributes",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_key_attributes_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_tx_rate_attributes,
            { "Attribute Type", "nl80211.tx_rate_attributes",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_tx_rate_attributes_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_txrate_gi,
            { "Attribute Type", "nl80211.txrate_gi",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_txrate_gi_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_band,
            { "Attribute Type", "nl80211.band",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_band_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_ps_state,
            { "Attribute Value", "nl80211.ps_state",
              FT_UINT32, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_ps_state_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_attr_cqm,
            { "Attribute Type", "nl80211.attr_cqm",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_attr_cqm_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_cqm_rssi_threshold_event,
            { "Attribute Type", "nl80211.cqm_rssi_threshold_event",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_cqm_rssi_threshold_event_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_tx_power_setting,
            { "Attribute Value", "nl80211.tx_power_setting",
              FT_UINT32, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_tx_power_setting_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_tid_config,
            { "Attribute Type", "nl80211.tid_config",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_tid_config_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_tx_rate_setting,
            { "Attribute Type", "nl80211.tx_rate_setting",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_tx_rate_setting_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_tid_config_attr,
            { "Attribute Type", "nl80211.tid_config_attr",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_tid_config_attr_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_packet_pattern_attr,
            { "Attribute Type", "nl80211.packet_pattern_attr",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_packet_pattern_attr_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_wowlan_triggers,
            { "Attribute Type", "nl80211.wowlan_triggers",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_wowlan_triggers_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_wowlan_tcp_attrs,
            { "Attribute Type", "nl80211.wowlan_tcp_attrs",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_wowlan_tcp_attrs_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_attr_coalesce_rule,
            { "Attribute Type", "nl80211.attr_coalesce_rule",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_attr_coalesce_rule_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_coalesce_condition,
            { "Attribute Type", "nl80211.coalesce_condition",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_coalesce_condition_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_iface_limit_attrs,
            { "Attribute Type", "nl80211.iface_limit_attrs",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_iface_limit_attrs_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_if_combination_attrs,
            { "Attribute Type", "nl80211.if_combination_attrs",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_if_combination_attrs_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_plink_state,
            { "Attribute Value", "nl80211.plink_state",
              FT_UINT8, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_plink_state_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_plink_action,
            { "Attribute Value", "nl80211.plink_action",
              FT_UINT8, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_plink_action_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_rekey_data,
            { "Attribute Type", "nl80211.rekey_data",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_rekey_data_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_hidden_ssid,
            { "Attribute Type", "nl80211.hidden_ssid",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_hidden_ssid_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_sta_wme_attr,
            { "Attribute Type", "nl80211.sta_wme_attr",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_sta_wme_attr_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_pmksa_candidate_attr,
            { "Attribute Type", "nl80211.pmksa_candidate_attr",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_pmksa_candidate_attr_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_tdls_operation,
            { "Attribute Value", "nl80211.tdls_operation",
              FT_UINT8, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_tdls_operation_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_ap_sme_features,
            { "Attribute Type", "nl80211.ap_sme_features",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_ap_sme_features_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_feature_flags,
            { "Attribute Type", "nl80211.feature_flags",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_feature_flags_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_ext_feature_index,
            { "Attribute Type", "nl80211.ext_feature_index",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_ext_feature_index_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_probe_resp_offload_support_attr,
            { "Attribute Type", "nl80211.probe_resp_offload_support_attr",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_probe_resp_offload_support_attr_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_connect_failed_reason,
            { "Attribute Value", "nl80211.connect_failed_reason",
              FT_UINT32, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_connect_failed_reason_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_timeout_reason,
            { "Attribute Value", "nl80211.timeout_reason",
              FT_UINT32, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_timeout_reason_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_scan_flags,
            { "Attribute Type", "nl80211.scan_flags",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_scan_flags_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_acl_policy,
            { "Attribute Value", "nl80211.acl_policy",
              FT_UINT32, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_acl_policy_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_smps_mode,
            { "Attribute Value", "nl80211.smps_mode",
              FT_UINT8, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_smps_mode_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_radar_event,
            { "Attribute Value", "nl80211.radar_event",
              FT_UINT32, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_radar_event_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_dfs_state,
            { "Attribute Type", "nl80211.dfs_state",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_dfs_state_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_protocol_features,
            { "Attribute Type", "nl80211.protocol_features",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_protocol_features_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_crit_proto_id,
            { "Attribute Value", "nl80211.crit_proto_id",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_crit_proto_id_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_rxmgmt_flags,
            { "Attribute Type", "nl80211.rxmgmt_flags",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_rxmgmt_flags_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_tdls_peer_capability,
            { "Attribute Type", "nl80211.tdls_peer_capability",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_tdls_peer_capability_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_sched_scan_plan,
            { "Attribute Type", "nl80211.sched_scan_plan",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_sched_scan_plan_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_bss_select_attr,
            { "Attribute Type", "nl80211.bss_select_attr",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_bss_select_attr_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_nan_function_type,
            { "Attribute Type", "nl80211.nan_function_type",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_nan_function_type_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_nan_publish_type,
            { "Attribute Type", "nl80211.nan_publish_type",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_nan_publish_type_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_nan_func_term_reason,
            { "Attribute Type", "nl80211.nan_func_term_reason",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_nan_func_term_reason_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_nan_func_attributes,
            { "Attribute Type", "nl80211.nan_func_attributes",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_nan_func_attributes_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_nan_srf_attributes,
            { "Attribute Type", "nl80211.nan_srf_attributes",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_nan_srf_attributes_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_nan_match_attributes,
            { "Attribute Type", "nl80211.nan_match_attributes",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_nan_match_attributes_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_external_auth_action,
            { "Attribute Value", "nl80211.external_auth_action",
              FT_UINT32, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_external_auth_action_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_ftm_responder_attributes,
            { "Attribute Type", "nl80211.ftm_responder_attributes",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_ftm_responder_attributes_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_ftm_responder_stats,
            { "Attribute Type", "nl80211.ftm_responder_stats",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_ftm_responder_stats_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_preamble,
            { "Attribute Type", "nl80211.preamble",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_preamble_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_peer_measurement_type,
            { "Attribute Type", "nl80211.peer_measurement_type",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_peer_measurement_type_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_peer_measurement_status,
            { "Attribute Type", "nl80211.peer_measurement_status",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_peer_measurement_status_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_peer_measurement_req,
            { "Attribute Type", "nl80211.peer_measurement_req",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_peer_measurement_req_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_peer_measurement_resp,
            { "Attribute Type", "nl80211.peer_measurement_resp",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_peer_measurement_resp_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_peer_measurement_peer_attrs,
            { "Attribute Type", "nl80211.peer_measurement_peer_attrs",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_peer_measurement_peer_attrs_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_peer_measurement_attrs,
            { "Attribute Type", "nl80211.peer_measurement_attrs",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_peer_measurement_attrs_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_peer_measurement_ftm_capa,
            { "Attribute Type", "nl80211.peer_measurement_ftm_capa",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_peer_measurement_ftm_capa_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_peer_measurement_ftm_req,
            { "Attribute Type", "nl80211.peer_measurement_ftm_req",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_peer_measurement_ftm_req_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_peer_measurement_ftm_failure_reasons,
            { "Attribute Type", "nl80211.peer_measurement_ftm_failure_reasons",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_peer_measurement_ftm_failure_reasons_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_peer_measurement_ftm_resp,
            { "Attribute Type", "nl80211.peer_measurement_ftm_resp",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_peer_measurement_ftm_resp_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_obss_pd_attributes,
            { "Attribute Type", "nl80211.obss_pd_attributes",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_obss_pd_attributes_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_bss_color_attributes,
            { "Attribute Type", "nl80211.bss_color_attributes",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_bss_color_attributes_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_iftype_akm_attributes,
            { "Attribute Type", "nl80211.iftype_akm_attributes",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_iftype_akm_attributes_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_fils_discovery_attributes,
            { "Attribute Type", "nl80211.fils_discovery_attributes",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_fils_discovery_attributes_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_unsol_bcast_probe_resp_attributes,
            { "Attribute Type", "nl80211.unsol_bcast_probe_resp_attributes",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_unsol_bcast_probe_resp_attributes_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_sae_pwe_mechanism,
            { "Attribute Type", "nl80211.sae_pwe_mechanism",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_sae_pwe_mechanism_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_sar_type,
            { "Attribute Type", "nl80211.sar_type",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_sar_type_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_sar_attrs,
            { "Attribute Type", "nl80211.sar_attrs",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_sar_attrs_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_sar_specs_attrs,
            { "Attribute Type", "nl80211.sar_specs_attrs",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_sar_specs_attrs_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_mbssid_config_attributes,
            { "Attribute Type", "nl80211.mbssid_config_attributes",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_mbssid_config_attributes_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_ap_settings_flags,
            { "Attribute Type", "nl80211.ap_settings_flags",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_ap_settings_flags_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_wiphy_radio_attrs,
            { "Attribute Type", "nl80211.wiphy_radio_attrs",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_wiphy_radio_attrs_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_wiphy_radio_freq_range,
            { "Attribute Type", "nl80211.wiphy_radio_freq_range",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_wiphy_radio_freq_range_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_s1g_short_beacon_attrs,
            { "Attribute Type", "nl80211.s1g_short_beacon_attrs",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_s1g_short_beacon_attrs_vals_ext), 0x00,
              NULL, HFILL },
        },
        { &hf_nl80211_nan_capabilities,
            { "Attribute Type", "nl80211.nan_capabilities",
              FT_UINT16, BASE_DEC | BASE_EXT_STRING,
              VALS_EXT_PTR(&ws_nl80211_nan_capabilities_vals_ext), 0x00,
              NULL, HFILL },
        },
/* }}} */
    };

    static int *ett[] = {
        &ett_nl80211,
        &ett_nl80211_frame,
        &ett_nl80211_tag,
        &ett_nl80211_mlo_links,
/* Extracted using tools/generate-nl80211-fields.py */
/* Definitions from linux/nl80211.h {{{ */
        &ett_nl80211_commands,
        &ett_nl80211_attrs,
        &ett_nl80211_iftype,
        &ett_nl80211_sta_flags,
        &ett_nl80211_sta_p2p_ps_status,
        &ett_nl80211_he_gi,
        &ett_nl80211_he_ltf,
        &ett_nl80211_he_ru_alloc,
        &ett_nl80211_eht_gi,
        &ett_nl80211_eht_ru_alloc,
        &ett_nl80211_rate_info,
        &ett_nl80211_sta_bss_param,
        &ett_nl80211_sta_info,
        &ett_nl80211_tid_stats,
        &ett_nl80211_txq_stats,
        &ett_nl80211_mpath_flags,
        &ett_nl80211_mpath_info,
        &ett_nl80211_band_iftype_attr,
        &ett_nl80211_band_attr,
        &ett_nl80211_wmm_rule,
        &ett_nl80211_frequency_attr,
        &ett_nl80211_bitrate_attr,
        &ett_nl80211_reg_initiator,
        &ett_nl80211_reg_type,
        &ett_nl80211_reg_rule_attr,
        &ett_nl80211_sched_scan_match_attr,
        &ett_nl80211_reg_rule_flags,
        &ett_nl80211_dfs_regions,
        &ett_nl80211_user_reg_hint_type,
        &ett_nl80211_survey_info,
        &ett_nl80211_mntr_flags,
        &ett_nl80211_mesh_power_mode,
        &ett_nl80211_meshconf_params,
        &ett_nl80211_mesh_setup_params,
        &ett_nl80211_txq_attr,
        &ett_nl80211_ac,
        &ett_nl80211_channel_type,
        &ett_nl80211_key_mode,
        &ett_nl80211_chan_width,
        &ett_nl80211_bss_scan_width,
        &ett_nl80211_bss_use_for,
        &ett_nl80211_bss_cannot_use_reasons,
        &ett_nl80211_bss,
        &ett_nl80211_bss_status,
        &ett_nl80211_auth_type,
        &ett_nl80211_key_type,
        &ett_nl80211_mfp,
        &ett_nl80211_wpa_versions,
        &ett_nl80211_key_default_types,
        &ett_nl80211_key_attributes,
        &ett_nl80211_tx_rate_attributes,
        &ett_nl80211_txrate_gi,
        &ett_nl80211_band,
        &ett_nl80211_ps_state,
        &ett_nl80211_attr_cqm,
        &ett_nl80211_cqm_rssi_threshold_event,
        &ett_nl80211_tx_power_setting,
        &ett_nl80211_tid_config,
        &ett_nl80211_tx_rate_setting,
        &ett_nl80211_tid_config_attr,
        &ett_nl80211_packet_pattern_attr,
        &ett_nl80211_wowlan_triggers,
        &ett_nl80211_wowlan_tcp_attrs,
        &ett_nl80211_attr_coalesce_rule,
        &ett_nl80211_coalesce_condition,
        &ett_nl80211_iface_limit_attrs,
        &ett_nl80211_if_combination_attrs,
        &ett_nl80211_plink_state,
        &ett_nl80211_plink_action,
        &ett_nl80211_rekey_data,
        &ett_nl80211_hidden_ssid,
        &ett_nl80211_sta_wme_attr,
        &ett_nl80211_pmksa_candidate_attr,
        &ett_nl80211_tdls_operation,
        &ett_nl80211_ap_sme_features,
        &ett_nl80211_feature_flags,
        &ett_nl80211_ext_feature_index,
        &ett_nl80211_probe_resp_offload_support_attr,
        &ett_nl80211_connect_failed_reason,
        &ett_nl80211_timeout_reason,
        &ett_nl80211_scan_flags,
        &ett_nl80211_acl_policy,
        &ett_nl80211_smps_mode,
        &ett_nl80211_radar_event,
        &ett_nl80211_dfs_state,
        &ett_nl80211_protocol_features,
        &ett_nl80211_crit_proto_id,
        &ett_nl80211_rxmgmt_flags,
        &ett_nl80211_tdls_peer_capability,
        &ett_nl80211_sched_scan_plan,
        &ett_nl80211_bss_select_attr,
        &ett_nl80211_nan_function_type,
        &ett_nl80211_nan_publish_type,
        &ett_nl80211_nan_func_term_reason,
        &ett_nl80211_nan_func_attributes,
        &ett_nl80211_nan_srf_attributes,
        &ett_nl80211_nan_match_attributes,
        &ett_nl80211_external_auth_action,
        &ett_nl80211_ftm_responder_attributes,
        &ett_nl80211_ftm_responder_stats,
        &ett_nl80211_preamble,
        &ett_nl80211_peer_measurement_type,
        &ett_nl80211_peer_measurement_status,
        &ett_nl80211_peer_measurement_req,
        &ett_nl80211_peer_measurement_resp,
        &ett_nl80211_peer_measurement_peer_attrs,
        &ett_nl80211_peer_measurement_attrs,
        &ett_nl80211_peer_measurement_ftm_capa,
        &ett_nl80211_peer_measurement_ftm_req,
        &ett_nl80211_peer_measurement_ftm_failure_reasons,
        &ett_nl80211_peer_measurement_ftm_resp,
        &ett_nl80211_obss_pd_attributes,
        &ett_nl80211_bss_color_attributes,
        &ett_nl80211_iftype_akm_attributes,
        &ett_nl80211_fils_discovery_attributes,
        &ett_nl80211_unsol_bcast_probe_resp_attributes,
        &ett_nl80211_sae_pwe_mechanism,
        &ett_nl80211_sar_type,
        &ett_nl80211_sar_attrs,
        &ett_nl80211_sar_specs_attrs,
        &ett_nl80211_mbssid_config_attributes,
        &ett_nl80211_ap_settings_flags,
        &ett_nl80211_wiphy_radio_attrs,
        &ett_nl80211_wiphy_radio_freq_range,
        &ett_nl80211_s1g_short_beacon_attrs,
        &ett_nl80211_nan_capabilities,
/* }}} */
    };

    proto_netlink_nl80211 = proto_register_protocol("Linux 802.11 Netlink", "nl80211", "nl80211");
    proto_register_field_array(proto_netlink_nl80211, hf, array_length(hf));
    proto_register_subtree_array(ett, array_length(ett));

    netlink_nl80211_handle = register_dissector("nl80211", dissect_netlink_nl80211, proto_netlink_nl80211);
    ieee80211_handle = find_dissector_add_dependency("wlan", proto_netlink_nl80211);
    ieee80211_tag_dissector_table = find_dissector_table("wlan.tag.number");
}

void
proto_reg_handoff_netlink_nl80211(void)
{
    dissector_add_string("genl.family", "nl80211", netlink_nl80211_handle);
}

/*
 * Editor modelines  -  https://www.wireshark.org/tools/modelines.html
 *
 * Local variables:
 * c-basic-offset: 4
 * tab-width: 8
 * indent-tabs-mode: nil
 * End:
 *
 * vi: set shiftwidth=4 tabstop=8 expandtab:
 * :indentSize=4:tabSize=8:noTabs=true:
 */
