/** @file
 *
 * Wireshark - Network traffic analyzer
 * By Gerald Combs <gerald@wireshark.org>
 * Copyright 1998 Gerald Combs
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 */

#include <epan/proto.h>
#include <epan/addr_resolv.h>
#include <epan/prefs.h>
#include <epan/maxmind_db.h>
#include <epan/conversation_table.h>

#include <wsutil/utf8_entities.h>
#include <wsutil/filesystem.h>
#include <wsutil/application_flavor.h>

#include <ui/qt/main_application.h>
#include <ui/qt/filter_action.h>
#include <ui/qt/models/atap_data_model.h>
#include <ui/qt/utils/variant_pointer.h>
#include <ui/qt/widgets/traffic_tab.h>
#include <ui/qt/widgets/traffic_tree.h>
#include <ui/qt/widgets/traffic_types_list.h>
#include <ui/qt/widgets/detachable_tabwidget.h>

#include <QStringList>
#include <QTreeView>
#include <QList>
#include <QMap>
#include <QPushButton>
#include <QMenu>
#include <QSortFilterProxyModel>
#include <QTabBar>
#include <QJsonDocument>
#include <QJsonArray>
#include <QJsonObject>
#include <QJsonValue>
#include <QTextStream>
#include <QClipboard>
#include <QMessageBox>
#include <QUrl>
#include <QTemporaryFile>
#include <QHBoxLayout>

TabData::TabData() :
    _name(QString()),
    _protoId(-1)
{}

TabData::TabData(QString name, int protoId) :
    _name(name),
    _protoId(protoId)
{}

QString TabData::name() const
{
    return _name;
}

int TabData::protoId() const
{
    return _protoId;
}


TrafficTab::TrafficTab(QWidget * parent) :
    DetachableTabWidget(parent),
    _recentList(nullptr),
    _recentColumnList(nullptr)
{
    _createModel = nullptr;
    _createDelegate = nullptr;
    _disableTaps = false;
    _nameResolution = false;
    _absoluteTime = false;
    _limitToDisplayFilter = false;
    _nanoseconds = false;
    _machineReadable = false;
    setTabBasename(QString());
}

TrafficTab::~TrafficTab()
{}

void TrafficTab::setProtocolInfo(QString tableName, TrafficTypesList * trafficList, GList ** recentList, GList ** recentColumnList, ATapModelCallback createModel)
{
    setTabBasename(tableName);

    _allProtocols = trafficList->protocols();
    if (createModel)
        _createModel = createModel;

    _recentList = recentList;
    _recentColumnList = recentColumnList;

    setOpenTabs(trafficList->protocols(true));
}

void TrafficTab::setDelegate(ATapCreateDelegate createDelegate)
{
    if (! createDelegate)
        return;

    _createDelegate = createDelegate;


    for (int idx = 0; idx < count(); idx++) {
        if (qobject_cast<QTreeView *>(widget(idx)))
        {
            QTreeView * tree = qobject_cast<QTreeView *>(widget(idx));
            tree->setItemDelegate(createDelegate(tree));
        }
    }
}

QTreeView * TrafficTab::createTree(int protoId)
{
    TrafficTree * tree = new TrafficTree(tabBasename(), _recentColumnList, this);

    if (_createModel) {
        ATapDataModel * model = _createModel(protoId, "");
        model->useAbsoluteTime(_absoluteTime);
        model->limitToDisplayFilter(_limitToDisplayFilter);
        model->useNanosecondTimestamps(_nanoseconds);
        model->setResolveNames(_nameResolution);
        model->setParent(tree);
        connect(model, &ATapDataModel::tapListenerChanged, tree, &TrafficTree::tapListenerEnabled);

        model->enableTap();
        model->setMachineReadable(_machineReadable);

        if (_createDelegate)
        {
            tree->setItemDelegate(_createDelegate(tree));
        }

        TrafficDataFilterProxy * proxyModel = new TrafficDataFilterProxy(tree);
        proxyModel->setSourceModel(model);
        tree->setModel(proxyModel);

        QItemSelectionModel * ism = new QItemSelectionModel(proxyModel, tree);
        tree->setSelectionModel(ism);

        /* The default (historical) selection mode is 'Single'
         * However, we allow some protocols (IP/TCP/UDP..) to use 'Extended'
         * for I/O Graph facility
         */
        QString protoname = proto_get_protocol_short_name(find_protocol_by_id(protoId));
        bool useExtendedSelection = protoname.toUtf8().data()== QString("TCP") ||
                                        protoname.toUtf8().data()== QString("UDP") ||
                                        protoname.toUtf8().data()== QString("IPv4") ||
                                        protoname.toUtf8().data()== QString("IPv6") ||
                                        protoname.toUtf8().data()== QString("Ethernet");

        if(useExtendedSelection) {
            tree->setSelectionMode(QAbstractItemView::ExtendedSelection);
            connect(ism, &QItemSelectionModel::selectionChanged, this, [=](const QItemSelection &sel, const QItemSelection &prev){ doSelectionChange(sel, prev); });
        }
        else {
            tree->setSelectionMode(QAbstractItemView::SingleSelection);
            connect(ism, &QItemSelectionModel::currentChanged, this, &TrafficTab::doCurrentIndexChange);
        }

        // Initially resize to the header widths (inc. hidden/filtered columns).
        for (int col = 0; col < tree->model()->columnCount(); col++) {
            tree->resizeColumnToContents(col);
        }

        tree->applyRecentColumns();

        tree->sortByColumn(0, Qt::AscendingOrder);

        connect(proxyModel, &TrafficDataFilterProxy::modelReset, this, [tree]() {
            if (tree->model()->rowCount() > 0) {
                for (int col = 0; col < tree->model()->columnCount(); col++) {
                    tree->widenColumnToContents(col);
                }
            }
        });
        connect(proxyModel, &TrafficDataFilterProxy::modelReset, this, &TrafficTab::modelReset);
        connect(proxyModel, &TrafficDataFilterProxy::layoutChanged, this, &TrafficTab::modelReset);

        /* If the columns for the tree have changed, contact the tab. By also having the tab
         * columns changed signal connecting back to the tree, it will propagate to all trees
         * registered with this tab. Attention, this heavily relies on the fact, that all
         * tree data models are identical */
        connect(tree, &TrafficTree::columnsHaveChanged, this, &TrafficTab::columnsHaveChanged);
        connect(this, &TrafficTab::columnsHaveChanged, tree, &TrafficTree::columnsChanged);
    }

    return tree;
}

void TrafficTab::useAbsoluteTime(bool absolute)
{
    if (absolute == _absoluteTime)
        return;

    _absoluteTime = absolute;
    for(int idx = 0; idx < count(); idx++)
    {
        ATapDataModel * atdm = dataModelForTabIndex(idx);
        if (atdm)
            atdm->useAbsoluteTime(absolute);
    }
}

void TrafficTab::useNanosecondTimestamps(bool nanoseconds)
{
    if (nanoseconds == _nanoseconds)
        return;

    _nanoseconds = nanoseconds;
    for(int idx = 0; idx < count(); idx++)
    {
        ATapDataModel * atdm = dataModelForTabIndex(idx);
        if (atdm)
            atdm->useNanosecondTimestamps(nanoseconds);
    }
}

void TrafficTab::limitToDisplayFilter(bool limit)
{
    if (limit == _limitToDisplayFilter)
        return;

    _limitToDisplayFilter = limit;
    for(int idx = 0; idx < count(); idx++)
    {
        ATapDataModel * atdm = dataModelForTabIndex(idx);
        if (atdm)
            atdm->limitToDisplayFilter(limit);
    }
}

void TrafficTab::setMachineReadable(bool machine)
{
    if (machine == _machineReadable)
        return;

    _machineReadable = machine;
    for(int idx = 0; idx < count(); idx++)
    {
        ATapDataModel * atdm = dataModelForTabIndex(idx);
        if (atdm)
            atdm->setMachineReadable(machine);
    }
}

void TrafficTab::disableTap()
{
    for(int idx = 0; idx < count(); idx++)
    {
        ATapDataModel * atdm = dataModelForTabIndex(idx);
        if (atdm)
            atdm->disableTap();
    }

    _disableTaps = true;
    emit disablingTaps();
}

void TrafficTab::setOpenTabs(QList<int> protocols)
{
    QList<int> tabs = _tabs.keys();
    blockSignals(true);

    foreach(int protocol, protocols)
    {
        if (! tabs.contains(protocol)) {
            insertProtoTab(protocol, false);
        }
        tabs.removeAll(protocol);
    }

    foreach(int protocol, tabs)
        removeProtoTab(protocol, false);

    blockSignals(false);

    emit tabsChanged(_tabs.keys());
    emit retapRequired();
}

void TrafficTab::insertProtoTab(int protoId, bool emitSignals)
{
    QList<int> lUsed = _tabs.keys();

    if (lUsed.contains(protoId) && lUsed.count() != count())
    {
        _tabs.clear();
        for (int idx = 0; idx < count(); idx++) {
            TabData tabData = qvariant_cast<TabData>(tabBar()->tabData(idx));
            _tabs.insert(tabData.protoId(), idx);
        }
        lUsed = _tabs.keys();
    }

    if (protoId <= 0 || lUsed.contains(protoId))
        return;

    QList<int> lFull = _allProtocols;
    int idx = (int) lFull.indexOf(protoId);
    if (idx < 0)
        return;

    QList<int> part = lFull.mid(0, idx);
    int insertAt = 0;
    if (part.count() > 0) {
        for (int cnt = idx - 1; cnt >= 0; cnt--) {
            if (lUsed.contains(part[cnt]) && part[cnt] != protoId) {
                insertAt = (int) lUsed.indexOf(part[cnt]) + 1;
                break;
            }
        }
    }

    QTreeView * tree = createTree(protoId);
    QString tableName = proto_get_protocol_short_name(find_protocol_by_id(protoId));
    TabData tabData(tableName, protoId);
    QVariant storage;
    storage.setValue(tabData);
    if (tree->model()->rowCount() > 0)
        tableName += QStringLiteral(" %1 %2").arg(UTF8_MIDDLE_DOT).arg(tree->model()->rowCount());

    int tabId = -1;
    if (insertAt > -1)
        tabId = insertTab(insertAt, tree, tableName);
    else
        tabId = addTab(tree, tableName);
    if (tabId >= 0)
        tabBar()->setTabData(tabId, storage);

    // Identify the last known opened tab
    int lastOpened_protoId = -1;
    GList *selected_tab = g_list_first(*_recentList);
    if (selected_tab != nullptr) {
        lastOpened_protoId = proto_get_id_by_short_name((const char *)selected_tab->data);
    }

    /* We reset the correct tab idxs. That operations is costly, but it is only
     * called during this operation and ensures, that other operations do not
     * need to iterate, but rather can lookup the indeces. */
    _tabs.clear();
    for (int idx = 0; idx < count(); idx++) {
        TabData tabData = qvariant_cast<TabData>(tabBar()->tabData(idx));
        _tabs.insert(tabData.protoId(), idx);
    }

    // Restore the last known opened tab
    if(lastOpened_protoId == protoId) {
        setCurrentIndex(tabId);
    }

    if (emitSignals) {
        emit tabsChanged(_tabs.keys());
        emit retapRequired();
    }
}

void TrafficTab::removeProtoTab(int protoId, bool emitSignals)
{
    if (_tabs.keys().contains(protoId)) {
        for(int idx = 0; idx < count(); idx++) {
            TabData tabData = qvariant_cast<TabData>(tabBar()->tabData(idx));
            if (protoId == tabData.protoId()) {
                removeTab(idx);
                break;
            }
        }
    }

    /* We reset the correct tab idxs. That operations is costly, but it is only
    * called during this operation and ensures, that other operations do not
    * need to iterate, but rather can lookup the indeces. */
    _tabs.clear();
    for (int idx = 0; idx < count(); idx++) {
        TabData tabData = qvariant_cast<TabData>(tabBar()->tabData(idx));
        _tabs.insert(tabData.protoId(), idx);
    }

    if (emitSignals) {
        emit tabsChanged(_tabs.keys());
        emit retapRequired();
    }
}

void TrafficTab::doCurrentIndexChange(const QModelIndex & cur, const QModelIndex &)
{
    if (! cur.isValid())
        return;

    const TrafficDataFilterProxy * proxy = qobject_cast<const TrafficDataFilterProxy *>(cur.model());
    if (! proxy)
        return;

    ATapDataModel * model = qobject_cast<ATapDataModel *>(proxy->sourceModel());
    if (! model)
        return;

    int tabId = _tabs[model->protoId()];
    emit tabDataChanged(tabId, 0);
}

void TrafficTab::doSelectionChange(const QItemSelection &, const QItemSelection &)
{
    int tabId = -1;

    QTreeView * tree = qobject_cast<QTreeView *>(currentWidget());
    if (tree) {
        const QModelIndexList indexes = tree->selectionModel()->selectedIndexes();
        for (const QModelIndex &index : indexes) {
            const TrafficDataFilterProxy * proxy = qobject_cast<const TrafficDataFilterProxy *>(index.model());
            if(proxy) {
                ATapDataModel * model = qobject_cast<ATapDataModel *>(proxy->sourceModel());
                if(model) {
                    tabId = _tabs[model->protoId()];
                    break;
                }
            }
        }
    }

    if(tabId >= 0)
        emit tabDataChanged(tabId, 0);
}

QVariant TrafficTab::currentItemData(int role)
{
    QTreeView * tree = qobject_cast<QTreeView *>(currentWidget());
    if (tree) {
        QModelIndex idx = tree->selectionModel()->currentIndex();
        /* In case no selection has been made yet, we select the topmostleft index,
         * to ensure proper handling. Especially ConversationDialog depends on this
         * method always returning data */
        if (!idx.isValid()) {
            TrafficDataFilterProxy * model = modelForTabIndex(currentIndex());
            idx = model->index(0, 0);
        }
        return idx.data(role);
    }

    return QVariant();
}

qlonglong TrafficTab::countSelectedItems(int role)
{
    qlonglong selectedCount = 0;
    QTreeView * tree = qobject_cast<QTreeView *>(currentWidget());
    if (tree && role>=0) {
        QModelIndex idx = tree->selectionModel()->currentIndex();

        /* In case no selection has been made yet, we select the topmostleft index,
         * to ensure proper handling. Especially ConversationDialog depends on this
         * method always returning data */
        if (!idx.isValid()) {
            TrafficDataFilterProxy * model = modelForTabIndex(currentIndex());
            idx = model->index(0, 0);
        }
        QModelIndexList idxs = tree->selectionModel()->selectedIndexes();

        const QModelIndexList indexes = tree->selectionModel()->selectedRows();
        selectedCount = indexes.size();
    }

    return selectedCount;
}

QList<QList<QVariant>> TrafficTab::selectedItemsIOGData()
{
    QList<QList<QVariant> > lst_selectedData;

    QTreeView * tree = qobject_cast<QTreeView *>(currentWidget());
    QList<QList<QVariant>> dataItems;
    if (tree) {

        QModelIndex idx = tree->selectionModel()->currentIndex();

        /* In case no selection has been made yet, we select the topmostleft index,
         * to ensure proper handling. Especially ConversationDialog depends on this
         * method always returning data */
        if (!idx.isValid()) {
            TrafficDataFilterProxy * model = modelForTabIndex(currentIndex());
            idx = model->index(0, 0);
            return lst_selectedData;
        }

        const TrafficDataFilterProxy * zemodel = qobject_cast<const TrafficDataFilterProxy *>(tree->model());

        QList<QVariant> lst_ids;
        QList<QVariant> lst_agg;

        const QModelIndexList lst_indexes = tree->selectionModel()->selectedRows();

        for (const QModelIndex &index : lst_indexes) {

            ATapDataModel *dataModel = qobject_cast<ATapDataModel *>(zemodel->sourceModel());
            int protoId = dataModel->protoId();

            if (qobject_cast<ConversationDataModel *>(dataModel)) {
                ConversationDataModel * conversationModel = qobject_cast<ConversationDataModel *>(dataModel);

                conv_item_t *conv_item = conversationModel->itemForRow( (zemodel->mapToSource(index)).row() );

                char *src_addr, *new_addr;
                src_addr = address_to_str(NULL, &conv_item->src_address);
                new_addr = wmem_strdup_printf(NULL, "\"%s\"", src_addr);
                src_addr = new_addr;

                char *dst_addr;
                dst_addr = address_to_str(NULL, &conv_item->dst_address);
                new_addr = wmem_strdup_printf(NULL, "\"%s\"", dst_addr);
                dst_addr = new_addr;

                /* Ordinary conversations have a conv_id and it's used to build the filter */
                if(conv_item->conv_id != CONV_ID_UNSET) {
                    // add to list (prepend with PROTO ID on 1st elt)
                    if(lst_ids.isEmpty())
                        lst_ids.append(protoId);
                    lst_ids.append(conv_item->conv_id);
                }

                /* Aggregated conversations use to build the filter */
                else {
                    // add to list (prepend with PROTO ID on 1st elt)
                    if(lst_agg.isEmpty())
                        lst_agg.append(protoId);
                    lst_agg.append(QString(src_addr));
                    lst_agg.append(QString(dst_addr));
                }

                wmem_free(NULL, src_addr);
                wmem_free(NULL, dst_addr);
            }
        }
        dataItems.append(lst_ids);
        dataItems.append(lst_agg);
    }
    return dataItems;
}

// update current tab label to include row count
void TrafficTab::modelReset()
{
    if (! qobject_cast<TrafficDataFilterProxy *>(sender()))
        return;

    TrafficDataFilterProxy * qsfpm = qobject_cast<TrafficDataFilterProxy *>(sender());
    if (!qsfpm || ! qobject_cast<ATapDataModel *>(qsfpm->sourceModel()))
        return;

    ATapDataModel * atdm = qobject_cast<ATapDataModel *>(qsfpm->sourceModel());
    int protoId = atdm->protoId();
    if (!_tabs.keys().contains(protoId))
        return;

    int tabIdx = _tabs[protoId];
    TabData tabData = qvariant_cast<TabData>(tabBar()->tabData(tabIdx));

    if (tabData.protoId() == protoId) {
        if (qsfpm->rowCount() == 0)
            setTabText(tabIdx, tabData.name());
        else
            setTabText(tabIdx, QStringLiteral("%1 %2 %3")
                .arg(tabData.name(), UTF8_MIDDLE_DOT)
                .arg(qsfpm->rowCount()));
    }

    emit tabDataChanged(tabIdx, 0);
}

TrafficDataFilterProxy * TrafficTab::modelForTabIndex(int tabIdx)
{
    if (tabIdx == -1)
        tabIdx = currentIndex();

    return modelForWidget(widget(tabIdx));
}

TrafficDataFilterProxy * TrafficTab::modelForWidget(QWidget * searchWidget)
{
    if (qobject_cast<QTreeView *>(searchWidget)) {
        QTreeView * tree = qobject_cast<QTreeView *>(searchWidget);
        if (qobject_cast<TrafficDataFilterProxy *>(tree->model())) {
            return qobject_cast<TrafficDataFilterProxy *>(tree->model());
        }
    }

    return nullptr;
}

ATapDataModel * TrafficTab::dataModelForTabIndex(int tabIdx)
{
    if (tabIdx == -1)
        tabIdx = currentIndex();

    return dataModelForWidget(widget(tabIdx));
}

ATapDataModel * TrafficTab::dataModelForWidget(QWidget * searchWidget)
{
    TrafficDataFilterProxy * qsfpm = modelForWidget(searchWidget);
    if (qsfpm && qobject_cast<ATapDataModel *>(qsfpm->sourceModel())) {
        return qobject_cast<ATapDataModel *>(qsfpm->sourceModel());
    }
    return nullptr;
}

void TrafficTab::setFilter(QString filter)
{
    for (int idx = 0; idx < count(); idx++ )
    {
        ATapDataModel * atdm = dataModelForTabIndex(idx);
        if (! atdm)
            continue;
        atdm->setFilter(filter);
    }
}

void TrafficTab::setNameResolution(bool checked)
{
    if (checked == _nameResolution)
        return;

    for (int idx = 0; idx < count(); idx++ )
    {
        ATapDataModel * atdm = dataModelForTabIndex(idx);
        if (! atdm)
            continue;
        atdm->setResolveNames(checked);

    }

    _nameResolution = checked;

    /* Send the signal, that all tabs have potentially changed */
    emit tabDataChanged(-1, 0);
}

bool TrafficTab::hasNameResolution(int tabIdx)
{
    int tab = tabIdx == -1 || tabIdx >= count() ? currentIndex() : tabIdx;
    ATapDataModel * dataModel = dataModelForTabIndex(tab);
    if (! dataModel)
        return false;

    return dataModel->allowsNameResolution();
}

QMenu * TrafficTab::createCopyMenu(QWidget *parent)
{
    TrafficTree * tree = qobject_cast<TrafficTree *>(currentWidget());
    if ( ! tree)
        return nullptr;

    return tree->createCopyMenu(parent);
}

#ifdef HAVE_MAXMINDDB
bool TrafficTab::hasGeoIPData(int tabIdx)
{
    int tab = tabIdx == -1 || tabIdx >= count() ? currentIndex() : tabIdx;

    ATapDataModel * dataModel = dataModelForTabIndex(tab);
    if (! dataModel)
        return false;

    return dataModel->hasGeoIPData();
}

bool
TrafficTab::writeGeoIPMapFile(QFile * fp, bool json_only, TrafficDataFilterProxy * model)
{
    QTextStream out(fp);

    if (!json_only) {
        QFile ipmap(get_datafile_path("ipmap.html", application_configuration_environment_prefix()));

        if (!ipmap.open(QIODevice::ReadOnly)) {
            QMessageBox::warning(this, tr("Map file error"), tr("Could not open base file %1 for reading: %2")
                .arg(get_datafile_path("ipmap.html", application_configuration_environment_prefix()), g_strerror(errno))
            );
            return false;
        }

        /* Copy ipmap.html to map file. */
        QTextStream in(&ipmap);
        QString line;
        while (in.readLineInto(&line)) {
            out << line << Qt::endl;
        }

        out << QStringLiteral("<script id=\"ipmap-data\" type=\"application/json\">\n");
    }

    /*
     * Writes a feature for each resolved address, the output will look like:
     *  {
     *    "type": "FeatureCollection",
     *    "features": [
     *      {
     *        "type": "Feature",
     *        "geometry": {
     *          "type": "Point",
     *          "coordinates": [ -97.821999, 37.750999 ]
     *        },
     *        "properties": {
     *          "ip": "8.8.4.4",
     *          "autonomous_system_number": 15169,
     *          "autonomous_system_organization": "Google LLC",
     *          "city": "(omitted, but key is shown for documentation reasons)",
     *          "country": "United States",
     *          "radius": 1000,
     *          "packets": 1,
     *          "bytes": 1543
     *        }
     *      }
     *    ]
     *  }
     */

    QJsonObject root;
    root["type"] = "FeatureCollection";
    QJsonArray features;

    /* Append map data. */
    for(int row = 0; row < model->rowCount(QModelIndex()); row++)
    {
        QModelIndex index = model->mapToSource(model->index(row, 0));
        ATapDataModel *dataModel = qobject_cast<ATapDataModel *>(model->sourceModel());
        const mmdb_lookup_t * result = VariantPointer<const mmdb_lookup_t>::asPtr(dataModel->data(index, ATapDataModel::GEODATA_LOOKUPTABLE));

        if (!maxmind_db_has_coords(result)) {
            // result could be NULL if the caller did not trigger a lookup
            // before. result->found could be false if no MMDB entry exists.
            continue;
        }

        QJsonObject arrEntry;
        arrEntry["type"] = "Feature";
        QJsonObject geometry;
        geometry["type"] = "Point";
        QJsonArray coordinates;
        coordinates.append(QJsonValue(result->longitude));
        coordinates.append(QJsonValue(result->latitude));
        geometry["coordinates"] = coordinates;
        arrEntry["geometry"] = geometry;

        QJsonObject property;
        property["ip"] = dataModel->data(index, ATapDataModel::GEODATA_ADDRESS).toString();
        if (result->as_number && result->as_org) {
            property["autonomous_system_number"] = QJsonValue((int)(result->as_number));
            property["autonomous_system_organization"] = QJsonValue(result->as_org);
        }

        if (result->city)
            property["city"] = result->city;
        if (result->country)
            property["country"] = result->country;
        if (result->accuracy)
            property["radius"] = QJsonValue(result->accuracy);

        if (qobject_cast<EndpointDataModel *>(dataModel)) {
            EndpointDataModel * endpointModel = qobject_cast<EndpointDataModel *>(dataModel);
            property["packets"] = endpointModel->data(index.siblingAtColumn(EndpointDataModel::ENDP_COLUMN_PACKETS)).toString();
            property["bytes"] = endpointModel->data(index.siblingAtColumn(EndpointDataModel::ENDP_COLUMN_BYTES)).toString();
        }
        arrEntry["properties"] = property;
        features.append(arrEntry);
    }
    root["features"] = features;
    QJsonDocument doc;
    doc.setObject(root);

    out << doc.toJson();

    if (!json_only)
        out << QStringLiteral("</script>\n");

    out.flush();

    return true;
}

QUrl TrafficTab::createGeoIPMap(bool json_only, int tabIdx)
{
    int tab = tabIdx == -1 || tabIdx >= count() ? currentIndex() : tabIdx;
    ATapDataModel * dataModel = dataModelForTabIndex(tab);
    if (! (dataModel && dataModel->hasGeoIPData())) {
        QMessageBox::warning(this, tr("Map file error"), tr("No endpoints available to map"));
        return QUrl();
    }

    QString tempname = QStringLiteral("%1/ipmapXXXXXX.html").arg(QDir::tempPath());
    QTemporaryFile tf(tempname);
    if (!tf.open()) {
        QMessageBox::warning(this, tr("Map file error"), tr("Unable to create temporary file"));
        return QUrl();
    }

    if (!writeGeoIPMapFile(&tf, json_only, modelForTabIndex(tab))) {
        tf.close();
        return QUrl();
    }

    tf.setAutoRemove(false);
    return QUrl::fromLocalFile(tf.fileName());
}
#endif

void TrafficTab::detachTab(int tabIdx, QPoint pos) {
    ATapDataModel * model = dataModelForTabIndex(tabIdx);
    if (!model)
        return;

    TrafficTree * tree = qobject_cast<TrafficTree *>(widget(tabIdx));
    if (!tree)
        return;

    connect(this, &TrafficTab::disablingTaps ,tree , &TrafficTree::disableTap);
    DetachableTabWidget::detachTab(tabIdx, pos);

    removeProtoTab(model->protoId());
}

void TrafficTab::attachTab(QWidget * content, QString name)
{
    ATapDataModel * model = dataModelForWidget(content);
    if (!model) {
        DetachableTabWidget::attachTab(content, name);
        return;
    }

    insertProtoTab(model->protoId());
}
